/* Naval Battle for Pokitto */
/* Written by Richard Ames 2018 */

#include "Pokitto.h"

Pokitto::Core game;
Pokitto::Sound snd;

#define DESTROYER  0
#define CRUISER    1
#define BATTLESHIP 2
#define CARRIER    3

/* Ship states */
#define DEPLOYING  1
#define SELECTED   2

/* Game states */
#define SPLASH         0
#define SETUP          1
#define PLACE          2
#define ENGAGE         3
#define FLY_MISSILE    4
#define COMPUTER_PLAYS 5
#define WINNER         6

/* Placement states */
#define PLACING      0
#define NOT_SELECTED 1

/* Grid states */
#define GRID_SHIP 0x01
#define GRID_SHOT 0x02

#define MUXCOUNT  8
#define BTHRESH   1

#define GRID_SIZE 10

/* Simple sprite graphics, coded by hand */
const uint8_t ship2[] = {
 16, 5,
 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee,
 0xee, 0xee, 0x66, 0xee, 0xee, 0xee, 0xee, 0xee,
 0xee, 0x66, 0x76, 0x66, 0xee, 0xee, 0xee, 0xee,
 0x55, 0x55, 0x55, 0x55, 0x55, 0xee, 0xee, 0xee,
 0xe5, 0x55, 0x55, 0x55, 0x5e, 0xee, 0xee, 0xee    
};

const uint8_t ship2v[] = {
 8, 10,
 0xee, 0x5e, 0xee, 0xee,
 0xe5, 0x55, 0xee, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0xe5, 0x55, 0xee, 0xee,
 0xee, 0x5e, 0xee, 0xee
};

const uint8_t ship3[] = {
 16, 5,
 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee,
 0xee, 0xee, 0xee, 0x66, 0x6e, 0xee, 0xee, 0xee,
 0xee, 0xee, 0x66, 0x76, 0x76, 0x76, 0xee, 0xee,
 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x5e,
 0xe5, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xee    
};

const uint8_t ship3v[] = {
 8, 15,
 0xee, 0x5e, 0xee, 0xee,
 0xe5, 0x55, 0xee, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x5e, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0xe5, 0x55, 0xee, 0xee,
 0xee, 0x5e, 0xee, 0xee
};

const uint8_t ship4[] = {
 24, 5,
 0xee, 0xee, 0xee, 0xee, 0xee, 0xe6, 0x6e, 0xee, 0xee, 0xee, 0xee, 0xee,
 0xee, 0xee, 0xee, 0x66, 0x66, 0x66, 0x66, 0x6e, 0xee, 0xee, 0xee, 0xee,
 0xee, 0xee, 0x66, 0x76, 0x76, 0x76, 0x76, 0x76, 0x6e, 0xee, 0xee, 0xee,
 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xee, 0xee,
 0xe5, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x5e, 0xee, 0xee
};

const uint8_t ship4v[] = {
 8, 20,
 0xee, 0x5e, 0xee, 0xee,
 0xe5, 0x55, 0xee, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0xe5, 0x55, 0xee, 0xee,
 0xee, 0x5e, 0xee, 0xee
};

const uint8_t ship5[] = {
 32, 5,
 0xee, 0xee, 0xee, 0xee, 0x66, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee,
 0xee, 0xee, 0xee, 0xe6, 0x76, 0x76, 0x6e, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee, 0xee,
 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x5e, 0xee, 0xee, 0xee,
 0xe5, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0xee, 0xee, 0xee, 0xee,
 0xee, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x5e, 0xee, 0xee, 0xee, 0xee,
};

const uint8_t ship5v[] = {
 8, 25,
 0xee, 0x5e, 0xee, 0xee,
 0xe5, 0x55, 0xee, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x56, 0x55, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x56, 0x55, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0x56, 0x55, 0x5e, 0xee,
 0x56, 0x66, 0x5e, 0xee,
 0x55, 0x65, 0x5e, 0xee,
 0x55, 0x55, 0x5e, 0xee,
 0xe5, 0x55, 0xee, 0xee,
 0xee, 0x5e, 0xee, 0xee
};

const uint8_t missile[] = {
 8, 5,
 0x9e, 0x9e, 0x9e, 0xee,
 0xe9, 0x99, 0xee, 0xee,
 0x99, 0xa9, 0x9e, 0xee,
 0xe9, 0x99, 0xee, 0xee,
 0x9e, 0x9e, 0x9e, 0xee
};

const uint8_t splash[] = {
 8, 5,
 0x7e, 0xee, 0x7e, 0xee,
 0xe7, 0xe7, 0xee, 0xee,
 0xee, 0x7e, 0xee, 0xee,
 0xe7, 0xe7, 0xee, 0xee,
 0x7e, 0xee, 0x7e, 0xee
};

struct point {
    int x;
    int y;
};

struct point budge[] = {
        {-1,  1},
        { 0,  1},
        { 1,  1},
        {-1,  0},
        { 1,  0},
        {-1, -1},
        { 0, -1},
        { 1, -1}};

struct ship {
    int type;
    int state;
    int vert;
    struct point home;
    struct point dest;
    struct point now;
};

struct ship ship[] = {
    {0, 0, 0,  5, 24, 0, 0, 0, 0},
    {1, 0, 0,  5, 34, 0, 0, 0, 0},
    {1, 0, 0, 25, 34, 0, 0, 0, 0},
    {2, 0, 0,  5, 44, 0, 0, 0, 0},
    {3, 0, 0,  5, 54, 0, 0, 0, 0}};
    
int shiplen[] = {2, 3, 4, 5};
int shipinfo[] = {2, 3, 3, 4, 5};

/* Globals */    
int numships = (sizeof(ship) / sizeof(ship[0]));
int cursx;
int cursy;
int place_state;
int game_state;
int state_ctr;
char str1[8];
int debug0;
int debug1;
int debug2;
int debug3;
int mx;
int my;
int a_num;
int a_den;
int arc_c;
char pgrid[GRID_SIZE][GRID_SIZE];
char cgrid[GRID_SIZE][GRID_SIZE];
char heatmap[GRID_SIZE][GRID_SIZE];
int computer_playing;
int computer_guessing;
int computerx;
int computery;
int oldx;
int oldy;
int bdelay;

void show_debug(void)
{
    game.display.color = 12;
    game.display.setCursor(60, 72);
    sprintf(str1, "%d %d", debug0, debug1);
    game.display.print(str1);
    game.display.setCursor(60, 80);
    sprintf(str1, "%d %d", debug2, debug3);
    game.display.print(str1);
}

/* Convert from coarse grid to fine grid */
int cursx0(void)
{
    return cursx * 5 - 1;
}

/* Convert from coarse grid to fine grid */
int cursy0(void)
{
    return cursy * 5 - 2;
}

/* Show yellow box cursor */
void show_cursor(void)
{
    game.display.color = 10;        
    game.display.fillRectangle(cursx0(), cursy0(), 5, 5);
}

/* Updated ship on screen.  Could probably be rewritten as one case */
void display_ship(int index)
{
    switch (ship[index].type)
    {
    case DESTROYER:
        if (ship[index].vert == 0)
            game.display.drawBitmap(ship[index].now.x, ship[index].now.y, ship2);
        else
            game.display.drawBitmap(ship[index].now.x, ship[index].now.y, ship2v);
        break;
    case CRUISER:
        if (ship[index].vert == 0)
            game.display.drawBitmap(ship[index].now.x, ship[index].now.y, ship3);
        else
            game.display.drawBitmap(ship[index].now.x, ship[index].now.y, ship3v);
        break;
    case BATTLESHIP:
        if (ship[index].vert == 0)
            game.display.drawBitmap(ship[index].now.x, ship[index].now.y, ship4);
        else
            game.display.drawBitmap(ship[index].now.x, ship[index].now.y, ship4v);
        break;
    case CARRIER:
        if (ship[index].vert == 0)
            game.display.drawBitmap(ship[index].now.x, ship[index].now.y, ship5);
        else
            game.display.drawBitmap(ship[index].now.x, ship[index].now.y, ship5v);
        break;
    }
}

/* Draw computer grid on display.  Include hits and misses */
void draw_cgrid(void)
{
    int i;
    int j;
    int checkval;
    
    game.display.color = 11;            
    /* Draw grid for computer's ships */
    for (j = 0; j < 10; j++)
        for (i = 0; i < 10; i++)
        {
            game.display.fillRectangle(5 + i * 5, 19 + j * 5, 3, 3);
            checkval = cgrid[i][j] & (GRID_SHIP | GRID_SHOT);
            if (checkval == (GRID_SHIP | GRID_SHOT))
                game.display.drawBitmap(5 + i * 5, 19 + j * 5, missile);
            else if (checkval == GRID_SHOT)
                game.display.drawBitmap(5 + i * 5, 19 + j * 5, splash); 
        }
}

/* Draw player grid on screen.  Include hits and misses */
void draw_pgrid(void)
{
    int i;
    int j;
    int checkval;
    
    game.display.color = 12;            
    /* Draw grid for ship placement */
    for (j = 0; j < 10; j++)
        for (i = 0; i < 10; i++)
            game.display.fillRectangle(60 + i * 5, 19 + j * 5, 3, 3);
                      
    for (i = 0; i < numships; i++)
    {
        display_ship(i);
    }

    for (j = 0; j < 10; j++)
        for (i = 0; i < 10; i++)
        {
            checkval = pgrid[i][j] & (GRID_SHIP | GRID_SHOT);
            if (checkval == (GRID_SHIP | GRID_SHOT))
                game.display.drawBitmap(60 + i * 5, 19 + j * 5, missile);
            else if (checkval == GRID_SHOT)
                game.display.drawBitmap(60 + i * 5, 19 + j * 5, splash);
        } 
}

/* Check if player's ships are inside grid.  Return 1 if OK */
int check_pgrid(void)
{
    int shipno;
    struct ship *s;
    int endpoint;
    int x;
    int y;
    int i;

    /* Clear player grid ship placement info */
    for (x = 0; x < GRID_SIZE; x++)
        for (y = 0; y < GRID_SIZE; y++)
            pgrid[x][y] &= ~GRID_SHIP;
    
    for (shipno = 0; shipno < numships; shipno++)
    {

        s = &ship[shipno];
        /* Set up coarse coordinates */
        x = (s->now.x + 1) / 5;
        y = (s->now.y + 2) / 5;
        /* Check first end point */
        if ((x < 12) || (x > 21) ||
            (y < 4)  || (y > 13))
        {
            return 0;
        }
        /* Check second end point */
        if (s->vert)
        {
            endpoint = y + shiplen[s->type] - 1;
            if ((endpoint < 4) || (endpoint > 13))
            {
                return 0;
            }
        }
        else
        {
            endpoint = x + shiplen[s->type] - 1;
            if ((endpoint < 12) || (endpoint > 21))
            {
                return 0;
            }
        }
        /* Convert to pgrid relative offsets */
        x = x - 12;
        y = y - 4;
        for (i = 0; i < shiplen[s->type]; i++)
            if (s->vert)
                if (pgrid[x][y + i] & GRID_SHIP)
                {
                    return 0;
                }
                else
                    pgrid[x][y + i] |= GRID_SHIP;
            else
                if (pgrid[x + i][y] & GRID_SHIP)
                {
                    return 0;
                }
                else
                    pgrid[x + i][y] |= GRID_SHIP;
    }
    return 1;
}

/* Randomly place ships in computer grid, clear player grid */
void fill_grids(void)
{
   int x;
   int y;
   int len;
   int dir;
   int i;
   int shipno;
   int inplace;

   for (x = 0; x < GRID_SIZE; x++)
      for (y = 0; y < GRID_SIZE; y++)
      {
         cgrid[x][y] = 0;
         pgrid[x][y] = 0;
      }

   for (shipno = 0; shipno < sizeof(shipinfo) / sizeof(int); )
   {
      x = rand() % 10;
      y = rand() % 10;
      dir = rand() % 2;
      len = shipinfo[shipno];

      if ((dir == 0) && (x + len < GRID_SIZE))
      {
         inplace = 0;
         for (i = 0; i < len; i++)
            inplace += cgrid[x + i][y] & GRID_SHIP;
         if (inplace == 0)
         {
            for (i = 0; i < len; i++)
                cgrid[x + i][y] |= GRID_SHIP;
            shipno++;
         }
      }
      else if (y + len < GRID_SIZE)
      {
         inplace = 0;
         for (i = 0; i < len; i++)
            inplace += cgrid[x][y + i] & GRID_SHIP;
         if (inplace == 0)
         {
            for (i = 0; i < len; i++)
               cgrid[x][y + i] |= GRID_SHIP;
            shipno++;
         }
      }
   }
}

/* Computer selects target.  This is where we try to beat the player. */
/* Sets cursx and cursy */
void target_pgrid(void)
{
    int x;
    int y;
    int i;
    int shipno;
    int len;
    int misses;
    int hits;
    int hottest;
    int count;

    for (x = 0; x < GRID_SIZE; x++)
        for (y = 0; y < GRID_SIZE; y++)
            heatmap[x][y] = 0;
                
    /* Update heat map */
    /* Walk through every square, add points for every ship that would */
    /* fit there.  Extra points if we can see a ship *is* there */
    for (x = 0; x < GRID_SIZE; x++)
        for (y = 0; y < GRID_SIZE; y++)
            for (shipno = 0; shipno < sizeof(shipinfo) / sizeof(int); shipno++)
            {
                len = shipinfo[shipno];
                /* Check horizontal case */
                misses = 0;
                for (i = 0; i < len; i++)
                {
                    if ((x + i >= GRID_SIZE) ||
                        ((pgrid[x + i][y] & (GRID_SHIP | GRID_SHOT)) ==
                          GRID_SHOT))
                        misses++;
                }
                if (misses == 0)
                {
                    hits = 0;
                    for (i = 0; i < len; i++)
                    {
                        if ((pgrid[x + i][y] & (GRID_SHIP | GRID_SHOT)) ==
                             (GRID_SHIP | GRID_SHOT))
                             hits++;
                    }
                    for (i = 0; i < len; i++)
                        heatmap[x + i][y] += 1 + hits;
                }
                /* Check vertical case */
                misses = 0;
                for (i = 0; i < len; i++)
                {
                    if ((y + i >= GRID_SIZE) ||
                        ((pgrid[x][y + i] & (GRID_SHIP | GRID_SHOT)) ==
                          GRID_SHOT))
                        misses++;
                }
                if (misses == 0)
                {
                    hits = 0;
                    for (i = 0; i < len; i++)
                    {
                        if ((pgrid[x][y + i] & (GRID_SHIP | GRID_SHOT)) ==
                             (GRID_SHIP | GRID_SHOT))
                             hits++;
                    }
                    for (i = 0; i < len; i++)
                        heatmap[x][y + i] += 1 + hits;
                }
            }

    /* Find the hottest points in the heat map.  Ignore if we have already */
    /* targeted that point */
    hottest = 0;
    for (x = 0; x < GRID_SIZE; x++)
        for (y = 0; y < GRID_SIZE; y++)
            if (!(pgrid[x][y] & GRID_SHOT))
            {
                if (heatmap[x][y] > hottest)
                {
                    count = 1;
                    hottest = heatmap[x][y];
                }
                else if (heatmap[x][y] == hottest)
                    count++;
            }

    /* Select random instance of hottest from 0..count-1 */            
    count = rand() % count;
    for (x = 0; x < GRID_SIZE; x++)
        for (y = 0; y < GRID_SIZE; y++)
            if (!(pgrid[x][y] & GRID_SHOT))
            {
                if (heatmap[x][y] == hottest)
                {
                    if (count == 0)
                    {
                        cursx = x + 12;
                        cursy = y + 4;
                        return;
                    }
                    else
                        count--;
                }
            }
}

/* Check for winner.  Returns 0 - none, 1 - computer, 2 - player */
int check_winner(void)
{
    int x;
    int y;
    int misses;
    
    misses = 0;
    for (x = 0; x < GRID_SIZE; x++)
        for (y = 0; y < GRID_SIZE; y++)
        {
            if (cgrid[x][y] & GRID_SHIP)
            {
                if (!(cgrid[x][y] & GRID_SHOT))
                    misses++;
            }
        }            
    if (misses == 0)
        return 1;
        
    misses = 0;
    for (x = 0; x < GRID_SIZE; x++)
        for (y = 0; y < GRID_SIZE; y++)
        {
            if (pgrid[x][y] & GRID_SHIP)
            {
                if (!(pgrid[x][y] & GRID_SHOT))
                    misses++;
            }
        }
    if (misses == 0)
        return 2;
        
    return 0;
}

/* Initialize globals so game can be restarted */
void reset_game(void)
{
    int sn;
    
    game_state = SPLASH;
    place_state = NOT_SELECTED;
    for (sn = 0; sn < numships; sn++)
    {
        ship[sn].vert = 0;
        ship[sn].now.x = 55;
        ship[sn].now.y = 0;
        ship[sn].dest.x = ship[sn].home.x;
        ship[sn].dest.y = ship[sn].home.y;
    }
    state_ctr = 0;
    cursx = 5;
    cursy = 5;
    fill_grids();
}

/* Display splash screen */
void splash_screen(void)
{
    game.display.fillScreen(3);
    game.display.setCursor(10, 10);
    game.display.color = 12;
    game.display.print("Naval Battle");
    state_ctr++;
    if (state_ctr > 30)
        game_state = SETUP;
}

/* Find ship closest to cursor, returns index of ship */
int closest_ship(int x, int y)
{
    int i;
    int pole_value;
    int distance;
    int closest;
    
    closest = 0;
    pole_value = 10000;
    for (i = 0; i < numships; i++)
    {
        distance = (ship[i].now.x - x) * (ship[i].now.x - x) +
                   (ship[i].now.y - y) * (ship[i].now.y - y);
        if (distance < pole_value)
        {
            pole_value = distance;
            closest = i;
        }
    }
    return closest;
}

/* Pull ship in the direction of the cursor */
void pull_ship(int s)
{
    int i;
    int closest;
    int pole_value;
    int dx;
    int dy;
    int distance;
    
    if ((ship[s].now.x != ship[s].dest.x) || 
        (ship[s].now.y != ship[s].dest.y))
    {
        closest = 0;
        pole_value = 10000;
        for (i = 0; i < 8; i++)
        {
            dx = ship[s].now.x + budge[i].x - ship[s].dest.x;
            dy = ship[s].now.y + budge[i].y - ship[s].dest.y;
            distance = dx * dx + dy * dy;
            if (distance < pole_value)
            {
                pole_value = distance;
                closest = i;
            }
        }
        ship[s].now.x += budge[closest].x;
        ship[s].now.y += budge[closest].y;
    }
}

/* Draw initial game state, with ships ready to be deployed */
void setup_game(void)
{
    int i;
    int j;
    int in_motion;
    int sn;
    
    game.display.fillScreen(3);
            
    /* Draw grid for ship placement */
    for (j = 0; j < 10; j++)
        for (i = 0; i < 10; i++)
            game.display.fillRectangle(60 + i * 5, 19 + j * 5, 3, 3);

    in_motion = 0;
    for (sn = 0; sn < numships; sn++)
        if ((ship[sn].now.x != ship[sn].dest.x) ||
            (ship[sn].now.y != ship[sn].dest.y))
        {
           snd.playTone(1, 33+3*sn, 255, 1, 0);
           for (i = 0; i < 3; i++)
               pull_ship(sn);
           in_motion = 1;
           break;
        }
    for (sn = 0; sn < numships; sn++)
        display_ship(sn);
    if (!in_motion)
    {
        snd.playTone(1, 400, 255, 200);
        game_state = PLACE;
    }
}

/* Allow player to place ships in player grid */
void place_ships(void)
{
    int i;
    int j;
    static int flasher;
    static int closest;
    flasher++;
    game.display.fillScreen(3);

    game.display.color = 10;
    
    if (game.buttons.upBtn())
    {
        bdelay++;
        if (bdelay > BTHRESH)
            if (cursy >= 1)
                cursy--;
    }
    else if (game.buttons.downBtn())
    {
        bdelay++;
        if (bdelay > BTHRESH)
            if (cursy < 16)
                cursy++;
    }
    else if (game.buttons.leftBtn())
    {
        bdelay++;
        if (bdelay > BTHRESH)
            if (cursx >= 1)
                cursx--;
    }
    else if (game.buttons.rightBtn())
    {
        bdelay++;
            if (bdelay > BTHRESH)
            if (cursx < 21)
                cursx++;
    }
    else
        bdelay = 0;
    if (game.buttons.released(BTN_A))
        if (place_state == PLACING)
        {
            place_state = NOT_SELECTED;
            ship[closest].state &= ~SELECTED;
            setOSC(&osc1,0,WTRI,400,255,500);
        }
        else
        {
            if ((cursx == 15) && (cursy == 15))
            {
                cursx = 6;
                cursy = 8;
                game_state = ENGAGE;
            }
            else
            {
                snd.playTone(1, 400, 255, 200);
                closest = closest_ship(cursx0(), cursy0());
                ship[closest].state |= SELECTED | DEPLOYING;
                place_state = PLACING;
            }
        }
    if (game.buttons.released(BTN_B))
        if (place_state == PLACING)
        {
            ship[closest].vert = 1 - ship[closest].vert;
            snd.playTone(3, 100, 255, 100);
        }

    game.display.setCursor(5, 5);
    game.display.color = 12;
    game.display.print("Deploy Ships");
    game.display.setCursor(5, 72);
    game.display.print("A - Select");
    game.display.setCursor(5, 80);
    game.display.print("B - Rotate");

    show_cursor();                

    game.display.color = 12;            
    /* Draw grid for ship placement */
    for (j = 0; j < 10; j++)
        for (i = 0; i < 10; i++)
            game.display.fillRectangle(60 + i * 5, 19 + j * 5, 3, 3);

    flasher = flasher++ % MUXCOUNT;                        
    for (i = 0; i < numships; i++)
    {
        if (ship[i].state & SELECTED)
        {
            ship[i].dest.x = cursx0();
            ship[i].dest.y = cursy0();
        }
        pull_ship(i);
        if ((ship[i].state & DEPLOYING) && !(ship[i].state & SELECTED) &&
            (ship[i].now.x == ship[i].dest.x) &&
            (ship[i].now.y == ship[i].dest.y))
            ship[i].state &= ~DEPLOYING;
            
        if ((ship[i].state & SELECTED) && (flasher < MUXCOUNT / 2))
            ;
        else
            display_ship(i);
    }

    if (check_pgrid())
    {
        game.display.setCursor(82, 72);
        game.display.color = 12;
        game.display.print("Done");
        game.display.drawRect(73, 72, 6, 7);
    }   
}

/* Allow player to select target in computer grid */
void engage_ships(void)
{
    int m1x;
    int m1y;
    int shipno;
    
    game.display.fillScreen(3);

    game.display.color = 10;
    
    if (game.buttons.upBtn())
    {
        bdelay++;
        if (bdelay > BTHRESH)
            if (cursy >= 1)
                cursy--;
    }
    else if (game.buttons.downBtn())
    {
        bdelay++;
        if (bdelay > BTHRESH)
            if (cursy < 16)
                cursy++;
    }
    else if (game.buttons.leftBtn())
    {
        bdelay++;
        if (bdelay > BTHRESH)
            if (cursx >= 1)
                cursx--;
    }
    else if (game.buttons.rightBtn())
    {
        bdelay++;
            if (bdelay > BTHRESH)
            if (cursx < 21)
                cursx++;
    }
    else
        bdelay = 0;   

    if (game.buttons.released(BTN_A))
    {
        if (cursx >= 1 && cursx <= 10 &&
            cursy >= 4 && cursy <= 13)
        {
        snd.playTone(1, 200, 255, 200);
        shipno = rand() % numships;
        mx = ship[shipno].now.x;
        my = ship[shipno].now.y;

        /* The goal here is to fly a parabolic arc from the selected */
        /* ship to the cursor position.  This calculation might be */
        /* returning the wrong values for a nice parabola, but it works */
        m1x = cursx0();
        m1y = cursy0();
        a_num = m1y - my;
        a_den = (m1x * m1x) - (mx * mx);
        arc_c = m1y - ((a_num * m1x * m1x) / a_den);
        game_state = FLY_MISSILE;
        }
    }

    game.display.setCursor(5, 5);
    game.display.color = 12;
    game.display.print("Select Target");
                
    show_cursor();
    draw_cgrid();
    draw_pgrid();
}

void computer_plays(void)
{
    int m1x;
    int m1y;
    
    game.display.fillScreen(3);
    
    game.display.setCursor(5, 5);
    game.display.color = 12;
    game.display.print("Computer shoots");    
    
    show_cursor();
    draw_cgrid();
    draw_pgrid();

    target_pgrid();
    snd.playTone(1, 200 + 10 * cursx, 255, 200);
    computer_guessing++;
    if (computer_guessing > 10)
    {
       snd.playTone(1, 200, 255, 200);
       mx = 20;
       my = 20;
       m1x = cursx0();
       m1y = cursy0();
       a_num = m1y - my;
       a_den = (m1x * m1x) - (mx * mx);
       arc_c = m1y - ((a_num * m1x * m1x) / a_den);
       computer_playing = 1;
       game_state = FLY_MISSILE;
    }  
}

/* Animate missile flying from source to target at cursor */
void fly_missile(void)
{
    game.display.fillScreen(3);
    show_cursor();
    draw_cgrid();
    draw_pgrid();
    if (computer_playing)
    {
        mx++;
        if (mx >= cursx0())
        {
            pgrid[cursx - 12][cursy - 4] |= GRID_SHOT;
            cursx = oldx;
            cursy = oldy;
            computer_playing = 0;
            /* How to set noise with playTone()? */
            setOSC(&osc1,1,4,400,255,500);
            if (check_winner())
                game_state = WINNER;
            else
                game_state = ENGAGE;
        }
    }
    else
    {
        mx--;
        if (mx <= cursx0())
        {
            cgrid[cursx - 1][cursy - 4] |= GRID_SHOT;
            oldx = cursx;
            oldy = cursy;
            cursx = 12;
            cursy = 4;
            setOSC(&osc1,1,4,400,255,500);
            computer_guessing = 0;
            if (check_winner())
                game_state = WINNER;
            else
                game_state = COMPUTER_PLAYS;
        }
    }
    my = (mx * mx * a_num) / a_den + arc_c;
    game.display.drawBitmap(mx, my, missile);
}

/* Display winner screen */
void winner(void)
{
    game.display.fillScreen(3);
    show_cursor();
    draw_cgrid();
    draw_pgrid();
    
    game.display.setCursor(5, 5);
    game.display.color = 12;
    if (check_winner() == 2)
        game.display.print("Computer Wins!");
    else
        game.display.print("You Win!");
    game.display.setCursor(5, 72);
    game.display.print("A to play again");
    
    if (game.buttons.released(BTN_A))
    {
        reset_game();
    }
}

/* Main loop */
int main ()
{
    reset_game();
    game.begin();
    game.display.width = 110;
    game.display.height = 88;
    game.display.invisiblecolor = 14;
    game.display.bgcolor = 3;
    game.display.setFont(font5x7);
    srand(game.getTime());

    game.display.loadRGBPalette(palettePico);
    game.setFrameRate(30);
    while (game.isRunning()) {
//      show_debug();

        if (game.update()) {
            switch (game_state)
            {
            case SPLASH:
                splash_screen();
                break;
            case SETUP:
                setup_game();
                break;
            case PLACE:
                place_ships();
                break;
            case ENGAGE:
                engage_ships();
                break;
            case FLY_MISSILE:
                fly_missile();
                break;
            case COMPUTER_PLAYS:
                computer_plays();
                break;
            case WINNER:
                winner();
                break;
            }
        }
    }
    return 0;
}
