/* L298HBridge Library v1.0
 * Copyright (c) 2017 Riaan Ehlers
 * riaan.ehlers@nmmu.ac.za
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef L298HBridge_H
#define L298HBridge_H
 
#include "mbed.h"
 
/** Class library for a L298 H-Bridge. The class is written for one H-Bridge of
 * the L298. Constructing the class twice will enable you to use both H-bridges.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "L298HBridge.h"

 * L298HBridge Motor(PB_4, PC_4, PC_5);
   
 * int main() 
 * {
 *    float i;
 *    while(1) 
 *    {
 *        Motor.Fwd();
 *       
 *        for(i=0;i<100;i++)
 *        {
 *            Motor.Speed(i);
 *            wait(0.1);
 *        }
 *                
 *        for(i=100;i>25;i--)
 *        {
 *            Motor.Speed(i);
 *            wait(0.1);
 *        }
 *        
 *        Motor.Rev();
 *                
 *        for(i=0;i<100;i++)
 *        {
 *            Motor.Speed(i);
 *            wait(0.1);
 *        }
 *                
 *        for(i=100;i>25;i--)
 *        {
 *            Motor.Speed(i);
 *            wait(0.1);
 *        }
 *    }
 * }
 * @endcode
 */
 
class L298HBridge {
  public:
    /** Create a L298HBridge object connected to the specified pins. 
    * Once created, the motor speed will be set to 0 (PWM signal will be 0%) and
    * the motor will be in the stop mode (neither forward or reverse).
    * @param ENPin  PwmOut compatible pin used to connect to L298's En(x) pin associated with enabling the H-Bridge.
    * @param FWDPin GPIO pin used to connect to L298's In(x) pin associated with forward direction.
    * @param REVPin GPIO pin used to connect to L298's In(x) pin associated with reverse direction.
    */
    L298HBridge(PinName ENPin, PinName FWDPin, PinName REVPin);
    
    /** Configure the H-Bridge to run the motor in the forward direction.    
    * @param None
    */
    void Fwd();
    
    /** Configure the H-Bridge to run the motor in the reverse direction.    
    * @param None
    */
    void Rev();
    
    /** Switch the H-Bridge off. The H-Bridge is not set to forward or reverse. 
    * @param None
    */
    void Stop();
    
   /** Change the motor's speed by adjusting the PWM signal.
    *  The value passed to the function can be any value from 0 to 100.
    *  Where 0 = 0% and 100 = 100%.
    * @param DutyPercent
    */
    void Speed(float DutyPercent);
 
  private:
    PwmOut _ENPin;
    DigitalOut _FWDPin, _REVPin;
};
 
#endif