#include "mbed.h"
#include <math.h>
#include <string.h>
#include "ILI9341.h"
#include "Arial28x28.h"
#include "meter.h"
#include "main.h"

#ifndef M_PI
#define M_PI 3.141593
#endif

extern ILI9341      *tft ;
extern meter        *tacho ;
double SIN[451] ; /* sin table for 45 to 90 degrees in 0.1 deg step */
double COS[451] ; /* cos table for 45 to 90 degrees in 0.1 deg step */

static void fillTriTable(void)
{
    double theta ;
    int i ;
    for (i = 0 ; i <= 450 ; i++ ) {
        theta = M_PI * (((double)(i + 450))/1800.0) ;
        SIN[i] = sin( theta ) ;
        COS[i] = cos( theta ) ;
    }
}

meter::meter(int x, int y, int width, int height, float min, float max) 
{
    _x = x ;
    _y = y ;
    _w = width ;
    _h = height ;
    _min = min ;
    _max = max ;
    _center_x = _x + _w / 2 ;
    _center_y = _y + (2 * _h) / 3 ;
    fillTriTable() ;
}

meter::~meter(void) 
{
}

void meter::drawFrame(void)
{
    tft->BusEnable(true) ;
    tft->fillrect(_x, _y, _x+ _w - 1, _y+(2 * _h)/3 - 1, White) ;
    tft->fillrect(_x, _y+(2 * _h)/3 - 1, _x + _w - 1, _y + _h - 1, Black) ;
    tft->rect(_x, _y, _x + _w - 1, _y + _h - 1, Green) ;
    tft->BusEnable(false) ;
}

void meter::drawScale(void)
{
    int x1, x2, y1, y2 ;
    double radius ;
    double append ;
    radius = _w / 2.0 ;
    tft->BusEnable(true) ;
    for (int i = 0 ; i < 450 ; i += 45 ) {
        x1 = (radius * COS[i] + 0.5) ;
        y1 = (radius * SIN[i] + 0.5) ;
        if ((i % 10) == 5) {
            append = 5 ;
        } else {
            append = 10 ;
        }
        x2 = ((radius + append) * COS[i] + 0.5) ;
        y2 = ((radius + append) * SIN[i] + 0.5) ;
        tft->line(_center_x + x1, _center_y - y1, _center_x + x2, _center_y - y2, Black) ;
        tft->line(_center_x - x1, _center_y - y1, _center_x - x2, _center_y - y2, Black) ;
    }
    tft->line(_center_x, _center_y - radius, _center_x, _center_y - (radius + 10), Black) ;
    tft->BusEnable(false) ;
}

void meter::drawHand(float value)
{
    static int prev_x1 = 0, prev_x2 = 0, prev_y1 = 0, prev_y2 = 0 ;
    int x1, x2, y1, y2 ;
    double theta ;
    double radius = _w / 20.0 ;
    double append = 8.0 * (_w / 20.0)  ;
    int i ;
    
    theta = 90.0 - (90.0 * (value / _max)) ;
    if (theta < 0.0) { theta = 0.0 ; }
    if (theta > 90.0) { theta = 90.0 ; }
    if (theta > 45.0) {
        i = (10.0 * (90.0 - theta) + 0.5) ; 
        x1 = _center_x - (radius * COS[i] + 0.5) ;
        x2 = _center_x - ((radius + append) * COS[i] + 0.5) ;
    } else {
        i = (10.0 * theta + 0.5) ;
        x1 = _center_x + (radius * COS[i] + 0.5) ;
        x2 = _center_x + ((radius + append) * COS[i] + 0.5) ;
    }

    y1 = _center_y - (radius * SIN[i] + 0.5) ;
    y2 = _center_y - ((radius + append) * SIN[i] + 0.5) ;
    tft->BusEnable(true) ;
    tft->line(prev_x1, prev_y1, prev_x2, prev_y2, White) ; /* erase prev */
    tft->line(x1, y1, x2, y2, Black) ;                     /* draw current */
    tft->BusEnable(false) ;
    prev_x1 = x1 ;
    prev_x2 = x2 ;
    prev_y1 = y1 ;
    prev_y2 = y2 ;
}

void meter::draw(float value) 
{
    tft->BusEnable(true) ;
    drawFrame() ;
    drawScale() ;
    drawHand(value) ;
    drawValue(value) ;
    tft->BusEnable(false) ;
}

void meter::drawValue(float value)
{
    char str[32] ;
    int v1, v2 ;
    v1 = (int)value ;
    v2 = ((int)(1000.0 * value) % 1000) ;
    sprintf(str, "%d.%03d V", v1, v2) ;
    tft->BusEnable(true) ;
    tft->locate(80, 190) ;
    tft->printf(str) ;
    wait(0.01) ;
    tft->BusEnable(false) ;
}

void meter::update(float value) 
{
    tft->BusEnable(true) ;
    drawHand(value) ;
    drawValue(value) ;
    tft->BusEnable(false) ;
}