/** mbed oscilloscope my implementation of a oscillo scope
 * Copyright (c) 2014, 2015 Motoo Tanaka @ Design Methodology Lab
 *
 * view.cpp
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "mbed.h"
#include <ILI9341.h>
#include "SPI_STMPE610.h"
#include "Arial12x12.h"
#include "Arial24x23.h"
#include "Arial28x28.h"
#include "Arial43x48_numb.h"
#include "vt100.h"
#include "TFTMenu.h"
#include "main.h"
#include "trig.h"
#include "view.h"

int pane_w = 200 ;
int pane_h = 70 ; // height per channel
int head_h = 20 ; // header height
int foot_h = 20 ; // footer height
int right_margin = 10 ;
int left_margin = 30 ;

void initTFT(void)
{
    //Configure the display driver
    TFT.BusEnable(true) ;
    TFT.FastWindow(true) ;
    TFT.background(Black);
    TFT.foreground(White);
    wait(0.01) ;
    TFT.cls();
    TFT.BusEnable(false) ;
}


void eraseChFrame(int ch)
{
    int x1, x2, y1, y2 ;
    x1 = left_margin ;
    x2 = left_margin + pane_w ;
    y1 = head_h + (pane_h + 5) * ch ;
    y2 = y1 + pane_h ;
    TFT.BusEnable(true) ;
    TFT.fillrect(x1,y1,x2, y2, Black) ;
    TFT.BusEnable(false) ;
}

void eraseGraphFrames(void)
{
    int ch ;
    TFT.BusEnable(true) ;
    for (ch = 0 ; ch < numAnalogIn ; ch++) {
        eraseChFrame(ch) ;
    }
    TFT.BusEnable(false) ;
}

void drawChFrame(int ch)
{
    int x1, x2, y1, y2,y_offset ;
    x1 = left_margin ;
    x2 = left_margin + pane_w ;
    y1 = head_h + (pane_h + 5) * ch ;
    y2 = y1 + pane_h ;
    TFT.BusEnable(true) ;
    TFT.set_font((unsigned char*) Arial12x12);
    TFT.background(Black) ;
    TFT.foreground(White) ;
    TFT.locate(5, y1+5) ;
    TFT.printf("3.0") ;
    TFT.locate(15, (y1+y2)/2) ;
    TFT.printf("V") ;
    TFT.locate(5, y2-10) ;
    TFT.printf("0.0") ;
    TFT.rect(x1,y1,x2,y2,Blue) ;
    y_offset = head_h + 1;
 
    for (int i = 0 ; i < pane_w ; i += 10) {
        x1 = left_margin + i ;
        x2 = x1 ;
        if ((i % 100) == 0) {
            y1 = head_h - 5 ;
        } else {
            y1 = head_h - 2 ;
        }
        y2 = head_h ;
        TFT.line(x1, y1, x2, y2, Blue) ;
        y1 = head_h + (pane_h + 5) * 2 - 5 ;
        if ((i % 100) == 0) {
            y2 = y1 + 5 ;
        } else {
            y2 = y1 + 2 ;
        }
        TFT.line(x1, y1, x2, y2, Blue) ;
    }

    for (float f = 0.0 ; f < vref ; f += 1.0) {
        y_offset = head_h + 1;
        y1 = ((pane_h - 2) * (1.0 - f/vref)) + y_offset ;
        y2 = y1 ;
        x1 = left_margin - 3 ;
        x2 = left_margin ;
        TFT.line(x1, y1, x2, y2, Blue) ;
        y_offset = head_h + (pane_h + 5) + 1;
        y1 = ((pane_h - 2) * (1.0 - f/vref)) + y_offset  ;
        y2 = y1 ;
        TFT.line(x1, y1, x2, y2, Blue) ;
    }
    TFT.BusEnable(false) ;
}

bool inChFrame(int ch, int x, int y)
{
    bool result = false ;
    int x1, x2, y1, y2 ;
    x1 = left_margin ;
    x2 = left_margin + pane_w ;
    y1 = head_h + (pane_h + 5) * ch ;
    y2 = y1 + pane_h ;
    if ((x1 < x)&&(x < x2)&&(y1 < y)&&(y < y2)) { // on 5-Mar-2015 '=' removed
        result = true ;
    }
    return( result ) ;
}

void plotCh_line(int turn, int ch, int index, int color)
{
    int x[2], y[2] ;
    int y_offset, data_pos, prev_pos ;
    
    y_offset = head_h + (pane_h + 5) * ch + 1;
    data_pos = (bor[turn] + index) % memLength ;
    prev_pos = (memLength + bor[turn] + index - 1) % memLength ;
    
    x[1] = left_margin + index + 1; 
    y[1] = ((pane_h - 2) * (1.0 - u2f(udata[turn][ch][data_pos]))) + y_offset ;

    if (index == 0) {
        x[0] = x[1] ;
        y[0] = y[1] ;
    } else {
        x[0] = left_margin + index ;
        y[0] = ((pane_h - 2) * (1.0 - u2f(udata[turn][ch][prev_pos]))) + y_offset ;
    }
    TFT.line(x[0],y[0],x[1],y[1], color) ;
}

void drawGraphFrames(void)
{
    TFT.BusEnable(true) ;
    for (int i = 0 ; i < numAnalogIn ; i++ ) {
        drawChFrame(i) ;
    }
    TFT.BusEnable(false) ;
}

void drawChart(void)
{
    int i, ch ;
    if (!frame_full) {
        return ;
    }
    if (mode == MODE_RUN) {
        timer.detach() ;
    }

    TFT.BusEnable(true) ;
    for (i = 0 ; i < memLength ; i++ ) {
        for (ch = 0 ; ch < numAnalogIn ; ch++ ) {
            plotCh_line(prev_page, ch, i, Black) ; // erase prev
            plotCh_line(page, ch, i, White) ;
        }
    }
    TFT.BusEnable(false) ;
    prev_page = page ;
    page = (page + 1)%2 ;
    sampling_status = ST_PRE_TRIG ;
    frame_full = false ;
    trig_index = 0 ;
    data_index = 0 ;
    if (mode == MODE_RUN) {
        timer.attach_us(&sampleAD, us_interval) ;
    }
}

void printMode(void)
{
    TFT.BusEnable(true) ;
    TFT.locate(80, 200) ;
    TFT.background(Black) ;

    if (mode == MODE_STOP) {
        TFT.foreground(Yellow) ;
        TFT.printf(" Stopped") ;
    } else if (mode == MODE_RUN) {
        TFT.foreground(White) ;
        TFT.printf(" Running") ;
    }
    TFT.BusEnable(false) ;
}

void printTimeDiv(void)
{
    char str[32] ;
    int ival, dotval ;
    if (us_interval < 100) { // 10 * us_interval < 1ms
        sprintf(str, " %3d us/div", 10 *us_interval) ;
    } else if (us_interval < 100000) { // 10 * us_interval >= 1ms
        ival = us_interval / 100 ;
        dotval = us_interval % 100 ;
        sprintf(str, " %3d.%0d ms/div", ival, dotval) ;
    }
    TFT.BusEnable(true) ;
    TFT.fillrect(65, 240, 175, 260, Black) ;
        
    TFT.background(Black) ;
    TFT.foreground(White) ;

    TFT.locate(65, 240) ;
    TFT.printf(str) ;
    TFT.BusEnable(false) ;
}

void printTrigMode(void)
{
    char str[60] ;
    TFT.BusEnable(true) ;
    TFT.fillrect(65, 280, 235, 310, Black) ;
 
    TFT.background(Black) ;
    TFT.foreground(White) ;

    if (trig_mode == TRIG_MODE_NONE) {
        sprintf(str, "%6s",
        trig_name[trig_mode]) ;
    } else {
        sprintf(str, "%6s ch%1d (%d, %.1f)",
        trig_name[trig_mode], trig_ch+1, trig_pos, trig_level) ;
    }
    TFT.locate(65, 280) ;
    TFT.printf(str) ;
    TFT.BusEnable(false) ;
}

void printTrigLevel(void)
{
    char str[16] ;
    sprintf(str, " %.1f", trig_level) ;
    TFT.BusEnable(true) ;
    TFT.locate(140, 280) ;
    TFT.background(Black) ;
    TFT.foreground(White) ;
    TFT.printf(str) ;
    TFT.BusEnable(false) ;
}

void eraseTrigMark(void)
{
    int x1, x2, y1, y2 ;
    TFT.BusEnable(true) ;
    x1 = left_margin - 3 ;
    x2 = left_margin + pane_w + 3 ;
    y1 = 1 ;
    y2 = head_h - 5 ;
    TFT.fillrect(x1,y1,x2, y2, Black) ;
    y1 = (head_h + (pane_h + 5)*2) + 5  ;
    y2 = y1 + 13 ;
    TFT.fillrect(x1,y1,x2, y2, Black) ;
    x1 = left_margin + pane_w + 1 ;
    x2 = x1 + 10 ;
    y1 = 0 ;
    y2 = head_h + (pane_h + 5) * (2) + 1;
    TFT.fillrect(x1, y1, x2, y2, Black) ;
    TFT.BusEnable(false) ;
}

void drawTrigPos()
{
    uint16_t color = Green ;
    int x1, x2, y1, y2 ;
    int y_offset ;
// draw down arrow
    TFT.BusEnable(true) ;
    x1 = left_margin + trig_pos + 1 ;
    x2 = x1 ;
    y1 = 5 ;
    y2 = head_h - 8 ;
    TFT.line(x1, y1, x2, y2, color) ;
    x1 = x2 - 3 ;
    y1 = y2 - 4 ;
    TFT.line(x1, y1, x2, y2, color) ;
    x1 = x2 + 3 ;
    TFT.line(x1, y1, x2, y2, color) ;
// draw up arrow
    x1 = x2 ;
    y1 = (head_h + (pane_h + 5)*2) + 5  ;
    y2 = y1 + 7 ;
    TFT.line(x1, y1, x2, y2, color) ;
    x2 = x1 - 3 ;
    y2 = y1 + 3 ;
    TFT.line(x1, y1, x2, y2, color) ;
    x2 = x1 + 3 ;
    TFT.line(x1, y1, x2, y2, color) ;
// draw level mark
    x1 = left_margin + pane_w + 3 ;
    x2 = x1 + 7 ;
    y_offset = head_h + (pane_h * (trig_ch+1)) +  (5 * trig_ch) - 1;
    y1 = y_offset - ((pane_h - 2) * (trig_level/vref)) ;
    y2 = y1 ;
    TFT.line(x1, y1, x2, y2, color) ;
    x2 = x1 + 3 ;
    y2 = y1 - 3 ;
    TFT.line(x1, y1, x2, y2, color) ;
    y2 = y1 + 3 ;
    TFT.line(x1, y1, x2, y2, color) ;
    TFT.BusEnable(false) ;
// printf("trig_ch = %d, trig level = %.2f, y1 = %d yoffset = %d\n", 
//         trig_ch, trig_level, y1, y_offset) ;
}

void drawTrigMark(void)
{
    eraseTrigMark() ;
    switch(trig_mode) {
    case TRIG_MODE_NONE:
        break ;
    case TRIG_MODE_RISE:
    case TRIG_MODE_FALL:   
    case TRIG_MODE_LEVEL:
        drawTrigPos() ;
        break ;
    default:
        printf("Unkown trigger mode %d\n\r",trig_mode) ;
        break ;
    }
}
