/** mbed oscilloscope my implementation of a oscillo scope
 * Copyright (c) 2014, 2015 Motoo Tanaka @ Design Methodology Lab
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "mbed.h"
#include <ILI9341.h>
#include "Arial12x12.h"
#include "Arial24x23.h"
#include "Arial28x28.h"
#include "Arial43x48_numb.h"
#include "SPI_STMPE610.h"
#include "vt100.h"
#include "TFTMenu.h"
#include "menu.h"
#include "trig.h"
#include "view.h"
#include "main.h"

vt100 tty ;
AnalogIn *ach[NUM_MAX_ANALOG_CH] ;
Ticker timer ;
DigitalOut tsc_cs(PIN_CS_TSC, 1) ;
DigitalOut backlight(PIN_BL_TFT, 0) ;

ILI9341 TFT(SPI_8, 10000000, 
    PIN_MOSI, PIN_MISO,  PIN_SCLK, 
    PIN_CS_TFT, PIN_RESET_TFT, PIN_DC_TFT, "Adafruit2.8") ;
SPI_STMPE610 TSC(PIN_MOSI, PIN_MISO, PIN_SCLK, PIN_CS_TSC) ;
TFTMenuItem *menu[NUM_MAX_MENU] ;
int numMenu = 0 ;

float vref = 3.28 ; // input range 0 - 3.3V
uint16_t udata[2][NUM_MAX_ANALOG_CH][CHART_LEN] ; // data storage ;
uint16_t bor[2] = {0, 0} ; // begin of ring buffer 
int data_index = 0 ;
int numAnalogIn = 2 ;
int memLength = CHART_LEN ;
int us_interval = MIN_INTERVAL ; // 20us = 50KHz
bool frame_full = false ;
int page = 0 ; // 
int prev_page = 0 ;
int mode = MODE_RUN ;

void sampleAD(void) ;
void drawMenu(void) ;
// void eraseGraphFrames(void) ;

/**
 * u2v unsigned int data value to voltage conversion 
 */
float u2v(uint16_t uvalue)
{
    return( vref * (float)uvalue / 65535.0 ) ;
}

/**
 * v2u voltage to unsigned int data conversion
 */
uint16_t v2u(float voltage)
{
    return(0xFFFF * voltage / vref) ;
}

/**
 * u2f unsigned int to float (0 - 1.0) conversion
 */
float u2f(uint16_t value)
{
    return( (float)value / 65535.0 ) ;
}

void initADC(void)
{
    if (numAnalogIn > 0) {
        ach[0] = new AnalogIn(PIN_ADC_CH0) ;
    }
    if (numAnalogIn > 1) {
        ach[1] = new AnalogIn(PIN_ADC_CH1) ;
    }

#if defined (TARGET_KL25Z)
    ADC0->CFG1 = ADC0->CFG1 & (
        ~(
          0x80 // LDLPC = 0 ; no low-power mode
        | 0x60 // ADIV = 1
        | 0x10 // Sample time short
        | 0x03 // input clock = BUS CLK
        )
    ) ; // clkdiv <= 1
    ADC0->CFG2 = ADC0->CFG2 
        | 0x03 ; // Logsample Time 11 = 2 extra ADCK
    ADC0->SC3 = ADC0->SC3 
        & (~(0x03)) ; // hardware avarage off
#endif
#if defined (TARGET_MAX32600MBED)
    uint32_t *pga_ctrl = (uint32_t*)0x40054004 ; /* ADC_PGA_CTRL */
    printf("PGA_CTRL(before) = 0x%08X\n", *pga_ctrl) ;
//    *pga_ctrl |= 0x02 ; /* clear bit[1:0] to set gain x1 (default) */
//    *pga_ctrl |= 0x40 ; /* skip PGA */
    printf("PGA_CTRL(after) = 0x%08X\n", *pga_ctrl) ;
#endif
}

void envChanged(void)
{
    timer.detach() ;
    
    eraseGraphFrames() ;
    drawGraphFrames() ;
    
    data_index = 0 ;
    page = 0 ; 
    prev_page = 0 ;
    frame_full = false ;
    if (trig_mode == TRIG_MODE_NONE) {
        sampling_status = ST_TRIG_HIT ;
        post_trig_index = 0 ;
    } else {
        sampling_status = ST_PRE_TRIG ;
    }
/*
    if (mode == MODE_RUN) {
        timer.attach_us(&sampleAD, us_interval) ;
    }
*/
}

bool checkTrigger(void)
{
    bool result = false ;
    int prev_index ;
    uint16_t current, prev ;
    if (data_index > 0) {
        prev_index = data_index - 1 ;
    } else {
        prev_index = memLength - 1 ; // memLength - 1 ;
    }
    current = udata[page][trig_ch][data_index] ;
    prev = udata[page][trig_ch][prev_index] ;
    
    switch(trig_mode) {
    case TRIG_MODE_NONE: // no trigger, return true at the beggining 
        trig_index = 0 ;
        result = true ;
        break ;
    case TRIG_MODE_RISE:
        if ((prev < utrig_level)&&(utrig_level <= current)) {
            result = true ;
        }
        break ;
    case TRIG_MODE_FALL:
        if ((prev > utrig_level)&&(utrig_level >= current)) {
            result = true ;
        } 
        break ;
    case TRIG_MODE_LEVEL:
        if (current > trig_level_margin) {
            if (((current + trig_level_margin) >= utrig_level)&&(utrig_level >= (current - trig_level_margin))) {
                result = true ;
            }
        } else {
            if ((current + trig_level_margin) >= utrig_level) {
                result = true ;
            }
        }
        break ;
    default:
        printf("Error: Unknown trigger mode %d\n", trig_mode) ;
        break ;
    }
    return( result ) ;
}

void sampleAD(void)
{
    if (frame_full) {
        return ;
    }
// timer.detach() ;
    for (int i = 0 ; i < numAnalogIn ; i++ ) {
        udata[page][i][data_index] = ach[i]->read_u16() ;
    }
// timer.attach_us(&sampleAD, us_interval) ;
    switch(sampling_status) {
    case ST_PRE_TRIG:
        if (data_index >= trig_pos) {
            sampling_status = ST_TRIG_WAIT ;
        } else {
            data_index = (data_index + 1) % memLength ;
        }
        break ;
    case ST_TRIG_WAIT:
        if (checkTrigger()) {
            trig_index = data_index ;
            post_trig_index = 0 ;
            sampling_status = ST_TRIG_HIT ;
        } else {
            data_index = (data_index + 1) % memLength ;
        }
        break ;
    case ST_TRIG_HIT:
        post_trig_index++ ;
        if (post_trig_index >= post_trig_len) {
            sampling_status = ST_BUF_FULL ;
            frame_full = true ;
            bor[page] = (memLength + trig_index - trig_pos) % memLength ;
            post_trig_index = 0 ;
            data_index = 0 ;
        } else {
            data_index = (data_index + 1) % memLength ;
        }
        break ;
    case ST_BUF_FULL:
        printf("mode = ST_BUF_FULL\n\r") ;
        break ;
    default:
        printf("unknown sample status %d\n", sampling_status) ;
        break ;
    }
}


bool doTrigValue(int x, int y)
{
    int i, y_offset ;
    bool result = false ;
    for (i = 0 ; i < numAnalogIn ; i++ ) {
        if (inChFrame(i, x, y)) {
            result = true ;
            trig_ch = i ;
            trig_pos = x - left_margin;
            post_trig_len = memLength - trig_pos ;
            y_offset = head_h + pane_h * i + 5 * (i - 1) + 1;
            trig_level = vref *(1 -  ((float)(y - y_offset) / (float)(pane_h - 2))) ;
            if (trig_level > vref) {
                trig_level = vref ;
            } else if (trig_level < 0.0) {
                trig_level = 0.0 ;
            }
            utrig_level = v2u( trig_level ) ; 
            break ;
        }
    }
    return( result ) ;
}

void hello()
{
    printf("=== simple oscilloscope ===\n\r") ;
    printf("built: %s\n\r",__DATE__) ;
}

int main() 
{
    int touched ;
    uint16_t x, y ;
    bool result ;
    tty.cls() ;
    backlight = 0 ;
    initTFT() ;
    initADC() ;
    initMenu() ;
    
    us_interval = MIN_INTERVAL ; 
    
    drawGraphFrames() ;
    drawMenu() ;
    
    backlight = 1 ;
    hello() ;
    timer.attach_us(&sampleAD, us_interval) ;
    while(1) {
        drawChart() ;
        touched = TSC.getPoint(&x, &y) ;
        if (touched && ((x != 0) && (y != 0))) { // discard value 0 for x, y
// printf("x = %d, y = %d\n",x, y) ;
            if (mode == MODE_RUN) {
                timer.detach() ;
            }
            result = doMenu(x, y) ;
            if (result != true) {
                result = doTrigValue(x, y) ;
                if (result) {
                    drawTrigMark() ;
                    printTrigMode() ;
                }
            }   
            if (mode == MODE_RUN) {
                   timer.attach_us(&sampleAD, us_interval) ;
            }
        }
        wait(0.1) ;
    }
}
