/** maze_vt100 a simple maze using vt100 and MMA8451Q libraries
 */
#include "mbed.h"
#include "MMA8451Q.h"
#include "vt100.h"
#include "maze.h"

#define DIR_STAY  0
#define DIR_UP    1
#define DIR_DOWN  2
#define DIR_RIGHT 3 
#define DIR_LEFT  4

#define MMA8451_I2C_ADDRESS (0x1D<<1)

typedef struct _pos {
    int x ;
    int y ;
} pos_type ;

MMA8451Q acc(PTE25, PTE24, MMA8451_I2C_ADDRESS) ;
vt100 tty ;
float threshold = 0.2 ;

/** Check if two pos_type values are same
 * @param pos_type a
 * @param pos_type b
 * @returns if a and b are same position
 */
bool isSame(pos_type a, pos_type b)
{
    return((a.x == b.x)&&(a.y == b.y)) ;
}

/** Draw the maze defined in the "maze.h"
 * @param pos_type *current actually the start point
 * @param pos_type *goal the position of the goal
 * @note those params are actually returned by this function
 */
void drawTextMaze(pos_type *current, pos_type *goal) 
{
    int x, y ;
    char c ;
    
    tty.setBG(7) ; // set background to white
    tty.setFG(0) ; // set foreground to black
    tty.cls() ;
    
    for (y = 0 ; y < MAZE_H ; y++ ) {
        for (x = 0 ; x < MAZE_W ; x++ ) {
            switch(maze[y][x]) {
            case 0:  // path
                c = ' ' ; 
                break ; 
            case 1:  // wall
                c = ' ' ; 
                tty.setBG(0) ; // set background to black
                tty.setFG(7) ; // set foreground to white
                break ; 
            case 2: // Start point  
                c = 'S' ;
                current->x = x ;
                current->y = y ; 
                break ; // start
            case 3: // Goal  
                c = 'G' ;
                tty.green() ;
                goal->x = x ;
                goal->y = y ; 
                break ; 
            default: // should not be here 
                c = '?' ; 
                break ; // wth?
            }
            tty.putChar(x+1, y+1, c) ;
            tty.setBG(7) ; // set background to white
            tty.setFG(0) ; // set foreground to black
        }
        printf("\n\r") ;
    }
}

/** Filter out too little move 
 * @param float in returned value from the acc
 * @returns float result filtered value of in
 */
float filterVal(float in)
{
    float result = 0.0 ;
    if ((-threshold > in)||(in > threshold)) {
        result = in ;
    }
    return( result ) ;
}

/** Decide which direction to go
 * @param float res[] acc value of x, y 
 * @returns int direction to move 
 */
int getDirection(float res[])
{
    float dx, dy ;
    int direction = DIR_STAY ;
    dx = filterVal(res[0]) ;
    dy = filterVal(res[1]) ;

    if ((dx*dx) > (dy*dy)) { // holizontal move
        if (dx > 0.0) {
            direction = DIR_DOWN ;
        } else if (dx < 0.0) {
            direction = DIR_UP ;
        }
    } else { // vertical move
        if (dy > 0.0) {
            direction = DIR_RIGHT ;
        } else if (dy < 0.0) {
            direction = DIR_LEFT ;
        }
    }
    return(direction) ;
}

/** Get next positon to move to
 * @param pos_type current where we are now
 * @param int direction which way we'd like to move
 * @returns the candidate positon for the next move
 */
pos_type getNext(pos_type current, int direction) 
{
    pos_type next = current ;
    switch(direction) {
    case DIR_STAY: 
        break ;
    case DIR_UP: 
        if (next.y > 0) { 
            next.y-- ; 
        }  
        break ;
    case DIR_DOWN: 
        if (next.y < (MAZE_H - 1)) { 
            next.y++ ; 
        } 
        break ;
    case DIR_RIGHT: 
        if (next.x < (MAZE_W - 1)) { 
            next.x++ ; 
        } 
        break ;
    case DIR_LEFT: 
        if (next.x > 0) { 
            next.x-- ; 
        } 
        break ;
    default: 
        break ;
    }
    return( next ) ;
}

/** Notice of the goal
 */
void showGoal(void)
{
    tty.blue() ;
    tty.frame((MAZE_W/2)-4,(MAZE_H/2)-1,(MAZE_W/2)+4,(MAZE_H/2)+1) ;
    tty.red() ;
    tty.locate((MAZE_W/2)-3, (MAZE_H/2)) ;
    printf("G O A L\n\r") ;
}

/** Check if we can move to the next position
 * @param pos_type next the position we'd like to move next
 * @returns if the position is empty (movable)
 */
bool checkMove(pos_type next) 
{
    bool result = false ;
    
    switch(maze[next.y][next.x]) {
    case POS_PATH:
    case POS_GOAL:
        result = true ;
        break ;
    case POS_START:
    case POS_WALL:
    default:
        result = false ;
        break ;
    } 
    return( result ) ;
}

/** main a simple maze program
 */
int main() {
    float res[3] ;
    pos_type current, next, goal ;
    int direction = DIR_STAY ;
    
    tty.cls() ;
    drawTextMaze(&current, &goal) ;
    tty.black() ;
    tty.putChar(current.x+1, current.y+1, '@') ;
    printf("\r\n") ;
    
    for (;;) {
        acc.getAccAllAxis(res) ;
        direction = getDirection(res) ;
        next = getNext(current, direction) ;
        if ((!isSame(current, next)) && checkMove(next)) {
           tty.putChar(current.x+1, current.y+1, ' ') ;
           tty.putChar(next.x+1, next.y+1, '@') ;
           tty.green() ;
           tty.putStr(1, MAZE_H+1, "\r\n") ;
           tty.black() ;
           current = next ;
           if (isSame(next, goal)) {
             break ; // goal in!
           }
        }
        wait(0.1) ;
    }
    showGoal() ;
    for (;;) {
        // wait for ever for reset
    }
}
