/** Table for sin() function 
 * Copyright (c) 2014, 2015 Motoo Tanaka @ Design Methodology Lab
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
/*
 * Since the target MCU KL25Z in FRDM-KL25Z does not have a FPU
 * I decided to use table instead of calling software implentation of
 * the sin() function, otherwise it was quite difficult to make the
 * program work in realtime.
 * Meantime, since the flash memory is limited, 
 * that table has only the first 1/4th (0 degree to 90 degree)
 * Although the value of sin() is -1.0 to 1.0
 * to implement it in unsigned int, the value is shifted to
 * 0x0000 to 0x7FFF, which stands for 0.0 to 1.0
 * with positive bias of 0x8000, which is defined as SIN_ZERO.
 * So negative sin() values are 0x0000 to 0x7FFF (-1.0 ~ -0.0?)
 * and positive sin() values are 0x8000 to 0xFFFF (0.0 ~ 1.0)
 */
#ifndef _SIN_TABLE_H_
#define _SIN_TABLE_H_
#define SIN_ZERO 0x8000
#define NUM_SIN_TABLE 2500
unsigned short sin_table[] = {
0x0000,0x0015,0x0029,0x003E,0x0052,0x0067,0x007C,0x0090,
0x00A5,0x00B9,0x00CE,0x00E2,0x00F7,0x010C,0x0120,0x0135,
0x0149,0x015E,0x0173,0x0187,0x019C,0x01B0,0x01C5,0x01DA,
0x01EE,0x0203,0x0217,0x022C,0x0240,0x0255,0x026A,0x027E,
0x0293,0x02A7,0x02BC,0x02D1,0x02E5,0x02FA,0x030E,0x0323,
0x0337,0x034C,0x0361,0x0375,0x038A,0x039E,0x03B3,0x03C8,
0x03DC,0x03F1,0x0405,0x041A,0x042E,0x0443,0x0458,0x046C,
0x0481,0x0495,0x04AA,0x04BE,0x04D3,0x04E8,0x04FC,0x0511,
0x0525,0x053A,0x054E,0x0563,0x0578,0x058C,0x05A1,0x05B5,
0x05CA,0x05DE,0x05F3,0x0608,0x061C,0x0631,0x0645,0x065A,
0x066E,0x0683,0x0697,0x06AC,0x06C1,0x06D5,0x06EA,0x06FE,
0x0713,0x0727,0x073C,0x0750,0x0765,0x077A,0x078E,0x07A3,
0x07B7,0x07CC,0x07E0,0x07F5,0x0809,0x081E,0x0833,0x0847,
0x085C,0x0870,0x0885,0x0899,0x08AE,0x08C2,0x08D7,0x08EB,
0x0900,0x0915,0x0929,0x093E,0x0952,0x0967,0x097B,0x0990,
0x09A4,0x09B9,0x09CD,0x09E2,0x09F6,0x0A0B,0x0A1F,0x0A34,
0x0A48,0x0A5D,0x0A71,0x0A86,0x0A9B,0x0AAF,0x0AC4,0x0AD8,
0x0AED,0x0B01,0x0B16,0x0B2A,0x0B3F,0x0B53,0x0B68,0x0B7C,
0x0B91,0x0BA5,0x0BBA,0x0BCE,0x0BE3,0x0BF7,0x0C0C,0x0C20,
0x0C35,0x0C49,0x0C5E,0x0C72,0x0C87,0x0C9B,0x0CB0,0x0CC4,
0x0CD9,0x0CED,0x0D02,0x0D16,0x0D2A,0x0D3F,0x0D53,0x0D68,
0x0D7C,0x0D91,0x0DA5,0x0DBA,0x0DCE,0x0DE3,0x0DF7,0x0E0C,
0x0E20,0x0E35,0x0E49,0x0E5E,0x0E72,0x0E86,0x0E9B,0x0EAF,
0x0EC4,0x0ED8,0x0EED,0x0F01,0x0F16,0x0F2A,0x0F3E,0x0F53,
0x0F67,0x0F7C,0x0F90,0x0FA5,0x0FB9,0x0FCE,0x0FE2,0x0FF6,
0x100B,0x101F,0x1034,0x1048,0x105C,0x1071,0x1085,0x109A,
0x10AE,0x10C3,0x10D7,0x10EB,0x1100,0x1114,0x1129,0x113D,
0x1151,0x1166,0x117A,0x118F,0x11A3,0x11B7,0x11CC,0x11E0,
0x11F5,0x1209,0x121D,0x1232,0x1246,0x125A,0x126F,0x1283,
0x1298,0x12AC,0x12C0,0x12D5,0x12E9,0x12FD,0x1312,0x1326,
0x133A,0x134F,0x1363,0x1377,0x138C,0x13A0,0x13B5,0x13C9,
0x13DD,0x13F2,0x1406,0x141A,0x142F,0x1443,0x1457,0x146C,
0x1480,0x1494,0x14A8,0x14BD,0x14D1,0x14E5,0x14FA,0x150E,
0x1522,0x1537,0x154B,0x155F,0x1574,0x1588,0x159C,0x15B0,
0x15C5,0x15D9,0x15ED,0x1602,0x1616,0x162A,0x163E,0x1653,
0x1667,0x167B,0x1690,0x16A4,0x16B8,0x16CC,0x16E1,0x16F5,
0x1709,0x171D,0x1732,0x1746,0x175A,0x176E,0x1783,0x1797,
0x17AB,0x17BF,0x17D3,0x17E8,0x17FC,0x1810,0x1824,0x1839,
0x184D,0x1861,0x1875,0x1889,0x189E,0x18B2,0x18C6,0x18DA,
0x18EE,0x1903,0x1917,0x192B,0x193F,0x1953,0x1968,0x197C,
0x1990,0x19A4,0x19B8,0x19CC,0x19E1,0x19F5,0x1A09,0x1A1D,
0x1A31,0x1A45,0x1A5A,0x1A6E,0x1A82,0x1A96,0x1AAA,0x1ABE,
0x1AD2,0x1AE6,0x1AFB,0x1B0F,0x1B23,0x1B37,0x1B4B,0x1B5F,
0x1B73,0x1B87,0x1B9C,0x1BB0,0x1BC4,0x1BD8,0x1BEC,0x1C00,
0x1C14,0x1C28,0x1C3C,0x1C50,0x1C64,0x1C78,0x1C8D,0x1CA1,
0x1CB5,0x1CC9,0x1CDD,0x1CF1,0x1D05,0x1D19,0x1D2D,0x1D41,
0x1D55,0x1D69,0x1D7D,0x1D91,0x1DA5,0x1DB9,0x1DCD,0x1DE1,
0x1DF5,0x1E09,0x1E1D,0x1E31,0x1E45,0x1E59,0x1E6D,0x1E81,
0x1E95,0x1EA9,0x1EBD,0x1ED1,0x1EE5,0x1EF9,0x1F0D,0x1F21,
0x1F35,0x1F49,0x1F5D,0x1F71,0x1F85,0x1F99,0x1FAD,0x1FC1,
0x1FD5,0x1FE9,0x1FFD,0x2011,0x2025,0x2038,0x204C,0x2060,
0x2074,0x2088,0x209C,0x20B0,0x20C4,0x20D8,0x20EC,0x2100,
0x2113,0x2127,0x213B,0x214F,0x2163,0x2177,0x218B,0x219F,
0x21B2,0x21C6,0x21DA,0x21EE,0x2202,0x2216,0x222A,0x223D,
0x2251,0x2265,0x2279,0x228D,0x22A1,0x22B4,0x22C8,0x22DC,
0x22F0,0x2304,0x2317,0x232B,0x233F,0x2353,0x2367,0x237A,
0x238E,0x23A2,0x23B6,0x23C9,0x23DD,0x23F1,0x2405,0x2419,
0x242C,0x2440,0x2454,0x2467,0x247B,0x248F,0x24A3,0x24B6,
0x24CA,0x24DE,0x24F2,0x2505,0x2519,0x252D,0x2540,0x2554,
0x2568,0x257B,0x258F,0x25A3,0x25B6,0x25CA,0x25DE,0x25F2,
0x2605,0x2619,0x262C,0x2640,0x2654,0x2667,0x267B,0x268F,
0x26A2,0x26B6,0x26CA,0x26DD,0x26F1,0x2704,0x2718,0x272C,
0x273F,0x2753,0x2766,0x277A,0x278E,0x27A1,0x27B5,0x27C8,
0x27DC,0x27EF,0x2803,0x2817,0x282A,0x283E,0x2851,0x2865,
0x2878,0x288C,0x289F,0x28B3,0x28C6,0x28DA,0x28ED,0x2901,
0x2914,0x2928,0x293B,0x294F,0x2962,0x2976,0x2989,0x299D,
0x29B0,0x29C4,0x29D7,0x29EB,0x29FE,0x2A11,0x2A25,0x2A38,
0x2A4C,0x2A5F,0x2A73,0x2A86,0x2A99,0x2AAD,0x2AC0,0x2AD4,
0x2AE7,0x2AFB,0x2B0E,0x2B21,0x2B35,0x2B48,0x2B5B,0x2B6F,
0x2B82,0x2B96,0x2BA9,0x2BBC,0x2BD0,0x2BE3,0x2BF6,0x2C0A,
0x2C1D,0x2C30,0x2C44,0x2C57,0x2C6A,0x2C7D,0x2C91,0x2CA4,
0x2CB7,0x2CCB,0x2CDE,0x2CF1,0x2D05,0x2D18,0x2D2B,0x2D3E,
0x2D52,0x2D65,0x2D78,0x2D8B,0x2D9F,0x2DB2,0x2DC5,0x2DD8,
0x2DEB,0x2DFF,0x2E12,0x2E25,0x2E38,0x2E4B,0x2E5F,0x2E72,
0x2E85,0x2E98,0x2EAB,0x2EBF,0x2ED2,0x2EE5,0x2EF8,0x2F0B,
0x2F1E,0x2F31,0x2F45,0x2F58,0x2F6B,0x2F7E,0x2F91,0x2FA4,
0x2FB7,0x2FCA,0x2FDE,0x2FF1,0x3004,0x3017,0x302A,0x303D,
0x3050,0x3063,0x3076,0x3089,0x309C,0x30AF,0x30C2,0x30D5,
0x30E8,0x30FB,0x310E,0x3121,0x3134,0x3147,0x315A,0x316D,
0x3180,0x3193,0x31A6,0x31B9,0x31CC,0x31DF,0x31F2,0x3205,
0x3218,0x322B,0x323E,0x3251,0x3264,0x3277,0x328A,0x329D,
0x32B0,0x32C2,0x32D5,0x32E8,0x32FB,0x330E,0x3321,0x3334,
0x3347,0x3359,0x336C,0x337F,0x3392,0x33A5,0x33B8,0x33CB,
0x33DD,0x33F0,0x3403,0x3416,0x3429,0x343B,0x344E,0x3461,
0x3474,0x3487,0x3499,0x34AC,0x34BF,0x34D2,0x34E4,0x34F7,
0x350A,0x351D,0x352F,0x3542,0x3555,0x3567,0x357A,0x358D,
0x35A0,0x35B2,0x35C5,0x35D8,0x35EA,0x35FD,0x3610,0x3622,
0x3635,0x3648,0x365A,0x366D,0x3680,0x3692,0x36A5,0x36B7,
0x36CA,0x36DD,0x36EF,0x3702,0x3714,0x3727,0x373A,0x374C,
0x375F,0x3771,0x3784,0x3796,0x37A9,0x37BB,0x37CE,0x37E0,
0x37F3,0x3805,0x3818,0x382A,0x383D,0x384F,0x3862,0x3874,
0x3887,0x3899,0x38AC,0x38BE,0x38D1,0x38E3,0x38F6,0x3908,
0x391B,0x392D,0x393F,0x3952,0x3964,0x3977,0x3989,0x399B,
0x39AE,0x39C0,0x39D2,0x39E5,0x39F7,0x3A0A,0x3A1C,0x3A2E,
0x3A41,0x3A53,0x3A65,0x3A78,0x3A8A,0x3A9C,0x3AAE,0x3AC1,
0x3AD3,0x3AE5,0x3AF8,0x3B0A,0x3B1C,0x3B2E,0x3B41,0x3B53,
0x3B65,0x3B77,0x3B8A,0x3B9C,0x3BAE,0x3BC0,0x3BD2,0x3BE5,
0x3BF7,0x3C09,0x3C1B,0x3C2D,0x3C40,0x3C52,0x3C64,0x3C76,
0x3C88,0x3C9A,0x3CAC,0x3CBF,0x3CD1,0x3CE3,0x3CF5,0x3D07,
0x3D19,0x3D2B,0x3D3D,0x3D4F,0x3D61,0x3D73,0x3D86,0x3D98,
0x3DAA,0x3DBC,0x3DCE,0x3DE0,0x3DF2,0x3E04,0x3E16,0x3E28,
0x3E3A,0x3E4C,0x3E5E,0x3E70,0x3E82,0x3E94,0x3EA6,0x3EB8,
0x3EC9,0x3EDB,0x3EED,0x3EFF,0x3F11,0x3F23,0x3F35,0x3F47,
0x3F59,0x3F6B,0x3F7D,0x3F8E,0x3FA0,0x3FB2,0x3FC4,0x3FD6,
0x3FE8,0x3FFA,0x400B,0x401D,0x402F,0x4041,0x4053,0x4064,
0x4076,0x4088,0x409A,0x40AC,0x40BD,0x40CF,0x40E1,0x40F3,
0x4104,0x4116,0x4128,0x4139,0x414B,0x415D,0x416F,0x4180,
0x4192,0x41A4,0x41B5,0x41C7,0x41D9,0x41EA,0x41FC,0x420E,
0x421F,0x4231,0x4242,0x4254,0x4266,0x4277,0x4289,0x429A,
0x42AC,0x42BE,0x42CF,0x42E1,0x42F2,0x4304,0x4315,0x4327,
0x4338,0x434A,0x435B,0x436D,0x437E,0x4390,0x43A1,0x43B3,
0x43C4,0x43D6,0x43E7,0x43F9,0x440A,0x441C,0x442D,0x443E,
0x4450,0x4461,0x4473,0x4484,0x4495,0x44A7,0x44B8,0x44CA,
0x44DB,0x44EC,0x44FE,0x450F,0x4520,0x4532,0x4543,0x4554,
0x4566,0x4577,0x4588,0x4599,0x45AB,0x45BC,0x45CD,0x45DE,
0x45F0,0x4601,0x4612,0x4623,0x4635,0x4646,0x4657,0x4668,
0x4679,0x468B,0x469C,0x46AD,0x46BE,0x46CF,0x46E0,0x46F2,
0x4703,0x4714,0x4725,0x4736,0x4747,0x4758,0x4769,0x477A,
0x478B,0x479D,0x47AE,0x47BF,0x47D0,0x47E1,0x47F2,0x4803,
0x4814,0x4825,0x4836,0x4847,0x4858,0x4869,0x487A,0x488B,
0x489C,0x48AD,0x48BE,0x48CF,0x48DF,0x48F0,0x4901,0x4912,
0x4923,0x4934,0x4945,0x4956,0x4967,0x4977,0x4988,0x4999,
0x49AA,0x49BB,0x49CC,0x49DD,0x49ED,0x49FE,0x4A0F,0x4A20,
0x4A30,0x4A41,0x4A52,0x4A63,0x4A74,0x4A84,0x4A95,0x4AA6,
0x4AB6,0x4AC7,0x4AD8,0x4AE9,0x4AF9,0x4B0A,0x4B1B,0x4B2B,
0x4B3C,0x4B4D,0x4B5D,0x4B6E,0x4B7F,0x4B8F,0x4BA0,0x4BB0,
0x4BC1,0x4BD2,0x4BE2,0x4BF3,0x4C03,0x4C14,0x4C24,0x4C35,
0x4C45,0x4C56,0x4C67,0x4C77,0x4C88,0x4C98,0x4CA9,0x4CB9,
0x4CCA,0x4CDA,0x4CEA,0x4CFB,0x4D0B,0x4D1C,0x4D2C,0x4D3D,
0x4D4D,0x4D5D,0x4D6E,0x4D7E,0x4D8F,0x4D9F,0x4DAF,0x4DC0,
0x4DD0,0x4DE0,0x4DF1,0x4E01,0x4E11,0x4E22,0x4E32,0x4E42,
0x4E53,0x4E63,0x4E73,0x4E83,0x4E94,0x4EA4,0x4EB4,0x4EC4,
0x4ED5,0x4EE5,0x4EF5,0x4F05,0x4F15,0x4F26,0x4F36,0x4F46,
0x4F56,0x4F66,0x4F76,0x4F87,0x4F97,0x4FA7,0x4FB7,0x4FC7,
0x4FD7,0x4FE7,0x4FF7,0x5007,0x5017,0x5027,0x5037,0x5048,
0x5058,0x5068,0x5078,0x5088,0x5098,0x50A8,0x50B8,0x50C8,
0x50D8,0x50E7,0x50F7,0x5107,0x5117,0x5127,0x5137,0x5147,
0x5157,0x5167,0x5177,0x5187,0x5196,0x51A6,0x51B6,0x51C6,
0x51D6,0x51E6,0x51F6,0x5205,0x5215,0x5225,0x5235,0x5244,
0x5254,0x5264,0x5274,0x5283,0x5293,0x52A3,0x52B3,0x52C2,
0x52D2,0x52E2,0x52F1,0x5301,0x5311,0x5320,0x5330,0x5340,
0x534F,0x535F,0x536F,0x537E,0x538E,0x539D,0x53AD,0x53BD,
0x53CC,0x53DC,0x53EB,0x53FB,0x540A,0x541A,0x5429,0x5439,
0x5448,0x5458,0x5467,0x5477,0x5486,0x5496,0x54A5,0x54B5,
0x54C4,0x54D3,0x54E3,0x54F2,0x5502,0x5511,0x5520,0x5530,
0x553F,0x554F,0x555E,0x556D,0x557D,0x558C,0x559B,0x55AB,
0x55BA,0x55C9,0x55D8,0x55E8,0x55F7,0x5606,0x5615,0x5625,
0x5634,0x5643,0x5652,0x5661,0x5671,0x5680,0x568F,0x569E,
0x56AD,0x56BC,0x56CC,0x56DB,0x56EA,0x56F9,0x5708,0x5717,
0x5726,0x5735,0x5744,0x5753,0x5762,0x5771,0x5781,0x5790,
0x579F,0x57AE,0x57BD,0x57CC,0x57DB,0x57E9,0x57F8,0x5807,
0x5816,0x5825,0x5834,0x5843,0x5852,0x5861,0x5870,0x587F,
0x588E,0x589C,0x58AB,0x58BA,0x58C9,0x58D8,0x58E7,0x58F5,
0x5904,0x5913,0x5922,0x5931,0x593F,0x594E,0x595D,0x596C,
0x597A,0x5989,0x5998,0x59A6,0x59B5,0x59C4,0x59D2,0x59E1,
0x59F0,0x59FE,0x5A0D,0x5A1C,0x5A2A,0x5A39,0x5A47,0x5A56,
0x5A65,0x5A73,0x5A82,0x5A90,0x5A9F,0x5AAD,0x5ABC,0x5ACA,
0x5AD9,0x5AE7,0x5AF6,0x5B04,0x5B13,0x5B21,0x5B30,0x5B3E,
0x5B4D,0x5B5B,0x5B6A,0x5B78,0x5B86,0x5B95,0x5BA3,0x5BB1,
0x5BC0,0x5BCE,0x5BDD,0x5BEB,0x5BF9,0x5C07,0x5C16,0x5C24,
0x5C32,0x5C41,0x5C4F,0x5C5D,0x5C6B,0x5C7A,0x5C88,0x5C96,
0x5CA4,0x5CB3,0x5CC1,0x5CCF,0x5CDD,0x5CEB,0x5CF9,0x5D08,
0x5D16,0x5D24,0x5D32,0x5D40,0x5D4E,0x5D5C,0x5D6A,0x5D78,
0x5D86,0x5D94,0x5DA3,0x5DB1,0x5DBF,0x5DCD,0x5DDB,0x5DE9,
0x5DF7,0x5E05,0x5E12,0x5E20,0x5E2E,0x5E3C,0x5E4A,0x5E58,
0x5E66,0x5E74,0x5E82,0x5E90,0x5E9E,0x5EAB,0x5EB9,0x5EC7,
0x5ED5,0x5EE3,0x5EF1,0x5EFE,0x5F0C,0x5F1A,0x5F28,0x5F36,
0x5F43,0x5F51,0x5F5F,0x5F6D,0x5F7A,0x5F88,0x5F96,0x5FA3,
0x5FB1,0x5FBF,0x5FCC,0x5FDA,0x5FE8,0x5FF5,0x6003,0x6010,
0x601E,0x602C,0x6039,0x6047,0x6054,0x6062,0x606F,0x607D,
0x608B,0x6098,0x60A6,0x60B3,0x60C1,0x60CE,0x60DB,0x60E9,
0x60F6,0x6104,0x6111,0x611F,0x612C,0x6139,0x6147,0x6154,
0x6162,0x616F,0x617C,0x618A,0x6197,0x61A4,0x61B2,0x61BF,
0x61CC,0x61D9,0x61E7,0x61F4,0x6201,0x620E,0x621C,0x6229,
0x6236,0x6243,0x6250,0x625E,0x626B,0x6278,0x6285,0x6292,
0x629F,0x62AD,0x62BA,0x62C7,0x62D4,0x62E1,0x62EE,0x62FB,
0x6308,0x6315,0x6322,0x632F,0x633C,0x6349,0x6356,0x6363,
0x6370,0x637D,0x638A,0x6397,0x63A4,0x63B1,0x63BE,0x63CB,
0x63D7,0x63E4,0x63F1,0x63FE,0x640B,0x6418,0x6425,0x6431,
0x643E,0x644B,0x6458,0x6465,0x6471,0x647E,0x648B,0x6498,
0x64A4,0x64B1,0x64BE,0x64CA,0x64D7,0x64E4,0x64F0,0x64FD,
0x650A,0x6516,0x6523,0x6530,0x653C,0x6549,0x6555,0x6562,
0x656F,0x657B,0x6588,0x6594,0x65A1,0x65AD,0x65BA,0x65C6,
0x65D3,0x65DF,0x65EC,0x65F8,0x6604,0x6611,0x661D,0x662A,
0x6636,0x6643,0x664F,0x665B,0x6668,0x6674,0x6680,0x668D,
0x6699,0x66A5,0x66B2,0x66BE,0x66CA,0x66D6,0x66E3,0x66EF,
0x66FB,0x6707,0x6714,0x6720,0x672C,0x6738,0x6744,0x6750,
0x675D,0x6769,0x6775,0x6781,0x678D,0x6799,0x67A5,0x67B1,
0x67BD,0x67C9,0x67D5,0x67E2,0x67EE,0x67FA,0x6806,0x6812,
0x681E,0x6829,0x6835,0x6841,0x684D,0x6859,0x6865,0x6871,
0x687D,0x6889,0x6895,0x68A1,0x68AC,0x68B8,0x68C4,0x68D0,
0x68DC,0x68E8,0x68F3,0x68FF,0x690B,0x6917,0x6922,0x692E,
0x693A,0x6946,0x6951,0x695D,0x6969,0x6974,0x6980,0x698C,
0x6997,0x69A3,0x69AF,0x69BA,0x69C6,0x69D1,0x69DD,0x69E9,
0x69F4,0x6A00,0x6A0B,0x6A17,0x6A22,0x6A2E,0x6A39,0x6A45,
0x6A50,0x6A5C,0x6A67,0x6A72,0x6A7E,0x6A89,0x6A95,0x6AA0,
0x6AAC,0x6AB7,0x6AC2,0x6ACE,0x6AD9,0x6AE4,0x6AF0,0x6AFB,
0x6B06,0x6B11,0x6B1D,0x6B28,0x6B33,0x6B3F,0x6B4A,0x6B55,
0x6B60,0x6B6B,0x6B77,0x6B82,0x6B8D,0x6B98,0x6BA3,0x6BAE,
0x6BB9,0x6BC5,0x6BD0,0x6BDB,0x6BE6,0x6BF1,0x6BFC,0x6C07,
0x6C12,0x6C1D,0x6C28,0x6C33,0x6C3E,0x6C49,0x6C54,0x6C5F,
0x6C6A,0x6C75,0x6C80,0x6C8B,0x6C96,0x6CA1,0x6CAB,0x6CB6,
0x6CC1,0x6CCC,0x6CD7,0x6CE2,0x6CED,0x6CF7,0x6D02,0x6D0D,
0x6D18,0x6D22,0x6D2D,0x6D38,0x6D43,0x6D4D,0x6D58,0x6D63,
0x6D6D,0x6D78,0x6D83,0x6D8D,0x6D98,0x6DA3,0x6DAD,0x6DB8,
0x6DC3,0x6DCD,0x6DD8,0x6DE2,0x6DED,0x6DF7,0x6E02,0x6E0C,
0x6E17,0x6E21,0x6E2C,0x6E36,0x6E41,0x6E4B,0x6E56,0x6E60,
0x6E6B,0x6E75,0x6E7F,0x6E8A,0x6E94,0x6E9F,0x6EA9,0x6EB3,
0x6EBE,0x6EC8,0x6ED2,0x6EDC,0x6EE7,0x6EF1,0x6EFB,0x6F06,
0x6F10,0x6F1A,0x6F24,0x6F2E,0x6F39,0x6F43,0x6F4D,0x6F57,
0x6F61,0x6F6B,0x6F76,0x6F80,0x6F8A,0x6F94,0x6F9E,0x6FA8,
0x6FB2,0x6FBC,0x6FC6,0x6FD0,0x6FDA,0x6FE4,0x6FEE,0x6FF8,
0x7002,0x700C,0x7016,0x7020,0x702A,0x7034,0x703E,0x7048,
0x7052,0x705B,0x7065,0x706F,0x7079,0x7083,0x708D,0x7096,
0x70A0,0x70AA,0x70B4,0x70BD,0x70C7,0x70D1,0x70DB,0x70E4,
0x70EE,0x70F8,0x7101,0x710B,0x7115,0x711E,0x7128,0x7132,
0x713B,0x7145,0x714E,0x7158,0x7162,0x716B,0x7175,0x717E,
0x7188,0x7191,0x719B,0x71A4,0x71AE,0x71B7,0x71C0,0x71CA,
0x71D3,0x71DD,0x71E6,0x71F0,0x71F9,0x7202,0x720C,0x7215,
0x721E,0x7228,0x7231,0x723A,0x7243,0x724D,0x7256,0x725F,
0x7269,0x7272,0x727B,0x7284,0x728D,0x7297,0x72A0,0x72A9,
0x72B2,0x72BB,0x72C4,0x72CD,0x72D6,0x72E0,0x72E9,0x72F2,
0x72FB,0x7304,0x730D,0x7316,0x731F,0x7328,0x7331,0x733A,
0x7343,0x734C,0x7355,0x735E,0x7366,0x736F,0x7378,0x7381,
0x738A,0x7393,0x739C,0x73A4,0x73AD,0x73B6,0x73BF,0x73C8,
0x73D0,0x73D9,0x73E2,0x73EB,0x73F3,0x73FC,0x7405,0x740E,
0x7416,0x741F,0x7428,0x7430,0x7439,0x7441,0x744A,0x7453,
0x745B,0x7464,0x746C,0x7475,0x747D,0x7486,0x748E,0x7497,
0x749F,0x74A8,0x74B0,0x74B9,0x74C1,0x74CA,0x74D2,0x74DB,
0x74E3,0x74EB,0x74F4,0x74FC,0x7504,0x750D,0x7515,0x751D,
0x7526,0x752E,0x7536,0x753F,0x7547,0x754F,0x7557,0x755F,
0x7568,0x7570,0x7578,0x7580,0x7588,0x7591,0x7599,0x75A1,
0x75A9,0x75B1,0x75B9,0x75C1,0x75C9,0x75D1,0x75D9,0x75E1,
0x75E9,0x75F1,0x75F9,0x7601,0x7609,0x7611,0x7619,0x7621,
0x7629,0x7631,0x7639,0x7641,0x7649,0x7650,0x7658,0x7660,
0x7668,0x7670,0x7678,0x767F,0x7687,0x768F,0x7697,0x769E,
0x76A6,0x76AE,0x76B6,0x76BD,0x76C5,0x76CD,0x76D4,0x76DC,
0x76E4,0x76EB,0x76F3,0x76FA,0x7702,0x770A,0x7711,0x7719,
0x7720,0x7728,0x772F,0x7737,0x773E,0x7746,0x774D,0x7755,
0x775C,0x7763,0x776B,0x7772,0x777A,0x7781,0x7788,0x7790,
0x7797,0x779E,0x77A6,0x77AD,0x77B4,0x77BC,0x77C3,0x77CA,
0x77D1,0x77D9,0x77E0,0x77E7,0x77EE,0x77F6,0x77FD,0x7804,
0x780B,0x7812,0x7819,0x7820,0x7827,0x782F,0x7836,0x783D,
0x7844,0x784B,0x7852,0x7859,0x7860,0x7867,0x786E,0x7875,
0x787C,0x7883,0x788A,0x7891,0x7897,0x789E,0x78A5,0x78AC,
0x78B3,0x78BA,0x78C1,0x78C7,0x78CE,0x78D5,0x78DC,0x78E3,
0x78E9,0x78F0,0x78F7,0x78FE,0x7904,0x790B,0x7912,0x7918,
0x791F,0x7926,0x792C,0x7933,0x793A,0x7940,0x7947,0x794D,
0x7954,0x795A,0x7961,0x7968,0x796E,0x7975,0x797B,0x7982,
0x7988,0x798E,0x7995,0x799B,0x79A2,0x79A8,0x79AF,0x79B5,
0x79BB,0x79C2,0x79C8,0x79CE,0x79D5,0x79DB,0x79E1,0x79E8,
0x79EE,0x79F4,0x79FA,0x7A01,0x7A07,0x7A0D,0x7A13,0x7A19,
0x7A1F,0x7A26,0x7A2C,0x7A32,0x7A38,0x7A3E,0x7A44,0x7A4A,
0x7A50,0x7A56,0x7A5D,0x7A63,0x7A69,0x7A6F,0x7A75,0x7A7B,
0x7A81,0x7A87,0x7A8C,0x7A92,0x7A98,0x7A9E,0x7AA4,0x7AAA,
0x7AB0,0x7AB6,0x7ABC,0x7AC1,0x7AC7,0x7ACD,0x7AD3,0x7AD9,
0x7ADE,0x7AE4,0x7AEA,0x7AF0,0x7AF5,0x7AFB,0x7B01,0x7B07,
0x7B0C,0x7B12,0x7B17,0x7B1D,0x7B23,0x7B28,0x7B2E,0x7B34,
0x7B39,0x7B3F,0x7B44,0x7B4A,0x7B4F,0x7B55,0x7B5A,0x7B60,
0x7B65,0x7B6B,0x7B70,0x7B76,0x7B7B,0x7B80,0x7B86,0x7B8B,
0x7B91,0x7B96,0x7B9B,0x7BA1,0x7BA6,0x7BAB,0x7BB1,0x7BB6,
0x7BBB,0x7BC0,0x7BC6,0x7BCB,0x7BD0,0x7BD5,0x7BDB,0x7BE0,
0x7BE5,0x7BEA,0x7BEF,0x7BF4,0x7BFA,0x7BFF,0x7C04,0x7C09,
0x7C0E,0x7C13,0x7C18,0x7C1D,0x7C22,0x7C27,0x7C2C,0x7C31,
0x7C36,0x7C3B,0x7C40,0x7C45,0x7C4A,0x7C4F,0x7C54,0x7C59,
0x7C5D,0x7C62,0x7C67,0x7C6C,0x7C71,0x7C76,0x7C7A,0x7C7F,
0x7C84,0x7C89,0x7C8E,0x7C92,0x7C97,0x7C9C,0x7CA0,0x7CA5,
0x7CAA,0x7CAE,0x7CB3,0x7CB8,0x7CBC,0x7CC1,0x7CC6,0x7CCA,
0x7CCF,0x7CD3,0x7CD8,0x7CDC,0x7CE1,0x7CE5,0x7CEA,0x7CEE,
0x7CF3,0x7CF7,0x7CFC,0x7D00,0x7D05,0x7D09,0x7D0D,0x7D12,
0x7D16,0x7D1B,0x7D1F,0x7D23,0x7D28,0x7D2C,0x7D30,0x7D34,
0x7D39,0x7D3D,0x7D41,0x7D45,0x7D4A,0x7D4E,0x7D52,0x7D56,
0x7D5A,0x7D5F,0x7D63,0x7D67,0x7D6B,0x7D6F,0x7D73,0x7D77,
0x7D7B,0x7D7F,0x7D83,0x7D87,0x7D8B,0x7D8F,0x7D93,0x7D97,
0x7D9B,0x7D9F,0x7DA3,0x7DA7,0x7DAB,0x7DAF,0x7DB3,0x7DB7,
0x7DBB,0x7DBE,0x7DC2,0x7DC6,0x7DCA,0x7DCE,0x7DD2,0x7DD5,
0x7DD9,0x7DDD,0x7DE1,0x7DE4,0x7DE8,0x7DEC,0x7DEF,0x7DF3,
0x7DF7,0x7DFA,0x7DFE,0x7E02,0x7E05,0x7E09,0x7E0C,0x7E10,
0x7E14,0x7E17,0x7E1B,0x7E1E,0x7E22,0x7E25,0x7E29,0x7E2C,
0x7E30,0x7E33,0x7E36,0x7E3A,0x7E3D,0x7E41,0x7E44,0x7E47,
0x7E4B,0x7E4E,0x7E51,0x7E55,0x7E58,0x7E5B,0x7E5F,0x7E62,
0x7E65,0x7E68,0x7E6C,0x7E6F,0x7E72,0x7E75,0x7E78,0x7E7C,
0x7E7F,0x7E82,0x7E85,0x7E88,0x7E8B,0x7E8E,0x7E91,0x7E94,
0x7E97,0x7E9A,0x7E9D,0x7EA0,0x7EA3,0x7EA6,0x7EA9,0x7EAC,
0x7EAF,0x7EB2,0x7EB5,0x7EB8,0x7EBB,0x7EBE,0x7EC1,0x7EC4,
0x7EC6,0x7EC9,0x7ECC,0x7ECF,0x7ED2,0x7ED4,0x7ED7,0x7EDA,
0x7EDD,0x7EDF,0x7EE2,0x7EE5,0x7EE8,0x7EEA,0x7EED,0x7EF0,
0x7EF2,0x7EF5,0x7EF7,0x7EFA,0x7EFD,0x7EFF,0x7F02,0x7F04,
0x7F07,0x7F09,0x7F0C,0x7F0E,0x7F11,0x7F13,0x7F16,0x7F18,
0x7F1B,0x7F1D,0x7F1F,0x7F22,0x7F24,0x7F27,0x7F29,0x7F2B,
0x7F2E,0x7F30,0x7F32,0x7F35,0x7F37,0x7F39,0x7F3B,0x7F3E,
0x7F40,0x7F42,0x7F44,0x7F46,0x7F49,0x7F4B,0x7F4D,0x7F4F,
0x7F51,0x7F53,0x7F55,0x7F57,0x7F5A,0x7F5C,0x7F5E,0x7F60,
0x7F62,0x7F64,0x7F66,0x7F68,0x7F6A,0x7F6C,0x7F6E,0x7F70,
0x7F71,0x7F73,0x7F75,0x7F77,0x7F79,0x7F7B,0x7F7D,0x7F7E,
0x7F80,0x7F82,0x7F84,0x7F86,0x7F87,0x7F89,0x7F8B,0x7F8D,
0x7F8E,0x7F90,0x7F92,0x7F93,0x7F95,0x7F97,0x7F98,0x7F9A,
0x7F9C,0x7F9D,0x7F9F,0x7FA0,0x7FA2,0x7FA3,0x7FA5,0x7FA7,
0x7FA8,0x7FA9,0x7FAB,0x7FAC,0x7FAE,0x7FAF,0x7FB1,0x7FB2,
0x7FB4,0x7FB5,0x7FB6,0x7FB8,0x7FB9,0x7FBA,0x7FBC,0x7FBD,
0x7FBE,0x7FC0,0x7FC1,0x7FC2,0x7FC3,0x7FC5,0x7FC6,0x7FC7,
0x7FC8,0x7FC9,0x7FCB,0x7FCC,0x7FCD,0x7FCE,0x7FCF,0x7FD0,
0x7FD1,0x7FD2,0x7FD4,0x7FD5,0x7FD6,0x7FD7,0x7FD8,0x7FD9,
0x7FDA,0x7FDB,0x7FDC,0x7FDD,0x7FDD,0x7FDE,0x7FDF,0x7FE0,
0x7FE1,0x7FE2,0x7FE3,0x7FE4,0x7FE5,0x7FE5,0x7FE6,0x7FE7,
0x7FE8,0x7FE8,0x7FE9,0x7FEA,0x7FEB,0x7FEB,0x7FEC,0x7FED,
0x7FEE,0x7FEE,0x7FEF,0x7FEF,0x7FF0,0x7FF1,0x7FF1,0x7FF2,
0x7FF2,0x7FF3,0x7FF4,0x7FF4,0x7FF5,0x7FF5,0x7FF6,0x7FF6,
0x7FF7,0x7FF7,0x7FF8,0x7FF8,0x7FF8,0x7FF9,0x7FF9,0x7FFA,
0x7FFA,0x7FFA,0x7FFB,0x7FFB,0x7FFB,0x7FFC,0x7FFC,0x7FFC,
0x7FFC,0x7FFD,0x7FFD,0x7FFD,0x7FFD,0x7FFE,0x7FFE,0x7FFE,
0x7FFE,0x7FFE,0x7FFE,0x7FFE,0x7FFF,0x7FFF,0x7FFF,0x7FFF,
0x7FFF,0x7FFF,0x7FFF,0x7FFF,0x7FFF,} ;
#endif
