/** mbed wave.h general wave form class for function generator
 * Copyright (c) 2014, 2015 Motoo Tanaka @ Design Methodology Lab
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef _WAVE_H_
#define _WAVE_H_

/** wave class basic class of wave form generator
 *
 */
class wave {
public:

/** constructor
 *
 * @param float v amplitude in volt 0.0 ~ 3.28 for FRDM-KL25Z
 * @param int f Frequency in Hz (1~10000Hz)
 * @param int duty Duty in percent (0 ~ 100%)
 * @param int p Phase in degree (0 ~ 359 degree)
 */
wave(float v, int f, int duty = 50, int p = 0) ; // volt, freq, duty, phase

/** destructor */
~wave(void) ;

/** mutator for _amp but in float voltage
 * 
 * @param float newvalue vlotage to assign (0.0 ~ 3.28V for FRDM-KL25Z)
 */
void volt(float newvalue) ; // set voltage

/** inspector for _amp but in float voltage
 *
 * @return float amplitude in vlot (0.0 ~ 3.28V for FRDM-KL25Z)
 */
float volt(void) ;          // get voltage

/** mutator for _amp in int value 
 *
 * @param int newvalue value for amplitude (0x0000 ~ 0xFFFF)
 */
void amp(int newvalue) ;   // set amp

/** inspector for _amp 
 *
 * @returns amplitude in unsigned short (0x0000 ~ 0xFFFF)
 */
int  amp(void) ;           // get amp

/** mutator for _cycle
 *
 * @param int newvalue length of 1 cycle of the wave 
 */
void cycle(int newvalue) ; // set cycle

/** inspector for _cycle
 *
 * @returns int length of 1 cycle in sample time unit
 */
int  cycle(void) ;         // get cycle

/** mutator for _phase
 *
 * @param int newvalue value for _phase in degree (0 ~ 359 degree)
 */
void phase(int newvalue) ; // set phase

/** inspector for _phase
 *
 * @return int phase in degree (0 ~ 359 degree)
 */
int  phase(void) ;         // get phase

/** mutator for _cycle but in frequency (Hz)
 *
 * @param int newvalue frequcney (1 ~ 10000Hz)
 */
void freq(int newvalue) ; // set freq

/** inspector for _cycle but in frequency (Hz)
 *
 * @returns int current frequency in Hz (1~10000Hz)
 */
int freq(void) ;         // get freq

/** mutator for _pos current positon in _cycle
 *
 * @param int newvalue (0 ~ _cycle-1)
 */
void pos(int newvalue) ;   // set position

/** insepector for _pos
 *
 * @returns int _pos current position in the _cycle
 */
int  pos(void) ;           // get position

/** mutator for _duty the percentage of positive value part
 *
 * @param int newvalue new duty value in percent (0~100%)
 */
virtual void duty(int newvalue) ; // set duty

/** inspector for _duty 
 *
 * @returns int duty (0~100%)
 */
virtual int duty(void) ;           // get duty

/** update internal value consistancy
 */
virtual void update(void) ;        // adjust for time unit change

/** advance position to be accessed through value() 
 *
 * @param int s steps to advance, 1 for default
 */
virtual void advance(int s = 1) ;     // advance 1 tu

/** inspector for the value at _pos
 *
 * @returns the value at _pos
 */
virtual int  value(void) ;         // returns current value

/** inspector to return class name
 */
virtual const char *name(void) ; 

protected:
/** utility function to convert phase to position
 *
 * @param p phase value to be converted to the position
 * @returns position of specified phase
 */
inline int phase2pos(int p) ;

/** utility function to convert position to phase
 *
 * @param p position to be converted to the phase
 * @returns phase of specified position
 */
inline int pos2phase(int p) ;

    int _amp ; /* 0~0xFFFF (12bit for Kinetis) */
    int _cycle ; /* 2~20000 10kHz ~ 1Hz */
    int _phase ; /* 0-360 degree */
    int _freq ; /* 1/cycle 50kHz ~ 1Hz */
    int _pos  ; /* position in cycle */
    int _duty ; /* 0-100 % */
    char *_name ;
private:
} ;

extern int sample_time ; // sampling interval in us
extern float vref ; // voltage at 0xFFFF

#endif