/** mbed wave.cpp general wave form class for function generator
 * Copyright (c) 2014, 2015 Motoo Tanaka @ Design Methodology Lab
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
/**
 * wave.cpp
 * root class of wave form classes
 * because I assumed Kinetis KL25Z
 * for the device, time unit is 10us
 * there for frequencey vs cycle is
 * freq  = 1000000 / (cycle * 10us)
 */
#include "wave.h"

int sample_time = 25 ; // 25us for single wave
float vref = 3.28 ; // full range voltage on my FRDM-KL25Z

wave::wave(float v, int f, int d, int p)
{
    volt( v ) ;
    freq( f ) ;
    duty( d ) ;
    phase( p ) ;
    _name = "wave" ;
}

wave::~wave(void) { }

int wave::phase2pos(int ph)
{
    int po = 0;
    if (_cycle != 0) {
        po = _cycle * ph / 360 ;
    }
    return( po ) ;
}

int wave::pos2phase(int po)
{
    int ph = 0 ;
    if (_cycle != 0) {
        ph = 360 * po / _cycle ;
    }
    return( ph ) ;
}

void wave::update(void)
{
    int tmp ;
    tmp = freq() ;
    freq(tmp) ;
}

void wave::volt(float v)
{
    _amp = 0xFFFF * v / vref ;
}

float wave::volt(void)
{
    return( _amp * vref / 0xFFFF ) ;
}

void wave::amp(int newvalue)
{
    _amp = newvalue ;
}

int wave::amp(void)
{
    return( _amp ) ;
}

void wave::cycle(int newvalue)
{
    _cycle = newvalue ;
    if (_cycle != 0) {
        _freq = 10000.0 / _cycle ;
    } else { // shall we take this as DC?
        _freq = 0.0 ;
    }
}

int wave::cycle(void)
{
    return( _cycle ) ;
}

void wave::phase(int newvalue)
{
    if (_cycle != 0) {
        _phase = newvalue ;
        _pos = phase2pos( _phase ) ;
    } else {
        _phase = 0 ;
        _pos = 0 ;
    }
}

int wave::phase(void)
{
    return( _phase ) ;
}

void wave::freq(int newvalue)
{
    _freq = newvalue ;
    if (_freq != 0.0) {
        _cycle =  1000000.0/(_freq * sample_time) ;
    } else {
        _cycle = 0 ;
    }
}

int wave::freq(void)
{
    return( _freq ) ;
}

void wave::pos(int p)
{
    if (_cycle != 0) {
        _pos = p % _cycle ;
        _phase = pos2phase( _pos ) ;
    }
}

int wave::pos(void)
{
    return( _pos ) ;
}

void wave::duty(int newvalue)
{
    _duty = newvalue ;
}

int wave::duty(void)
{
    return( _duty ) ;
}

void wave::advance(int s)
{
    if (_cycle != 0) {
        _pos = (_pos + s) % _cycle ;
        _phase = pos2phase( _pos ) ;
    }
}

/**
 * value should return the value at phase
 * but in this case just returns _amp
 */
int wave::value(void)
{
    return( _amp ) ;
}

const char *wave::name(void)
{
    return( _name ) ;
}
