/** mbed funcgen my implementation of a function generator
 * Copyright (c) 2014, 2015 Motoo Tanaka @ Design Methodology Lab
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#include "mbed.h"
#include "math.h"
#include "vt100.h"

#if defined (TARGET_KL25Z) 
#define PIN_SCLK        PTD1
#define PIN_MISO        PTD3 
#define PIN_MOSI        PTD2
#define PIN_CS_TFT      PTD0 
#define PIN_DC_TFT      PTD5 
#define PIN_CS_TSC      PTA13
#define PIN_BL_TFT      PTC9 
#define PIN_CS_SD       PTA4 
#define PIN_TSC_INTR    PTC6 /* place holder */
#define PIN_RESET_TFT   PTC5 /* place holder */
#define PIN_ADC_CH0     PTB0
#define PIN_ADC_CH1     PTB2
#define PIN_DAC_OUT     PTE30

#elif defined (TARGET_KL46Z)
#define PIN_SCLK        PTD5 
#define PIN_MISO        PTD7 
#define PIN_MOSI        PTD6
#define PIN_CS_TFT      PTD4 
#define PIN_DC_TFT      PTD2 
#define PIN_CS_TSC      PTA13
#define PIN_BL_TFT      PTC9 
#define PIN_CS_SD       PTA4 
#define PIN_TSC_INTR    PTC7 /* place holder */
#define PIN_RESET_TFT   PTC6 /* place holder */
#define PIN_ADC_CH0     PTB0
#define PIN_ADC_CH1     PTB2
#define PIN_DAC_OUT     PTE30

#elif defined (TARGET_K64F)
#define PIN_SCLK        PTD1
#define PIN_MISO        PTD3
#define PIN_MOSI        PTD2
#define PIN_CS_TFT      PTD0
#define PIN_DC_TFT      PTC4
// for board rev E or later
#define PIN_CS_TSC      PTC12
// for earlier boards use following line
// #define PIN_CS_TSC      PTA0
#define PIN_BL_TFT      PTC3
#define PIN_CS_SD       PTB23
#define PIN_TSC_INTR    PTC0 /* place holder */
#define PIN_RESET_TFT   PTC9 /* place holder */
#define PIN_ADC_CH0     PTB2
#define PIN_ADC_CH1     PTB10
#define PIN_DAC_OUT     DAC0_OUT

#elif defined (TARGET_K22F)
#define PIN_SCLK        PTD5
#define PIN_MISO        PTD7
#define PIN_MOSI        PTD6
#define PIN_CS_TFT      PTD4
#define PIN_DC_TFT      PTA1
#define PIN_CS_TSC      PTB19
#define PIN_BL_TFT      PTC6 
#define PIN_CS_SD       PTA4 
#define PIN_TSC_INTR    PTC7 /* place holder */
#define PIN_RESET_TFT   PTC9 /* place holder */
#define PIN_ADC_CH0     PTB0
#define PIN_ADC_CH1     PTC1
#define PIN_DAC_OUT     DAC0_OUT

#elif defined (TARGET_NUCLEO_F411RE)
#define PIN_SCLK        PA_5
#define PIN_MISO        PA_6
#define PIN_MOSI        PA_7
#define PIN_CS_TFT      PB_6
#define PIN_DC_TFT      PC_7
#define PIN_CS_TSC      PA_9
#define PIN_BL_TFT      PA_8
#define PIN_CS_SD       PB_5
#define PIN_TSC_INTR    PA_8 /* place holder */
#define PIN_RESET_TFT   PA_13 /* place holder */
#define PIN_ADC_CH0     PA_0
#define PIN_ADC_CH1     PA_4

#else
 #error TARGET NOT DEFINED
#define PIN_SCLK        D13 
#define PIN_MISO        D12 
#define PIN_MOSI        D11
#define PIN_CS_TFT      D10 
#define PIN_DC_TFT      D9 
#define PIN_CS_TSC      D8
#define PIN_BL_TFT      D7 
#define PIN_CS_SD       D4 
#define PIN_TSC_INTR    NC
#define PIN_RESET_TFT   NC
#define PIN_ADC_CH0     A0
#define PIN_ADC_CH1     A2

#endif

#include "wave.h"
#include "sinwave.h"
#include "sqrwave.h"
#include "sawwave.h"
#include "triwave.h"

#define MODE_PAUSE 0x00
#define MODE_RUN   0x01

#define MAX_WAVE 10



DigitalOut myled(LED1) ;
AnalogOut aout(PIN_DAC_OUT) ; // SignalOut

Ticker *timer ;
wave *wave_list[MAX_WAVE] ;
wave *w = 0 ;
int num_wave = 0 ;
int wave_index = 0 ;
bool ticked = false ;
int vdout = 0 ;
int avalue = 0 ;
int aindex = 0 ;
int mode = MODE_RUN ;
vt100 *tty = 0 ;

/** usage display list of available commands
 */
void usage(void)
{
    printf("=== funcgen (%s) ===\n\r", __DATE__) ;
    printf("run  : start wave generation\n\r") ;
    printf("stop : stop wave generation\n\r") ;
    printf("help : print this help\n\r") ;
    printf("sin <amp> <freq>        add a sine wave\n\r") ;
    printf("sqr <amp> <freq> <duty> add a square wave\n\r") ;
    printf("saw <amp> <freq>        add a saw wave\n\r") ;
    printf("tri <amp> <freq> <duty> add a triangle wave\n\r") ;
    printf("------------------\n\r") ;
    printf("list          (print list of waves)\n\r") ;
    printf("select <num>  (select a target wave)\n\r") ;
    printf("delete        (delete selected wave)\n\r") ;
    printf("amp <amp>     (0 ~ 3.28 V)\n\r") ;
    printf("freq <freq>   (0 ~ 12000 Hz)\n\r") ;
    printf("duty <duty>   (0 ~ 100 %%)\n\r") ;
    printf("sample <time> (25~100 us(?))\n\r") ;
    printf("------------------\n\r") ;
}

/** tick_time advance wave position(s)
 */
void tick_time(void)
{
    int i ;
    int value = 0 ;
    for (i = 0 ; i < num_wave ; i++ ) {
        if (wave_list[i]) {
            value += wave_list[i]->value() ;
            wave_list[i]->advance() ;
        }
    }
    if (value > 0xFFFF) { // clip
        value = 0xFFFF ;
    }
    aout.write_u16(value) ;
}

/** adjust sample time
 * For a single wave 25us seems to be sufficient
 * but adding more wave seems to require additional 
 * 8us for each wave function.
 * These values are only from my experiments
 * so may be there is/are better numbers/solutions.
 */
void adjust_sample_time(int newvalue = 0)
{
    int i ;
    if (timer) {
        timer->detach() ;
    }
    if (newvalue != 0) {
        sample_time = newvalue ;
    } else {
        sample_time = 17 + 8 * num_wave ;
    }
    for (i = 0 ; i < num_wave ; i++ ) {
        if (wave_list[i] != 0) {
            wave_list[i]->update() ;
        }
    }
    if (timer && (mode & MODE_RUN)) {
           timer->attach_us(&tick_time, sample_time) ;
    }   
}

void add_wave(wave *w)
{
    if (num_wave < MAX_WAVE) {
        timer->detach() ;
        wave_index = num_wave ;
        wave_list[wave_index] = w ;
        num_wave++ ;
        adjust_sample_time() ;
    } else {
        printf("Maximum number of waves exceeded\n\r") ;
    }
}

void delete_current_wave(void)
{
    int i ;
    if (wave_list[wave_index] != 0) {
        timer->detach() ;
        delete wave_list[wave_index] ;
        for (i = wave_index ; i < num_wave ;  i++ ) {
            wave_list[i] = wave_list[i+1] ;
        }
        num_wave-- ;
        wave_list[num_wave] = 0 ;
        if (wave_index >= num_wave) {
            wave_index = num_wave - 1 ;
        }
        w = wave_list[wave_index] ;
        adjust_sample_time() ;
    }
}

void list_waves(void)
{
    int i ;
    printf("sample time = %d(us) %d wave(s)\n\r", sample_time, num_wave) ;
    for (i = 0 ; i < num_wave ; i++ ) {
        if (i == wave_index) {
            printf("* ") ;
        } else {
            printf("  ") ;
        }
        printf("%d: ", i) ;
        if (wave_list[i]) {
            w = wave_list[i] ;
            printf("[ %s ] amp = %.2f V, freq = %d Hz, duty = %d %%",
            w->name(), w->volt(), w->freq(), w->duty()) ;
        }
        printf("\n\r") ;
    }
    w = wave_list[wave_index] ;
}
    
/** main funcgen main program
 */
int main() {
    char cmd[32] ;
    int freq, duty ;
    float volt ;
        
    tty = new vt100() ;
    tty->cls() ;
    usage() ;
    
    timer = new Ticker() ;
    w = new sinwave(3.0, 1000, 0) ;
    add_wave(w) ;
    
    timer->attach_us(&tick_time, sample_time) ;
    myled = 1 ; // turn the LED off
    
    while(1) {
        printf("> ") ;
        scanf("%s", cmd) ;
        if (strcmp(cmd, "run") == 0) { // start timer
            mode |= MODE_RUN ;
            timer->attach_us(&tick_time, sample_time) ;
        } else if (strcmp(cmd, "stop") == 0) { // stop timer
            mode ^= MODE_RUN ;
            timer->detach() ;
        } else if (strcmp(cmd, "delete") == 0) { // delete current wave
            delete_current_wave() ;
        } else if (strcmp(cmd, "select") == 0) { // select a wave to be handled
            scanf("%d", &wave_index) ;
            if (wave_index >= num_wave) {
                wave_index = num_wave - 1;
            }
            w = wave_list[wave_index] ;
        } else if (strcmp(cmd, "sample") == 0) { // set time unit
            scanf("%d", &sample_time) ;
            adjust_sample_time(sample_time) ;
        } else if (strcmp(cmd, "sin") == 0) { // new sin wave
            scanf("%f %d", &volt, &freq) ;
            w = new sinwave(volt, freq) ;
            add_wave(w) ;
        } else if (strcmp(cmd, "sqr") == 0) { // new sqr wave
            scanf("%f %d %d", &volt, &freq, &duty) ;
            w = new sqrwave(volt, freq, duty) ;
            add_wave(w) ;
        } else if (strcmp(cmd, "saw") == 0) { // new saw wave
            scanf("%f %d", &volt, &freq) ;
            w = new sawwave(volt, freq) ;
            add_wave(w) ;
        } else if (strcmp(cmd, "tri") == 0) { // new tri wave
            scanf("%f %d %d", &volt, &freq, &duty) ;
            w = new triwave(volt, freq, duty) ;
            add_wave(w) ;
        } else if (strcmp(cmd, "amp") == 0) {
            scanf("%f", &volt) ;
            if (w != 0) {
                w->volt(volt) ;
            }
        } else if (strcmp(cmd, "freq") == 0) {
            scanf("%d", &freq) ;
            if (w != 0) {
                w->freq(freq) ;
            }
        } else if (strcmp(cmd, "duty") == 0) {
            scanf("%d", &duty) ;
            if (w != 0) {
                w->duty(duty) ;
            }
        } else if (strcmp(cmd, "list") == 0) {
            list_waves() ;
        } else if (strcmp(cmd, "help") == 0) {
            usage() ;
        } else {
            usage() ;
        }
    }
}
