#ifndef _VCNL4010_H_
#define _VCNL4010_H_
/**
 * VCNL4010 Fully Integrated Proximity and Ambient Light Sensor with
 * Infrared Emitter, I2C Interface, and Interrupt Function
 */
  
class VCNL4010
{
public:
  /**
  * VCNL4010 constructor
  *
  * @param sda SDA pin
  * @param sdl SCL pin
  * @param addr addr of the I2C peripheral
  */
  VCNL4010(PinName sda, PinName scl, int addr);

  /**
  * VCNL4010 destructor
  */
  ~VCNL4010();

/**
 * Get Command Register
 * @param none
 * @returns uint8_t command register value (Register #0)
 * @note Register #0 Command Register (0x80)
 * @note bit[7] config_lock
 * @note bit[6] als_data_rdy
 * @note bit[5] prox_data_rdy
 * @note bit[4] als_od
 * @note bit[3] prox_od
 * @note bit[2] als_en
 * @note bit[1] prox_en
 * @note bit[0] selftimed_en
 */
uint8_t  getCommandReg(void) ;

/**
 * Set Command Register
 * @param uint8_t value to set
 * @returns none
 */
void     setCommandReg(uint8_t cmd) ;

/**
 * Get Prodcut ID (and Revision)
 * @param none
 * @returns Product ID and Revision Register (Register #1)
 * @note Register #1 Product ID Revision Register (0x81)
 * @note bit[7:4] Product ID
 * @note bit[3:0] Revision ID
 */
uint8_t  getProductID(void) ;

/**
 * Get Rate of Proximity Measurement
 * @param none
 * @returns Rate of Proximity Measurement (Register #2)
 * @note Register #2 Rate of Proximity Measurement (0x82)
 * @note bit[7:3] (n/a)
 * @note bit[2:0] Rate of Proximity Measurement
 * @note 000 -  1.95     measurements/s (DEFAULT)
 * @note 001 -  3.960625 measurements/s
 * @note 010 -  7.8125   measurements/s
 * @note 011 - 16.625    measurements/s
 * @note 100 - 31.25     measurements/s
 * @note 101 - 62.5      measurements/s
 * @note 110 - 125       measurements/s
 * @note 111 - 250       measurements/s
 */
uint8_t  getProxRate(void) ;

/**
 * Set Rate of Proximity Measurement
 * @param Rate of Proximity Measurement
 * @returns none
 */
void     setProxRate(uint8_t rate) ;

/**
 * Get IR LED Current Setting for Proximity Mode
 * @param none
 * @returns IR LED Current Settings (Register #3)
 * @note bit[7:6] Fuse prog ID
 * @note bit[5:0] IR LED current value
 * @note IR LED current = value(dec.) x 10mA
 * @note Valid Range = 0 to 20d.
 * @note 0 = 0mA, 1 = 10mA .. 20 = 200mA ((DEFAULT: 2 = 20mA)
 */ 
uint8_t  getIrLedCurrent(void) ;

/**
 * Set IR LED Current Setting for Proximity Mode
 * @param value to set
 * @returns none
 */
void     setIrLedCurrent(uint8_t IrLedCur) ;

/**
 * Get Ambient Light Parameter 
 * @param none
 * @returns Ambient Light Parameter (Register #4)
 * @note Register #4 Ambient Light Parameter Register (0x84)
 * @note bit[7]   Continuous conversion mode 0: Disable = DEFAULT 1: Enable
 * @note bit[6:4] als_rate
 * @note bit[3]   Auto offset compensation 0: Disable 1: Enable = DEFAULT
 * @note bit[2:0] Averaging function (number of measurements per run)
 * @note als_rate Ambient light measurement rate
 * @note 000 - 1 samples/s
 * @note 001 - 2 samples/s = DEFUAULT
 * @note 010 - 3 samples/s
 * @note 011 - 4 samples/s
 * @note 100 - 5 samples/s
 * @note 101 - 6 samples/s
 * @note 110 - 8 samples/s
 * @note 111 - 10 samples/s
 * @note Averaging function
 * @note Number of conversions = 2^decimal_value
 * @note 0 = 1 conv, 1 = 2 conv, 2 = 4 conv, .. t = 32 conv (DEFAULT) .. 7 = 128 conv
 */ 
uint8_t  getAlsParam(void) ;

/** 
 * Set Ambient Light Parameter
 * @param uint8_t value to set
 * @returns none
 */
void     setAlsParam(uint8_t param) ;

/** 
 * Get Ambient Light Result
 * @param none
 * @returns uint16_t the value of Ambient Light Result (Register #5, #6)
 * @note Register #5 and #6 Ambient Light Result Register (0x85, 0x86)
 * @note bit[15:8] MSB (address 0x85)
 * @note bit[7:0]  LSB (address 0x86)
 */
uint16_t getAls(void) ;

/** 
 * Get Proximity Measurement Result 
 * @param none
 * @returns uint16_t the value of Proximity Measurement Result (Register #7, #8)
 * @note Register #7 and #8 Proximity Measurement Result Register (0x87, 0x88)
 * @note bit[15:8] MSB (address 0x87)
 * @note bit[7:0]  LSB (address 0x88)
 */
uint16_t getProx(void) ;

/**
 * Get Interrupt Control Register value
 * @param none
 * @returns uint8_t the value of Interraupt Control (Register #9)
 * @note Register #9 Interrupt Control Register (0x89)
 * @note bit[7:5] Interrupt Count Exceed
 * @note bit[4] (n/a)
 * @note bit[3] INT_PROX_ready_EN Eneables interrupt generation at proximity data ready
 * @note bit[2] INT_ALS_ready_EN Enables interrupt generation at ambient data ready
 * @note bit[1] INT_THRES_EN Enables interrupt generation when high or low threshold is exceeded
 * @note bit[0] INT_THRES_SEL Thresholds are applied to 0:proximity 1:als measurement
 * @note Interrupt Count Exceed
 * @note 000 -   1 count = DEFAULT
 * @note 001 -   2 count
 * @note 010 -   4 count
 * @note 011 -   8 count
 * @note 100 -  16 count
 * @note 101 -  32 count
 * @note 110 -  64 count 
 * @note 111 - 128 count
 */ 
uint8_t  getIntCtrl(void) ;

/**
 * Set Interrupt Control Register
 * @param uint8_t the value to set
 * @returns none
 */
void     setIntCtrl(uint8_t ctrl) ;

/**
 * Get Low Threshold
 * @param none
 * @returns uint16_t Low Threshold (Register #10, #11)
 * @note Register #10 and #11 Low Threshold (0x8A, 0x8B)
 * @note bit[15:8] MSB (address 0x8A)
 * @note bit[7:0]  LSB (address 0x8B)
 */ 
uint16_t getLowThreshold(void) ;

/**
 * Set Low Threshold
 * @param uint16_t value to set
 * @returns none
 */
void     setLowThreshold(uint16_t thr) ;

/**
 * Get High Threshold
 * @param none
 * @returns uint16_t High Threshold (Register #12, #13)
 * @note Register #12 and #13 High Threshold (0x8C, 0x8D)
 * @note bit[15:8] MSB (address 0x8C)
 * @note bit[7:0]  LSB (address 0x8D)
 */
uint16_t getHighThreshold(void) ;

/**
 * Set High Threshold
 * @param uint16_t value to set
 * @returns none
 */
void     setHighThreshold(uint16_t thr) ;

/**
 * Get Interrupt Status
 * @param none
 * @returns uint8_t Interrupt Status (Register #14)
 * @note Register #14 Interrupt Status (0x8E)
 * @note bit[7:4] (n/a)
 * @note bit[3] int_prox_ready
 * @note bit[2] int_als_ready
 * @note bit[1] int_th_low
 * @note bit[0] int_th_hi
 * @note Once an interrupt is generated the corresponding status bit
 * @note goes to 1 and stays there unless it is cleared by writing a 1
 * @note in the corresponding bit.
 */
uint8_t  getIntStatus(void) ;

/**
 * Set Interrupt Status
 * @param uint8_t value to set
 * @returns none
 */
void     setIntStatus(uint8_t status) ;

/**
 * Get Proximity Modulator Timing Adjustment
 * @param none
 * @returns Proximity Modulator Timing Adjustment (Register #15)
 * @note Register #15 Proximity Modulator Timing Adjustment (0x8F) 
 * @note bit[7:5] Modulation delay time
 * @note bit[4:3] Proximity frequency
 * @note bit[2:0] Modulation deat time
 * @note The settings for best performance will be provided by Vishay.
 * @note With first samples this is evaluated to:
 * @note delay time = 0
 * @note frequency = 0 
 * @note deat time = 1 
 * @note With that Register #15 should be programmed with 1 (= default value).
 */ 
uint8_t  getProxModTiming(void) ;

/**
 * Set Proximity Modulator Timing Adjustment
 * @param uint8_t value to set
 * @returns none
 */
void     setProxModTiming(uint8_t timing) ;

private:
  I2C m_i2c;
  int m_addr;
  void readRegs(int addr, uint8_t * data, int len);
  void writeRegs(uint8_t * data, int len);
}; 

#endif /* _VCNL4010_H_ */