/** maze_vt100 a simple maze using UniGraphic and MMA8451Q libraries
 */
#include "mbed.h"
#include "BMI160.h"
// #include "MMA8451Q.h"
#include <ILI9341.h>
#include "Arial12x12.h"
#include "Arial24x23.h"
#include "Arial28x28.h"
#include "Arial43x48_numb.h"
#include "maze.h"

#define CELL_W  10
#define CELL_H  10
#define CELL_S   5
#define CELL_R   4
#define LEFT_OFFSET 20
#define TOP_OFFSET 20

#define DIR_STAY  0
#define DIR_UP    1
#define DIR_DOWN  2
#define DIR_RIGHT 3 
#define DIR_LEFT  4

#define MMA8451_I2C_ADDRESS (0x1D<<1)

typedef struct _pos {
    int x ;
    int y ;
} pos_type ;

extern BMI160 *acc ;
extern ILI9341 *tft ;
// extern DigitalOut backlight ;
float threshold = 0.2 ;

/** Check if two pos_type values are same
 * @param pos_type a
 * @param pos_type b
 * @returns if a and b are same position
 */
bool isSame(pos_type a, pos_type b)
{
    return((a.x == b.x)&&(a.y == b.y)) ;
}

/** Draw the maze defined in the "maze.h"
 * @param pos_type *current actually the start point
 * @param pos_type *goal the position of the goal
 * @note those params are actually returned by this function
 */
void drawTFTMaze(pos_type *current, pos_type *goal)
{
    int x, y, i, j ;
    tft->BusEnable(true) ;
    
//    backlight = 0 ;
    tft->foreground(White) ;
    tft->background(Black) ;
    tft->cls() ;
    for (j = 0 ; j < MAZE_H ; j++ ) {
        for (i = 0 ; i < MAZE_W ; i++ ) {
            x = i * CELL_W + LEFT_OFFSET ;
            y = j * CELL_H + TOP_OFFSET ;
            switch(maze[j][i]) {
            case 0: // path
                tft->fillrect(x, y, x+CELL_W-1, y+CELL_H-1, Black) ;
                break ;
            case 1: // wall
                tft->fillrect(x, y, x+CELL_W-1, y+CELL_H-1, DarkGrey) ;
                break ;
            case 2: // Start point
                tft->fillcircle(x+CELL_S, y+CELL_S, CELL_R, Green) ;
                current->x = i ;
                current->y = j ;
                break ;
            case 3: // Goal
                tft->fillcircle(x+CELL_S, y+CELL_S, CELL_R, Red) ;
                goal->x = i ;
                goal->y = j ;
                break ;
            default: // should not be here
                break ;
            }
        }
    }
//    backlight = 1 ;
    tft->BusEnable(false) ;
}

/** Filter out too little move 
 * @param float in returned value from the acc
 * @returns float result filtered value of in
 */
float filterVal(float in)
{
    float result = 0.0 ;
    if ((-threshold > in)||(in > threshold)) {
        result = in ;
    }
    return( result ) ;
}

/** Decide which direction to go
 * @param float res[] acc value of x, y 
 * @returns int direction to move 
 */
int getDirection(float res[])
{
    float dx, dy ;
    int direction = DIR_STAY ;
    dy = filterVal(res[0]) ; /* was dx */
    dx = filterVal(res[1]) ; /* was dy */

    if ((dx*dx) > (dy*dy)) { // holizontal move
        if (dx > 0.0) {
            direction = DIR_UP ;
        } else if (dx < 0.0) {
            direction = DIR_DOWN ;
        }
    } else { // vertical move
        if (dy > 0.0) {
            direction = DIR_LEFT ;
        } else if (dy < 0.0) {
            direction = DIR_RIGHT ;
        }
    }
    return(direction) ;
}

/** Get next positon to move to
 * @param pos_type current where we are now
 * @param int direction which way we'd like to move
 * @returns the candidate positon for the next move
 */
pos_type getNext(pos_type current, int direction) 
{
    pos_type next = current ;
    switch(direction) {
    case DIR_STAY: 
        break ;
    case DIR_UP: 
        if (next.y > 0) { 
            next.y-- ; 
        }  
        break ;
    case DIR_DOWN: 
        if (next.y < (MAZE_H - 1)) { 
            next.y++ ; 
        } 
        break ;
    case DIR_RIGHT: 
        if (next.x < (MAZE_W - 1)) { 
            next.x++ ; 
        } 
        break ;
    case DIR_LEFT: 
        if (next.x > 0) { 
            next.x-- ; 
        } 
        break ;
    default: 
        break ;
    }
    return( next ) ;
}

/** Notice of the goal
 */
void showGoal(void)
{
    int x0, y0, x1, y1 ;
    x0 = ((MAZE_W/2)-4) * CELL_W + LEFT_OFFSET ;
    y0 = ((MAZE_H/2)-1) * CELL_H + TOP_OFFSET ;
    x1 = ((MAZE_W/2)+4) * CELL_W + LEFT_OFFSET ;
    y1 = ((MAZE_H/2)+1) * CELL_H + TOP_OFFSET ;

    tft->BusEnable(true) ;
//    tft->set_font((unsigned char*) Arial12x12);
    tft->set_font((unsigned char*) Terminal6x8) ;
    tft->fillrect(x0, y0, x1, y1, Blue) ;
    tft->rect(x0,y0,x1,y1, Yellow) ;
    tft->foreground(Red) ;
    tft->background(Blue) ;
    x0 = ((MAZE_W/2) - 2) * CELL_W + LEFT_OFFSET ;
    y0 = (MAZE_H/2) * CELL_H - 3 + TOP_OFFSET ;
    tft->locate(x0, y0) ;
    tft->printf("G O A L") ;
    tft->foreground(Yellow) ;
    tft->background(Black) ;
    tft->locate(30, 300) ;
//    tft->printf("Use FRDM touch slider") ;
    tft->printf("           Retry!      ") ;
    tft->locate(5, 300) ;
    tft->printf("<< Prev") ;
    tft->locate(180, 300) ;
    tft->printf("Next >>") ;
    tft->BusEnable(false) ;
}

/** Check if we can move to the next position
 * @param pos_type next the position we'd like to move next
 * @returns if the position is empty (movable)
 */
bool checkMove(pos_type next) 
{
    bool result = false ;
    
    switch(maze[next.y][next.x]) {
    case POS_PATH:
    case POS_GOAL:
        result = true ;
        break ;
    case POS_START:
    case POS_WALL:
    default:
        result = false ;
        break ;
    } 
    return( result ) ;
}

/** main a simple maze program
 */
void doMaze(void) 
{
    float res[3] ;
    pos_type current, next, goal ;
    int direction = DIR_STAY ;
    int x, y ;
    
    drawTFTMaze(&current, &goal) ;
    
    for (;;) {
//        acc->getAccAllAxis(res) ;
        acc->getAcc(res) ;
        direction = getDirection(res) ;
        next = getNext(current, direction) ;
        if ((!isSame(current, next)) && checkMove(next)) {
           tft->BusEnable(true) ;
           x = current.x * CELL_W + LEFT_OFFSET ;
           y = current.y * CELL_H + TOP_OFFSET ;
           tft->fillrect(x, y, x+CELL_W-1, y+CELL_H-1, Black) ;
           x = next.x * CELL_W + LEFT_OFFSET ;
           y = next.y * CELL_H + TOP_OFFSET ;
           tft->fillcircle(x+CELL_S, y+CELL_S, CELL_R, Green) ; 
           tft->BusEnable(false) ;
           current = next ;
           if (isSame(next, goal)) {
             break ; // goal in!
           }
        }
        wait(0.1) ;
    }
    showGoal() ;
}
