/**
 * TFT test MAX32630FTHR
 * MCU Board: MAX32630FTHR
 * TFT Module: Adafruit 2.4 TFT with touch
 */
#include "mbed.h"
#include "max32630fthr.h"
#include "USBSerial.h"
#include "ILI9341.h"
#include "SPI_STMPE610.h"
#include "Arial12x12.h"
#include "Arial24x23.h"
#include "Arial28x28.h"
#include "Arial43x48_numb.h"
#include "BMI160.h"

#define PIN_MOSI SPI2_MOSI
#define PIN_MISO SPI2_MISO
#define PIN_SCLK SPI2_SCK
#define PIN_TSC_CS P3_3
#define PIN_CS_TFT P5_3
#define PIN_RESET_TFT P1_7 // dummy 
#define PIN_DC_TFT P5_4
#define PIN_LED_R P2_4
#define PIN_LED_G P2_5
#define PIN_LED_B P2_6
#define PIN_RX2  P2_0
#define PIN_TX2  P2_1
#define PIN_SDA2 P5_7
#define PIN_SCL2 P6_0
#define BMI160_I2C_ADDRESS 0x68

#define USE_TTY 1

MAX32630FTHR *sakura = 0 ;
ILI9341      *tft  = 0 ;
SPI_STMPE610 *tsc  = 0 ;
DigitalOut   *ledR = 0 ;
DigitalOut   *ledG = 0 ;
DigitalOut   *ledB = 0 ;
BMI160       *acc = 0 ;

// Hardware serial port over DAPLink
// Serial daplink(P2_1, P2_0);

// Virtual serial port over USB
USBSerial *tty = 0 ;
// USBSerial microUSB;

int page = 0 ;
int numPage = 4 ;

extern void doMaze(void) ;

void backlight(int value)
{
    uint8_t data[2] ;
    data[1] = 0x04 ; // GPIO-2 as target pin
    if (value) { /* backlight on */
        data[0] = 0x10 ; // GPIO_SET_PIN
    } else {
        data[0] = 0x11 ; // GPIO_CLR_PIN
    }
    tsc->writeRegs(data, 2) ;
}
        
void init_hardware(void)
{
    sakura = new MAX32630FTHR(MAX32630FTHR::VIO_3V3) ;
    Thread::wait(0.1) ;
    sakura->init(MAX32630FTHR::VIO_3V3) ;
#if USE_TTY
    tty = new USBSerial() ;
#endif
    tft = new ILI9341(SPI_8, 16000000, 
    SPI2_MOSI, SPI2_MISO,  SPI2_SCK, 
    PIN_CS_TFT, PIN_RESET_TFT, PIN_DC_TFT, "Adafruit2.4") ;
    tft->BusEnable(false) ;
    wait(0.1) ;
    
    tsc = new SPI_STMPE610(SPI2_MOSI, SPI2_MISO, SPI2_SCK, PIN_TSC_CS) ;
    tsc->spi_format(8, 0) ; 
    tsc->calibrate(3552, 400, 439, 3680) ;
/* enable tsc's gpio-2 for backlight */
    uint8_t data[2] ;
    data[0] = 0x17 ; // GPIO_ALT_FUNC
    data[1] = 0x04 ; // GPIO-2 to be GPIO
    tsc->writeRegs(data, 2) ; 
    data[0] = 0x13 ; // GPIO_DIR
    data[1] = 0x04 ; // GPIO-2 as output
    tsc->writeRegs(data, 2) ;

    ledR = new DigitalOut(PIN_LED_R) ;
    ledG = new DigitalOut(PIN_LED_G) ;
    ledB = new DigitalOut(PIN_LED_B) ;
    acc = new BMI160(PIN_SDA2, PIN_SCL2, BMI160_I2C_ADDRESS) ;
    // acc_set_pmu_mode 0x10 | ACC_PMU_XXX
    acc->setCMD(0x10 | ACC_PMU_NORMAL) ;
    Thread::wait(100) ;
    acc->setCMD(0x14 | GYR_PMU_NORMAL) ;
    Thread::wait(100) ;
    acc->setCMD(0x08 | MAG_PMU_SUSPEND) ;
    Thread::wait(100) ;
    acc->setCMD(0x03) ; /* start_foc */
    Thread::wait(100) ;
}

void initTFT(void)
{
    //Configure the display driver
    tft->BusEnable(true) ;
    tft->FastWindow(true) ;
    tft->background(Black);
    tft->foreground(White);
    Thread::wait(0.01) ;
    tft->cls();
    tft->BusEnable(false) ;
}

void screen1(void) // Welcome Screen
{
    tft->BusEnable(true) ;
//    backlight = 0 ;
    backlight(0) ;
    tft->background(White) ;
    wait(0.1) ;
    tft->cls() ;
    wait(0.1) ;
    
    tft->set_font((unsigned char*) Arial24x23);
    tft->foreground(Red) ;
    tft->locate(80, 40) ;
    tft->printf("MBED") ;
    tft->foreground(Blue);
    tft->locate(60, 80) ;
    tft->printf("2.4\"TFT") ; 
    tft->locate(40, 120) ;
    tft->printf("with touch") ;
    tft->foreground(Black);
    tft->set_font((unsigned char*) Arial12x12);
    tft->foreground(Blue) ;
    tft->locate(30, 180) ;
    tft->printf("This program is running on") ;
    tft->locate(30, 200) ;
    tft->printf("Maxim MAX32630FTHR with") ;
    tft->locate(30, 220) ;
    tft->printf("a program developed in mbed") ;
    tft->foreground(Green) ;
    tft->locate(30, 260) ;
    tft->printf("To advance demo page, touch") ;
    tft->locate(30, 280) ;
    tft->printf("and hold right side of screen") ;
    tft->locate(30, 300) ;
    tft->printf("until the next screen starts") ;
    tft->BusEnable(false) ;
//    backlight = 1 ;
    backlight(1) ;
}


void screen2(void) // Graphics
{
    //Draw some graphics
    int i, x[2], y[2] ;
//    backlight = 0 ;
    backlight(0) ;
    tft->BusEnable(true) ;
    tft->background(Black);
    wait(0.1) ;
    tft->foreground(White);
    wait(0.1) ;
    tft->cls() ;
    wait(0.1) ;
    tft->set_font((unsigned char*) Arial12x12);
    tft->locate(90,0);
    tft->printf("Graphics");
    
    x[0] = 25 ; x[1] = 224 ;
    y[0] = 20 ; y[1] = 219 ;
    for (i = 20 ; i < 220 ; i += 10) {
        tft->line(i+5, y[0], i+5, y[1], Blue) ;
        tft->line(x[0], i, x[1], i, Blue) ;
    }
    tft->line(125, y[0], 125, y[1], Green) ;
    tft->line(x[0], 120, x[1], 120, Green) ;
    tft->rect(x[0],y[0], x[1], y[1], Green) ;
    tft->locate(10, 20) ;
    tft->printf("V") ;
    tft->locate(0, 115) ;
    tft->printf("0.0") ;
    tft->locate(115, 225) ;
    tft->printf("0.0") ;
    tft->locate(215, 225) ;
    tft->printf("T") ;

    double s;
    for (int i = x[0]; i < 225; i++) {
        s = 40 * sin((long double)i / 20);
        tft->pixel(i, 120 + (int)s, White);
    }
    
    tft->fillrect(10, 240, 229, 309, White) ;
    tft->rect(10, 240, 229, 309, Red) ;
    tft->rect(11, 241, 228, 308, Red) ;
    
    tft->background(White) ;
    tft->foreground(Black) ;
    tft->locate(20, 250) ;
    tft->printf("With QVGA resolution") ;
    tft->locate(20, 270) ;
    tft->printf("simple graphics drawing") ;
    tft->locate(20, 290) ;
    tft->printf("capability is provided") ;
    tft->BusEnable(false) ;
//    backlight = 1 ;
    backlight(1) ;
}    

float clip(float src)
{
    float value ;
    value = src ;
    if (value < (float)0.0) {
        value = (float)0.0 ;
    } else if (value > (float)2.0) {
        value = (float)2.0 ;
    }
    return( value ) ;
}

void screen3(void)
{
    const int num_signal = 6 ;
    int t = 0 ;
    int pt = 0 ; // previous t 
    int i, s, x, y ;
    unsigned int data[num_signal] ; // for x, y, z 
    unsigned int prev[num_signal] ;
    unsigned short signalHeight = 19 ; // 39 ;
    unsigned short voffset[num_signal] = { 30, 80, 130, 180, 230, 280 } ; // for x, y, z
    unsigned short color[num_signal] = { Red, Green, Yellow, Red, Green, Yellow } ;
    char *label[num_signal] = { "X", "Y", "Z", "x", "y", "z" } ;
    unsigned short paneX[2] = {20, 235} ;
    unsigned short paneH = 41 ; // 81 ;
    float value[num_signal] ;
    float acc_range ;
    float gyr_range ;
 
//    backlight = 1 ;
    backlight(1) ;
    acc_range = (float)acc->getAccRange() ;
    gyr_range = (float)acc->getGyrRange() ;
    tft->BusEnable(true) ;
    tft->background(Black) ;
    tft->foreground(White) ;
//    tft->cls() ;

    for (s = 0 ; s < num_signal ; s++ ) { 
        tft->fillrect(paneX[0], voffset[s], paneX[1], voffset[s]+paneH, Black) ;
    }
    for (i = 0 ; i < 10 ; i++ ) {
        y = i * 4 ; // i * 8 ;
        for (s = 0 ; s < num_signal ; s++) {
            tft->line(paneX[0], voffset[s] + y, paneX[1], voffset[s] + y, Blue) ;
        }
    }
    for (x = 30 ; x < paneX[1] ; x += 10 ) {
        for (s = 0 ; s < num_signal ; s++) {
            tft->line(x, voffset[s], x, voffset[s]+paneH, Blue) ;
        }
    } 
    for (s = 0 ; s < num_signal ; s++ ) {
        tft->rect(paneX[0], voffset[s], paneX[1], voffset[s]+paneH, White) ;
    }
    tft->set_font((unsigned char*) Arial12x12);

    for (s = 0 ; s < num_signal ; s++ ) {
        tft->locate(5, voffset[s]+15) ;
        tft->printf(label[s]) ;
    }

    tft->locate(40, 10) ;
    tft->printf("MAX32630FTHR / BMI160") ;
    
    acc->getAcc(value) ;
    for (s = 0 ; s < 3 ; s++) {
        prev[s] = voffset[s] + (signalHeight * clip((acc_range + value[s])/(acc_range * (float)1.1))) ;
    }
    acc->getGyr(&value[3]) ;
    for ( ; s < num_signal ; s++ ) {
        prev[s] = voffset[s] + (signalHeight * clip((gyr_range + value[s])/(gyr_range * (float)1.1))) ;
    }
    pt = paneX[0] ;
//    backlight = 1 ;
    for(t = 21 ; t < paneX[1] ; t++) {
        acc->getAcc(value) ;
        data[0] = voffset[0] + (signalHeight * clip((acc_range + value[0])/(acc_range * (float)1.1))) ;
        data[1] = voffset[1] + (signalHeight * clip((acc_range + value[1])/(acc_range * (float)1.1))) ;
        data[2] = voffset[2] + (signalHeight * clip((acc_range + value[2])/(acc_range * (float)1.1))) ;
        
        acc->getGyr(&value[3]) ;
        data[3] = voffset[3] + (signalHeight * clip((gyr_range + value[3])/(gyr_range * (float)1.1))) ;
        data[4] = voffset[4] + (signalHeight * clip((gyr_range + value[4])/(gyr_range * (float)1.1))) ;
        data[5] = voffset[5] + (signalHeight * clip((gyr_range + value[5])/(gyr_range * (float)1.1))) ;

        for (s = 0 ; s < num_signal ; s++ ) {
            tft->line(pt, prev[s], t, data[s], color[s]) ;
        }
        for (s = 0 ; s < num_signal ; s++ ) {
            prev[s] = data[s] ;
        }      
        pt = t ;
        Thread::wait(20) ;
    }
    tft->BusEnable(false) ;
}

void incPage(void)
{
    page++ ;
    if (page >= numPage) {
        page = 0 ;
    }
}

void decPage(void) 
{
    page-- ;
    if (page < 0) {
        page = numPage - 1 ;
    }
}

// main() runs in its own thread in the OS
// (note the calls to Thread::wait below for delays)
int main()
{
    int prevPage = 99 ;
    bool waitTouch = false ;
    uint16_t x, y, z ;
    
    init_hardware() ;
    backlight(0) ;
    initTFT() ;
    backlight(1) ;
    
#if USE_TTY
    tty->printf("MAX32630 test (%s)\n", __DATE__) ;
#endif
//     tft->cls() ;
    for(;;) {
        switch(page) {
        case 0:
            if (prevPage != page) {
                screen1() ;
            }
            waitTouch = true ;
            break ;
        case 1:
            if (prevPage != page) {
                screen2() ; 
            }
            waitTouch = true ;
            break ;
        case 2:
            if (prevPage != page) {
                backlight(0) ;
                tft->BusEnable(true) ;
                tft->background(Black) ;
                tft->foreground(White) ;
                tft->cls() ;
                tft->BusEnable(false) ;
            }
            screen3() ; 
            waitTouch = false ;
            break ;           
        case 3:
            if (prevPage != page) {
                doMaze() ;
                waitTouch = true ;
            }
            break ;
        default:
            page = 0 ; 
            break ;
        }
        prevPage = page ;

        do {
            tsc->getPoint(&x, &y, &z) ;
            if ((x != 0)&&(y != 0)) {
#if USE_TTY
                tty->printf("%d, %d, %d\n", x, y, z) ;
#endif
                if (x < 50) { // left
                    decPage() ;
                } else if (x > 190) { // right
                    incPage() ;
                }
                waitTouch = false ;
            }
        } while(waitTouch != false) ; 
        
        Thread::wait(100) ;
    }
}

