#ifndef _RTC8564_H_
#define _RTC8654_H_
#include "mbed.h"
/**
* EPSON TOYOCOM Real Time Clock Module
* RTC-8564JE/NB (ref: ETM11J-06)
*
* @code

* @endcode
*/
class RTC8564
{
public:
  /**
  * RTC8564 constructor
  *
  * @param sda SDA pin
  * @param sdl SCL pin
  * @param addr addr of the I2C peripheral
  */
  RTC8564(PinName sda, PinName scl, int addr);

  /**
  * RTC8564 destructor
  */
  ~RTC8564();

/** 
 * setReg : set arbitrary register
 * @param addr register address
 * @param data value to assign
 * @returns none
 */
void setReg(int addr, uint8_t data) ;

/**
 * getReg : get arbitrary register value
 * @param addr register address
 * @returns uint8_t register value
 */
uint8_t getReg(int addr) ;

/**
 * setAll : set all registers at once
 * @param data 16 x uint8_t values 
 */
void setAll(uint8_t data[]) ; 

/**
 * getAll : get values of all registers
 * @param data 16 x uint8_t values to receive
 * @returns none
 */
void getAll(uint8_t data[]) ; 

/**
 * setControl1 : assign value to Control 1 register (0x00) 
 * @param data uint8_t value to assign
 * @note bit[5] is STOP_BIT 1: stop 0: start 
 * @note all other bits should be 0
 */
void setControl1(uint8_t data) ;

/**
 * getControl1 : get Control 1 register value
 * @param none
 * @returns uint8_t value of Control 1 register
 */
uint8_t getControl1(void) ;

/**
 * setControl2 : assign value to Control 2 register (0x01)
 * @param data uint8_t value to assign
 * @note bit[0]: TIE, bit[1]: AIE, bit[2]: TF, bit[3]: AF
 * @note bit[4]: TI/TP
 */
void setControl2(uint8_t data) ;

/**
 * getControl2 : get value of Control 2 register (0x01)
 * @param none
 * @returns uint8_t value of Control 1 register
 */
uint8_t getControl2(void) ;

/**
 * set seconds 
 * @param data uint8_t value to assign (0..59)
 */
void setSeconds(uint8_t data) ;

/**
 * get seconds
 * @param none
 * @returns uint8_t seconds (0..59)
 */
uint8_t getSeconds(void) ;

/**
 * set minutes
 * @param data uint8_t value to assign (0..59)
 */
void setMinutes(uint8_t data) ;

/**
 * get minutes
 * @param none
 * @returns uint8_t minutes (0..59)
 */
uint8_t getMinutes(void) ;

/**
 * set hours
 * @param data uint8_t value to assign (0..23)
 */
void setHours(uint8_t data) ;

/**
 * get hours
 * @param none
 * @returns uint8_t hours (0..23)
 */
uint8_t getHours(void) ;

/**
 * set days
 * @param data uint8_t value to assing (0..31)
 */
void setDays(uint8_t data) ;

/**
 * get days
 * @param none
 * @returns uint8_t days (0..31)
 */
uint8_t getDays(void) ;

/**
 * set day of week
 * @param data 
 * @note 0: Sunday, 1: Monday, 2: Tuesday,
 * @note 3: Wednesday, 4: Thursday, 5: Friday, 6: Saturday
 */
void setWeekdays(uint8_t data) ;

/**
 * get day of week
 * @param none
 * @returns uint8_t day of week
 */
uint8_t getWeekdays(void) ;

/**
 * set months
 * @param data value to assign (1..12)
 */
void setMonths(uint8_t data) ;

/**
 * get months
 * @param none
 * @returns uint8_t monts (1..12)
 */
uint8_t getMonths(void) ;

/**
 * set years
 * @param data lower 2 digits of the year.
 * @note Currently 20xx is assumed.
 */
void setYears(uint8_t data) ;

/**
 * get Years
 * @param none
 * @returns uint8_t years
 */
uint8_t getYears(void) ;

/**
 * set Minute Alarm
 * @param uint8_t data AE_BIT(0x80) is enable,
 * @note rest of 7bit is 3bit and 4bit BCD value (00..59)
 */
void setMinuteAlarm(uint8_t data) ;

/**
 * get Minute Alarm
 * @param none
 * @returns uint8_t Minute Alarm
 */
uint8_t getMinuteAlarm(void) ;

/**
 * set Hour Alarm
 * @param uint8_t data AE_BIT(0x80) is enable
 * @note rest of 6bit is 2bit and 4bit BCD value (00..23)
 */
void setHourAlarm(uint8_t data) ;

/**
 * get Hour Alarm
 * @param none
 * @returns Hour Alarm
 */
uint8_t getHourAlarm(void) ;

/**
 * set Day Alarm
 * @param uint8_t data AE_BIT(0x80) is enable
 * @note rest of 6bit is 2bit and 4bit BCD value (00..31)
 */
void setDayAlarm(uint8_t data) ;

/**
 * get Day Alarm
 * @param none
 * @returns uint8_t Day Alarm
 */
uint8_t getDayAlarm(void) ;

/**
 * set day of week alarm
 * @param uint8_t data AE_BIT(0x80) is enable
 * @note rest of 3bit is BCD value (0..6)
 */
void setWeekdayAlarm(uint8_t data) ;

/**
 * get day of week alarm
 * @param none
 * @returns day of week alarm
 */
uint8_t getWeekdayAlarm(void) ;

/**
 * set CLKOUT Frequency 
 * @param uint8_t data FE_BIT(0x80) is enable
 * @note bit[1] FD1, bit[0] FD0 to select FREQ.
 * @note CLKOUT outputs only when CLKOE pin is "H"
 * @note and FE_BIT is 1.
 * @note FD1=0, FD0=0 : 32768 Hz,
 * @note FD1=0, FD0=1 : 1024 Hz,
 * @note FD1=1, FD0=0 : 32 Hz
 * @note FD1=1, FD0=1 : 1 Hz
 * @note When STOP_BIT is high, only 32768 Hz will be output
 * @note and other freq clock will not be generated.
 */
void setCLKOUTFrequency(uint8_t data) ;

/**
 * get CLKOUT Frequency
 * @param none
 * @returns uint8_t CLKOUT Frequency
 * @note please refer to setCLKOUTFrequency comment
 * @note and Datasheet for details
 */
uint8_t getCLKOUTFrequency(void) ;

/**
 * set Timer Control
 * @param uint8_t data TE_BIT (0x80) is enable
 * @note bit[1] TD1, bit[0] TD0
 * @note please refer to the Datasheet for the details
 */
void setTimerControl(uint8_t data) ;

/**
 * get Timer Control
 * @pram none
 * @returns uint8_t Timer Control register value
 */
uint8_t getTimerControl(void) ;

/**
 * set Timer
 * @param uint8_t data preset value for the count down timer
 * @note please refer to the Datasheet for the details
 */
void setTimer(uint8_t data) ;

/**
 * get Timer
 * @param none
 * @returns uint8_t timer value
 */
uint8_t getTimer(void) ;

/**
 * stop RTC
 * @note assert the STOP_BIT of the control1 register
 */
void stop(void) ;

/**
 * start RTC
 * @note clear the STOP_BIT of the control1 register
 */
void start(void) ;

private:
  I2C m_i2c;
  int m_addr;
  void readRegs(int addr, uint8_t * data, int len);
  void writeRegs(uint8_t * data, int len);

};


#endif /* _RTC8564_H_ */