#include "mbed.h"
#include "RTC8564.h"

/* Register Definitions */
#define REG_CONTROL1         0x00
#define REG_CONTROL2         0x01
#define REG_SECONDS          0x02
#define REG_MINUTES          0x03
#define REG_HOURS            0x04
#define REG_DAYS             0x05
#define REG_WEEKDAYS         0x06
#define REG_MONTHS           0x07
#define REG_YEARS            0x08
#define REG_MINUTE_ALARM     0x09
#define REG_HOUR_ALARM       0x0A
#define REG_DAY_ALARM        0x0B
#define REG_WEEKDAY_ALARM    0x0C
#define REG_CLKOUT_FREQUENCY 0x0D
#define REG_TIMER_CONTROL    0x0E
#define REG_TIMER            0x0F

/* Register bit masks */
/* REG_CONTROL1 (0x00) */
#define STOP_BIT             0x20

/* REG_CONTROL2 (0x01) */
#define TI_TP_BIT            0x10
#define AF_BIT               0x08
#define TF_BIT               0x04
#define AIE_BIT              0x02
#define TIE_BIT              0x01

/* REG_SECONDS (0x02) */
#define VL_BIT               0x80
#define SECONDS_MASK         0x7F

/* REG_MINUTES (0x03) */
#define MINUTES_MASK         0x7F

/* REG_HOURS (0x04) */
#define HOURS_MASK           0x3F

/* REG_DAYS (0x05) */
#define DAYS_MASK            0x3F

/* REG_WEEKDAYS (0x06) */
#define WEEKDAYS_MASK        0x07

/* REG_MONTHS (0x07) */
#define CENTURY_BIT          0x80
#define MONTHS_MASK          0x1F

/* REG_YEARS (0x08) */
#define YEARS_MASK           0xFF

/* REG_MINUTE_ALARM (0x09)  */
#define AE_BIT               0x80
#define MINUTE_ALARM_MASK    0x7F

/* REG_HOUR_ALARM (0x0A)   */
// #define AE_BIT            0x80 (already defined)
#define HOUR_ALARM_MASK     0x3F

/* REG_DAY_ALARM (0x0B) */
// #define AE_BIT            0x80 (already defined)
#define DAY_ALARM_MASK      0x3F

/* REG_WEEKDAY_ALARM (0x0C) */
// #define AE_BIT            0x80 (already defined)
#define WEEKDAY_ALARM_MASK  0x07

/* REG_CLKOUT_FREQUENCY (0x0D) */
#define FE_BIT              0x80
#define FD1_BIT             0x02
#define FD0_BIT             0x01

/* REG_TIMER_CONTROL (0x0E) */
#define TE_BIT              0x80
#define TD1_BIT             0x02
#define TD0_BIT             0x01

/* REG_TIMER (0x0F) */
/* bit values, 128, 64, 32, 16, 8, 4, 2, 1 (so what?) */

static int VL = 0 ; /* POR or LVD bit */
static int Century = 0 ; /* dawn of the new Century */

static uint8_t i2bcd(uint8_t ival) 
{
    uint8_t bcd = 0 ;
    bcd = ((ival / 10)<<4) | (ival % 10) ;
    return(bcd) ;    
}

static uint8_t bcd2i(uint8_t bcd) 
{
    uint8_t ival ;
    ival = ((bcd >> 4)*10)+(bcd & 0x0F) ;
    return(ival) ;
}

RTC8564::RTC8564(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr<<1) {
    // activate the peripheral
    wait(1) ; /* 1 sec required for the clock to be stable */
}

RTC8564::~RTC8564() { }

void RTC8564::readRegs(int addr, uint8_t * data, int len) {
    char t[1] = {addr} ;
    m_i2c.write(m_addr, t, 1, true) ;
    m_i2c.read(m_addr, (char*)data, len) ;
    wait_us(100) ;
}

void RTC8564::writeRegs(uint8_t * data, int len) {
   m_i2c.write(m_addr, (char *)data, len) ;
   wait_us(100) ;
}

/* mutators */
void RTC8564::setReg(int addr, uint8_t value) 
{
    uint8_t data[2] ;
    data[0] = addr ;
    data[1] = value ;
    writeRegs(data, 2) ;
}

void RTC8564::setAll(uint8_t value[])
{
    uint8_t *data ;
    data = new uint8_t[17] ;
    data[0] = 0x00 ;
    for (int i = 0 ; i < 0x10 ; i++ ) {
        data[i+1] = value[i] ;
    }
    writeRegs(data, 17) ;
    delete data ;
}

void RTC8564::setControl1(uint8_t data) 
{ 
    setReg(REG_CONTROL1, data) ; 
}

void RTC8564::setControl2(uint8_t data) 
{ 
    setReg(REG_CONTROL2, data) ; 
}

void RTC8564::setSeconds(uint8_t value) 
{
    uint8_t bcd ;
    value %= 60 ; /* sec is 0..59 */
    bcd = i2bcd(value) ;
    setReg(REG_SECONDS, bcd) ;
}

void RTC8564::setMinutes(uint8_t data) 
{
    uint8_t bcd ;
    data %= 60 ; /* minute is 0 .. 59 */
    bcd = i2bcd(data) ;
    setReg(REG_MINUTES, bcd) ;
}

void RTC8564::setHours(uint8_t data) 
{
    uint8_t bcd ;
    data %= 24 ; /* hour is 0 .. 23 */
    bcd = i2bcd(data) ;
    setReg(REG_HOURS, bcd) ;
}

void RTC8564::setDays(uint8_t data) 
{
    uint8_t bcd ;
    if (data == 0) { data = 1 ; } 
    data %= 32 ; /* day is 1 .. 31 */
    bcd = i2bcd(data) ;
    setReg(REG_DAYS, bcd) ;
}

void RTC8564::setWeekdays(uint8_t data) 
{
    data %= 7 ; /* weekday is 0 to 7 */
    setReg(REG_WEEKDAYS, data) ;   
}

void RTC8564::setMonths(uint8_t data) 
{
    uint8_t bcd ;
    data = ((data - 1)%12) + 1 ; /* month is 1 .. 12 */
    bcd = i2bcd(data) ;
    setReg(REG_MONTHS, bcd) ;
}

void RTC8564::setYears(uint8_t data) 
{
    uint8_t bcd ;
    data %= 100 ; /* year is 0 .. 99 */
    bcd = i2bcd(data) ;
    setReg(REG_YEARS, bcd) ;
}

void RTC8564::setMinuteAlarm(uint8_t data) 
{
//    data &= (AE_BIT | MINUTE_ALARM_MASK ) ; /* all 8bit is valid */
    setReg(REG_MINUTE_ALARM, data) ;
}

void RTC8564::setHourAlarm(uint8_t data) 
{
    data &= (AE_BIT | HOUR_ALARM_MASK ) ;
    setReg(REG_HOUR_ALARM, data) ;
}

void RTC8564::setDayAlarm(uint8_t data) 
{
    data &= (AE_BIT | DAY_ALARM_MASK) ;
    setReg(REG_DAY_ALARM, data) ;
}

void RTC8564::setWeekdayAlarm(uint8_t data) 
{
    data &= (AE_BIT | WEEKDAY_ALARM_MASK) ;
    setReg(REG_WEEKDAY_ALARM, data) ;
}

void RTC8564::setCLKOUTFrequency(uint8_t data) 
{
    data &= (FE_BIT | FD1_BIT | FD0_BIT) ;
    setReg(REG_CLKOUT_FREQUENCY, data) ;
}

void RTC8564::setTimerControl(uint8_t data) 
{
    data &= (TE_BIT | TD1_BIT | TD0_BIT ) ;
    setReg(REG_TIMER_CONTROL, data) ;
}

void RTC8564::setTimer(uint8_t data) 
{
    setReg(REG_TIMER, data) ;
}

/* inspectors */
uint8_t RTC8564::getReg(int addr) 
{
    uint8_t data ;
    readRegs(addr, &data, 1) ;
    return( data ) ;
}

void    RTC8564::getAll(uint8_t data[]) 
{
    readRegs(0x00, data, 16) ;
}

uint8_t RTC8564::getControl1(void) 
{
    uint8_t data ;
    data = getReg(REG_CONTROL1) ;
    return( data ) ;
}

uint8_t RTC8564::getControl2(void) 
{
    uint8_t data ;
    data = getReg(REG_CONTROL2) ;
    return( data ) ;
}

uint8_t RTC8564::getSeconds(void) 
{
    uint8_t bcd, ival ;
    bcd = getReg(REG_SECONDS) ;
    if (bcd & VL_BIT) {
        VL = 1 ;
    } else {
        VL = 0 ;
    }
    bcd &= SECONDS_MASK ;
    ival = bcd2i(bcd) ;
    return( ival ) ;
}

uint8_t RTC8564::getMinutes(void) 
{
    uint8_t bcd, ival ;
    bcd = getReg(REG_MINUTES) ;
    bcd &= MINUTES_MASK ;
    ival = bcd2i(bcd) ;
    return( ival ) ;
}

uint8_t RTC8564::getHours(void) 
{
    uint8_t bcd, ival ;
    bcd = getReg(REG_HOURS) ;
    bcd &= HOURS_MASK ;
    ival = bcd2i(bcd) ;
    return( ival ) ;       
}

uint8_t RTC8564::getDays(void) 
{
    uint8_t bcd, ival ;
    bcd = getReg(REG_DAYS) ;
    bcd &= DAYS_MASK ;
    ival = bcd2i(bcd) ;
    return( ival ) ;     
}

uint8_t RTC8564::getWeekdays(void) 
{
    uint8_t bcd, ival ;
    bcd = getReg(REG_WEEKDAYS) ;
    bcd &= WEEKDAYS_MASK ;
    ival = bcd2i(bcd) ;
    return( ival ) ;
}

uint8_t RTC8564::getMonths(void) 
{
    uint8_t bcd, ival ;
    bcd = getReg(REG_MONTHS) ;
    if (bcd & CENTURY_BIT) {
        Century = 1 ; /* We are now in the year 2100! (so what?) */
    }
    bcd &= WEEKDAYS_MASK ; /* Sorry, I don't care about year 2100 */
    ival = bcd2i(bcd) ;
    return( ival ) ;
}

uint8_t RTC8564::getYears(void) 
{
    uint8_t bcd, ival ;
    bcd = getReg(REG_YEARS) ;
    ival = bcd2i(bcd) ;
    return( ival ) ;
}

uint8_t RTC8564::getMinuteAlarm(void) 
{
    return( getReg(REG_MINUTE_ALARM) ) ;
}

uint8_t RTC8564::getHourAlarm(void) 
{
    uint8_t data ;
    data = getReg(REG_HOUR_ALARM) ;
    data &= (AE_BIT | HOUR_ALARM_MASK) ;
    return( data ) ;
}

uint8_t RTC8564::getDayAlarm(void) 
{
    uint8_t data ;
    data = getReg(REG_DAY_ALARM) ;
    data &= (AE_BIT | DAY_ALARM_MASK) ;
    return( data ) ;
}

uint8_t RTC8564::getWeekdayAlarm(void) 
{
    uint8_t data ;
    data = getReg(REG_WEEKDAY_ALARM) ;
    data &= (AE_BIT | WEEKDAY_ALARM_MASK) ;
    return( data ) ;
}

uint8_t RTC8564::getCLKOUTFrequency(void) 
{
    uint8_t data ;
    data = getReg(REG_CLKOUT_FREQUENCY) ;
    data &= (FE_BIT | FD1_BIT | FD0_BIT) ;
    return( data ) ;
}

uint8_t RTC8564::getTimerControl(void) 
{
    uint8_t data ;
    data = getReg(REG_TIMER_CONTROL) ;
    data &= (TE_BIT | TD1_BIT | TD0_BIT) ;
    return( data ) ;
}

uint8_t RTC8564::getTimer(void) 
{
    return( getReg(REG_TIMER) ) ;
}

void RTC8564::start(void)
{
    setReg(REG_CONTROL1, 0x00) ;
}

void RTC8564::stop(void)
{
    setReg(REG_CONTROL1, 0x01) ;
}