/**
 * KL25Z UART EDGE 
 * Temp Sensor test for FRDM-KL25Z and SigFox
 *****/
 
#include "mbed.h"

#define PIN_AN0     PTB0
#define PIN_AN1     PTB1
#define PIN_AN2     PTB2
#define PIN_AN3     PTB3 

#define PIN_LEDR    PTB18
#define PIN_LEDG    PTB19
#define PIN_LEDB    PTD1

#define UART_TX     PTE0
#define UART_RX     PTE1
#define BAUD_RATE   9600

#define MODE_IDLE   0
#define MODE_ALART  1
#define MODE_DATA   2
#define ACK         (0x3E)
// #define ACK (0x06)
#define TILDE       '~'
#define CR          '\r'
#define DOT         '.'
#define SPACE       ' '

// R0=5.0, float R1=4.95, float B=3324, float T0=298.15
#define R0          5.0
#define R1          4.95
#define B           3324
#define T0          298.15

DigitalOut *led_r = 0 ;
DigitalOut *led_g = 0 ;
DigitalOut *led_b = 0 ;

AnalogIn *an[4] = { 0, 0, 0, 0 } ;
#define NUM_ANIN 4

Serial *tty = 0 ;

int mode = MODE_IDLE ;

void init_hardware(void)
{
    tty = new Serial(UART_TX, UART_RX, BAUD_RATE) ;
    led_r = new DigitalOut(PIN_LEDR, 1) ;
    led_g = new DigitalOut(PIN_LEDG, 1) ;
    led_b = new DigitalOut(PIN_LEDB, 1) ;
    an[0] = new AnalogIn(PIN_AN0) ;
    an[1] = new AnalogIn(PIN_AN1) ;
    an[2] = new AnalogIn(PIN_AN2) ;
    an[3] = new AnalogIn(PIN_AN3) ;
}

void knock(void)
{
    uint8_t c = CR ;
    printf("Knocking on the host's door\n") ;
    printf("Sending 0x%02X\n", c) ;
    tty->putc(c) ;
    wait(0.5) ;
}

int wait_ack(void)
{
    char c = 0xFF ;
    printf("Waiting for ACK ") ;
    for (int i = 0 ; i < 3 ; i++ ) {
        if (tty->readable()) {
            c = tty->getc() ;
        } else {
            printf(".") ;
        }
        wait(1) ;
    }
    printf(" ") ;
    if (c == ACK) {
        printf("ACK received\n") ;
    } else {
        printf("Unexpected char [0x%02X] received\n", c) ;
    }
    return( c ) ;
}

float getTemp(int n)
{
    float result = 0.0 ;
    float f, raw, rr1, t ;
    if (an[n]) {
        f = an[n]->read() ;
 // printf("AN[%d] = %.2f\n",n, f) ;
        raw = f * 3.3 ;
        rr1 = R1 * raw / (3.3 - raw) ;
        t = 1.0 / (log(rr1 / R0) / B + (1/T0)) ;
        result = t - 273.15 ;
    }
    return( result ) ;
}

void send_data(void)
{
    int i, dummy ;
    float temp[4] ;
    int8_t data[10] ;
    data[0] = TILDE ;
    for (i = 0 ; i < 4 ; i++ ) {
        temp[i] = getTemp(i) ; 
        printf("temp[%d] = %.2f\n", i, temp[i]) ;
        dummy = (int)(100.0 * temp[i] + 0.5)  ;
        data[i * 2 + 1] = (int8_t)(dummy / 100) ;
        data[i * 2 + 2] = (int8_t)(dummy % 100) ;
    }
    data[9] = 0 ;
    printf("Sending data ...") ;
    for (i = 0 ; i < 10 ; i++ ) { /* data must be 10bytes including the header */
        tty->putc(data[i]) ;
        wait(0.1) ;
    }
    printf("Done\n") ;
}

void doLED(void)
{
    switch(mode) {
    case MODE_IDLE:  *led_r = 1 ; *led_g = 1 ; *led_b = 0 ; break ;
    case MODE_ALART: *led_r = 0 ; *led_g = 1 ; *led_b = 1 ; break ;
    case MODE_DATA:  *led_r = 1 ; *led_g = 0 ; *led_b = 1 ; break ;
    default:         *led_r = 1 ; *led_g = 1 ; *led_b = 1 ; break ;
    }
}

int main() {
    init_hardware() ;
    printf("Edge Started\n") ;
    while(1) {
        doLED() ;
        switch(mode) {
        case MODE_IDLE:        
            knock() ;
            mode = MODE_ALART ;
            break ;
        case MODE_ALART:
            if (wait_ack() == ACK) {
                mode = MODE_DATA ;
            } else {
                mode = MODE_IDLE ;
            }
            break ;
        case MODE_DATA:
            send_data() ;
            wait_ack() ;            
            mode = MODE_IDLE ;
            printf("Sleeping ...") ;
            fflush(stdout) ;
            wait(20.0) ; /* minimum 15 sec */
            printf("Done\n") ;
            break ;
        default:
            printf("Unexpected mode [%d]\n", mode) ;
            mode = MODE_IDLE ;
            break ;
        }
        wait(0.1) ;
    }
}
