#ifndef GAMEENGINE_H
#define GAMEENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Enemy.h"
#include "Enemy21.h"
#include "Enemy22.h"
#include "EnemyBoss.h"
#include "EnemyBeam.h"
#include "EnemyBeam2.h"
#include "EnemyBeam3.h"
#include "Beam.h"
#include "Spacecraft.h"

// gap from edge of screen
#define GAP 2

/** GameEngine Class
 * @brief  The game engine for the mission game
 * @author Rex Roshan Raj
 */
class GameEngine
{

public:

    /** Constructor */
    GameEngine();
    
    /** Destructor */
    ~GameEngine();

     /** Initialise the parameters for the game engine
    *@param size - size of the beam
    *@param spacecraft_xpos - x position of the player's spacecraft
    *@param spacecraft_ypos - y position of the player's spacecraft
    *@param enemy_xpos - x position of the enemy in stage one
    *@param enemy_ypos - y position of the enemy in stage one
    *@param enemy2_xpos - x position of the enemy in stage two
    *@param enemy2_ypos - y position of the enemy in stage two
    *@param enemy3_xpos - x position of the boss in stage three
    *@param enemy3_ypos - y position of the boss in stage three
    *@param beam_size - size of the beam in stage one and three
    *@param beam2_size - size of the beam in stage two
    *@param speed - speed of the objects moving
    */
    void init(int spacecraft_xpos,int spacecraft_ypos,int enemy_xpos, int enemy_ypos,int enemy2_xpos, int enemy2_ypos,int enemy3_xpos, int enemy3_ypos, int beam_size, int beam2_size, int speed);
    
    /** Read the input 
    * @param Gamepad pad
    * @brief Reads the input from the Gamepad
    */
    void read_input(Gamepad &pad);
    
    /** Updates mission one
    * @param Gamepad pad
    * @param N5110 lcd
    */
    void update_mission_one(Gamepad &pad,N5110 &lcd);
    
    
    /** Updates mission two
    * @param Gamepad pad
    * @param N5110 lcd
    */
    void update_mission_two(Gamepad &pad,N5110 &lcd);
    
    
    /** Updates mission three
    * @param Gamepad pad
    * @param N5110 lcd
    */
    void update_mission_three(Gamepad &pad,N5110 &lcd);
    
    /** Draws objects in mission one
    * @param N5110 lcd
    * @brief Draws the characters and the beams involved in mission one
    */
    void draw_mission_one(Gamepad &pad,N5110 &lcd);
    
    /** Draws objects in mission two
    * @param N5110 lcd
    * @brief Draws the characters and the beams involved in mission two
    */
    void draw_mission_two(Gamepad &pad,N5110 &lcd);
    
    /** Draws objects in mission three
    * @param N5110 lcd
    * @brief Draws the characters and the beams involved in mission three
    */
    void draw_mission_three(Gamepad &pad,N5110 &lcd);
    
    /** Gets the stage of the game
    * @returns a value from 1 to 6
    */
    int get_game_stage();
    
    /** Sets the stage of the game
    * @brief Sets the stage of the game to be zero
    * @returns a value of 0
    */
    int restart_game_stage();
   
    
private:

    void check_enemy_collisions(Gamepad &pad);  // check if the player spacecraft beam collided with the enemy in mission one
    void check_enemy2_collisions(Gamepad &pad); // check if the player spacecraft beam collided with the enemy in mission two
    void check_enemy3_collisions(Gamepad &pad); // check if the player spacecraft beam collided with the enemy in mission three
    
    void check_wall_collision(Gamepad &pad);   // check if the enemy in mission one has collided with the wall 
    void check_wall_collisions(Gamepad &pad);  // check if the enemy in mission two has collided with the wall
    void check_wall2_collisions(Gamepad &pad); // check if the enemy in mission three has collided with the wall
    
    void enemy_dead(N5110 &lcd,Gamepad &pad);  // check if enemy spacecraft is dead
    void enemy2_dead(N5110 &lcd,Gamepad &pad);  // check if enemy spacecraft is dead
    void enemy3_dead(N5110 &lcd,Gamepad &pad);  // check if enemy spacecraft is dead
    
    void spacecraft_dead(N5110 &lcd,Gamepad &pad); // check if the player spacecraft is dead
    void spacecraft2_dead(N5110 &lcd,Gamepad &pad); // check if the player spacecraft is dead
    void spacecraft3_dead(N5110 &lcd,Gamepad &pad); // check if the player spacecraft is dead
     
    void check_spacecraft_collisions(Gamepad &pad); // check if the enemy spacecraft beam collided with the player spacecraft
    void check_spacecraft2_collisions(Gamepad &pad); // check if the enemy spacecraft beam collided with the player spacecraft
    void check_spacecraft3_collisions(Gamepad &pad); // check if the enemy spacecraft beam collided with the player spacecraft

    void check_enemybeam_collisions(Gamepad &pad);  // check if the enemy beam in mission one collided with the wall
    void check_enemybeam2_collisions(Gamepad &pad); // check if the enemy beam in mission two collided with the wall
    void check_enemybeam3_collisions(Gamepad &pad); // check if the enemy beam in mission three collided with the wall
    
    void check_playerbeam_collision(Gamepad &pad); // check if the player's spacecraft beam collided with the wall
    
    void draw_enemy_health(N5110 &lcd);            // draw the enemy health in mission one
    void draw_enemy21_health(N5110 &lcd);          // draw the first enemy health in mission two
    void draw_enemy22_health(N5110 &lcd);          // draw the second enemy health in mission two
    void draw_enemyboss_health(N5110 &lcd);        // draw the enemy health in mission three
    void draw_spacecraft_health(Gamepad &pad);     // draw the player's spacecraft health
    
    Spacecraft _p1; // player spacecraft
    Enemy _e1;      // first enemy spacecraft
    Enemy21 _e21;   // first enemy spacecraft in mission two
    Enemy22 _e22;   // second enemy spacecraft in mission two
    EnemyBoss _e3;  // enemy in mission three
    
    Beam _beam;             // player spacecraft's beam
    EnemyBeam _enemybeam;   // enemy beam in mission one
    EnemyBeam2 _enemybeam2; // enemy beam in mission two
    EnemyBeam3 _enemybeam3; // enemy beam in mission three
     
    int _spacecraft_xpos; // x position of the spacecrafts
    int _spacecraft_ypos; // y position of the spacecrafts
    
    int _enemy_xpos;    // x position of the enemy from the WIDTH 
    int _enemy_ypos;    // y position of the enemy from the HEIGHT 
    int _e1a;           // x position of the enemy
    int _e1b;           // y position of the enemy
    int _p1x;
    
    int _enemy2_xpos; // x position of the second enemy from the WIDTH 
    int _enemy2_ypos; // y position of the second enemy from the HEIGHT
    int _e2a;         // x position of the first enemy in mission two
    int _e21b;        // y position of the first enemy in mission two
    int _e22b;        // y position of the second enemy in mission two
    
    int _enemy3_xpos; // x position of the boss enemy from the WIDTH 
    int _enemy3_ypos; // y position of the boss enemy from the HEIGHT
    int _e3a;
    int _e3b;
    int _e3c;
    int _e3d;
    int _e3f;
    
    int _speed;      // the beginning speed
    int _beam_size;  // beam size in mission one and three
    int _beam2_size; // beam size in mission two
    int _stage;      // current game stage
    
    Direction _d;
    float _mag;
    
    bool  _L;
    bool  _R;
    bool  _ebeam;
    bool  _ebeam21;
    bool  _ebeam22;
    bool  _ebeam31;
    bool  _ebeam32;
    bool   spacebeam;
    bool  _ebeam3boss;

};

#endif