/**
 * mbed library for 4D Systems uOLED-160-G1
 *
 *
 * Copyright (c) 2010 Steven Blair
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "mbed.h"
#include "OLED160G1.h"


OLED160G1::OLED160G1(PinName serialTx, PinName serialRx, PinName resetPin) : s(serialTx, serialRx), reset(resetPin) {
    s.baud(OLED_BAUDRATE);
    
    while (s.readable()) {
        s.getc();
    }
    
    locate(0, 0);
    setFontColor(0xFFFF);
    _fontSize = OLED_FONT5X7;
}

void OLED160G1::resetDisplay() {
    reset = 0;
    wait_ms(200);
    reset = 1;
    wait_ms(200);
}

void OLED160G1::getResponse() {
    char response = OLED_NAK;
    lastCount = 0;
    NAKCount = 0;
    
    while (!s.readable() || response == OLED_NAK) {
        wait_ms(1);
        lastCount++;
        if (s.readable()) {
            response = s.getc();    // Read response
            if (response == OLED_ACK) {
                return;
            }
            else if (response == OLED_NAK) {
                NAKCount++;
            }
        }
    }
}

// Initialise OLED display. You must first activate serial comunication!
void OLED160G1::init() {
    resetDisplay();

    wait_ms(OLED_INIT_DELAY);       // wait for initialisation

    s.putc(OLED_DETECT_BAUDRATE);   // send byte for OLED to autodetect baudrate
    
    getResponse();
}

int OLED160G1::toRGB(int red, int green, int blue) {
    int outR = ((red * 31) / 255);
    int outG = ((green * 63) / 255);
    int outB = ((blue * 31) / 255);

    return (outR << 11) | (outG << 5) | outB;
}

void OLED160G1::eraseScreen() {
    s.putc(OLED_CLEAR);
    
    wait(1);    // TODO: why needed?
    
    getResponse();
    
    _row = 0;
    _column = 0;
}

void OLED160G1::drawPixel(char x, char y, int color) {
    s.putc(OLED_PUTPIXEL);
    s.putc(x);
    s.putc(y);

    s.putc(color >> 8);     // MSB
    s.putc(color & 0xFF);   // LSB

    getResponse();
}

void OLED160G1::drawLine(char x1, char y1, char x2, char y2, int color) {
    s.putc(OLED_LINE);      // Line

    s.putc(x1);
    s.putc(y1);
    s.putc(x2);
    s.putc(y2);

    s.putc(color >> 8);      // MSB          
    s.putc(color & 0xFF);    // LSB

    getResponse();
}

void OLED160G1::drawRectangle(char x, char y, char width, char height, int color) {
    s.putc(OLED_RECTANGLE); 

    s.putc(x);
    s.putc(y);

    s.putc(x+width);
    s.putc(y+height);

    s.putc(color >> 8);      // MSB          
    s.putc(color & 0xFF);    // LSB
    
    //
   // if (filled == 1) { printByte(0x01); }   // Filled
    //else { printByte(0x00); }               // Outline
    //
    getResponse();
}

void OLED160G1::drawCircle(char x, char y, char radius, int color) {
    s.putc(OLED_CIRCLE); 

    s.putc(x);
    s.putc(y);
    s.putc(radius);

    s.putc(color >> 8);     // MSB          
    s.putc(color & 0xFF);   // LSB

    getResponse();
}

void OLED160G1::setFontSize(char fontSize) {
    s.putc(OLED_SETFONTSIZE);
    s.putc(fontSize);
    _fontSize = fontSize;
    
    getResponse();
}

void OLED160G1::setFontColor(int fontColor) {
    _fontColor = fontColor;
}

void OLED160G1::setPenSize(char penSize) {
    s.putc(OLED_PEN_SIZE);
    
    s.putc(penSize);
    
    _penSize = penSize;
    
    getResponse();
}

void OLED160G1::setTextBackgroundType(char textBackgroundType) {
    s.putc(OLED_SET_TEXT_BACKGROUND_TYPE);
    s.putc(textBackgroundType);
    
    getResponse();
}

void OLED160G1::setBackgroundColor(int color) {
    s.putc(OLED_SET_BACKGROUND_COLOR);
    
    s.putc(color >> 8);      // MSB          
    s.putc(color & 0xFF);    // LSB
    
    getResponse();
}

void OLED160G1::drawText(char column, char row, char font_size, char *mytext, int color) {
    s.putc(OLED_TEXT);
    
    // Adjust to center of the screen (26 Columns at font size 0)
    //int newCol = 13 - (strlen(mytext)/2);
    //printByte(newCol); // column
    s.putc(column); // column
    
    s.putc(row); // row
    s.putc(font_size); // font size (0 = 5x7 font, 1 = 8x8 font, 2 = 8x12 font)

    s.putc(color >> 8);      // MSB          
    s.putc(color & 0xFF);    // LSB

    for (int i = 0;  i < strlen(mytext); i++) {
        s.putc(mytext[i]); // character to write
    }
    s.putc(0x00);   // string terminator (always 0x00)

    getResponse();
}

void OLED160G1::drawSingleChar(char column, char row, char theChar, int color) {
    s.putc(OLED_TEXTFORMATED);
    
    s.putc(theChar);
    s.putc(column);
    s.putc(row);
    
    s.putc(color >> 8);      // MSB          
    s.putc(color & 0xFF);    // LSB

    getResponse();
}

void OLED160G1::displayControl(char mode, char value) {
    s.putc(OLED_COMMAND_CONTROL);
    
    s.putc(mode);
    s.putc(value);
    
    getResponse();
}

char OLED160G1::getPenSize() {
    return _penSize;
}
    
int OLED160G1::_putc(int value) {
    if (value == '\n') {
        _column = 0;
        _row++;
        if(_row >= rows()) {
            _row = 0;
        }
    } else {
        drawSingleChar(_column, _row, value, _fontColor);
        
        wait_ms(1);    //TODO: why is this needed?
        
        _column++;
        if (_column >= columns()) {
            _column = 0;
            _row++;
            if(_row >= rows()) {
                _row = 0;
            }
        }
    }
    return value;
}

int OLED160G1::_getc() {
    return -1;
}

void OLED160G1::locate(int column, int row) {
    _column = column;
    _row = row;
}

int OLED160G1::rows() { 
    return 16;
}

int OLED160G1::columns() { 
    return 26;
}