/*********************************************************
*ServoDrive.cpp                                          *
*Author: Elijah Orr                                      *
*                                                        *  
*A library of functions that can be used to control the  * 
*RenBuggy via a servo motor.                             *
*********************************************************/

#ifndef SERVODRIVE_C
#define SERVODRIVE_C

#include "mbed.h"
#include "ServoDrive.h"


/* A pin is configured as PwmOut to output the PWM signal that will
control the servo. Two pins are configured as DigitalOut to switch
the drive motors on/off. Information about classes used here can be
found in the mbed library. */
PwmOut servo(servoPin);
DigitalOut Lmotor(LeftMotorPin);
DigitalOut Rmotor(RightMotorPin);

/* functions to start and stop the drive motors simply set the DigitalOut
pins to 1 or 0 to switch the motors on/off */

/****************************************************************
* Function: go()                                                *
*                                                               *
* Enables the two drive motors of the RenBuggy                  *
*                                                               *
* Inputs: none                                                  *
*                                                               *
* Returns: none                                                 *
****************************************************************/
extern void go()
{
    Lmotor = Rmotor = 1;
}

/****************************************************************
* Function: stop()                                              *
*                                                               *
* Disables the two drive motors of the RenBuggy                 *
*                                                               *
* Inputs: none                                                  *
*                                                               *
* Returns: none                                                 *
****************************************************************/
extern void stop()
{
    Lmotor = Rmotor = 0;
}

/* a function is set up to configure the PWM output so that it will have the
correct properties to operate the servo. */

/****************************************************************
* Function: configurePWM()                                      *
*                                                               *
* Configures a PWM signal for controlling the servo             *
*                                                               *
* Inputs: Period is the period of the signal in microseconds    *
* and Pulsewidth is the pulsewidth of the signal in microseconds*                                                   
*                                                               *
* Returns: none                                                 *
****************************************************************/
extern void configurePWM(int Period, int Pulsewidth)
{
    /* classes that set the properties of the PWM signal are passed the variables 
    Period and Pulsewidth to set the frequency and duty cycle of the signal. */
    servo.period_us(Period);
    servo.pulsewidth_us(Pulsewidth);
}

/* to make the main function easy to write, setDirection will allow a value in 
degrees to be passed to it, and then do any necessary conversions. angle must be
a float as the conversion involves non integer values. */

/****************************************************************
* Function: setDirection()                                      *
*                                                               *
* Sets the direction in which the steering servo is pointing    *
*                                                               *
* Inputs: A floating point value representing the angle of the  *
* servo in degrees                                              *                                                   
*                                                               *
* Returns: none                                                 *
****************************************************************/
extern void setDirection(float angle)
{   
    /* it is possible to pass numbers outside the operational range of the servo
    to setDirection, to resolve this there are two checks here. If angle is outside 
    the boundaries it will be set to either the maximum or minimum value. */
    if(angle < 0)
    {
        angle = 0;
    }
    if(angle > 90)
    {
        angle = 90;
    }
    /* to properly control the servo, the pulsewidth must be between 1000 and 2000 
    micro seconds, so a simple conversion is used to translate the angle into a value
    within these bounds. The new value is stored in the variable pulse. */
    int pulse = 1000 + ((angle/90)*1000);
    servo.pulsewidth_us(pulse); 
}

#endif //SERVODRIVE_C