/*********************************************************
*TimedMovement.cpp                                       *
*Author: Elijah Orr                                      *
*                                                        *  
*A library of functions that can be used to control the  * 
*RenBuggy.                                               *
*********************************************************/

#ifndef TIMEDMOVEMENT_C
#define TIMEDMOVEMENT_C

/* necessary includes */
#include "mbed.h"
#include "TimedMovement.h"

/* PwmOut is a class included in the mbed.h library that allows
a pin to be configured as a PWM output. This is used to control
the speed of the motors. Lmotor and Rmotor are chosen names for
the pins, LeftMotorPin and RightMotorPin (see TimedMovement.h)
specify the physical pins to be used. */
PwmOut Lmotor(LeftMotorPin);
PwmOut Rmotor(RightMotorPin);

/* Function definitions contain the code that will execute when 
the function is called. These must have the same return type
and parameters as the function declarations. */

/****************************************************************
* Function: forward()                                           *
*                                                               *
* Moves the RenBuggy directly forwards                          *
*                                                               *
* Inputs: A floating point value representing the length of     *
* time the buggy will move for                                  *
*                                                               *
* Returns: none                                                 *
****************************************************************/
extern void forward(float time)
{
    /* Lmotor and Rmotor are set to 1.0 (i.e. the motors will 
    operate at full speed). As both motors will move at the 
    same speed, the RenBuggy will go directly forward. */
    Lmotor = Rmotor = 1.0;
    /* the program will wait here for the length of time passed 
    to the function before continuing. wait() is a function 
    provided from mbed.h */
    wait(time);
    stop();
}

/****************************************************************
* Function: left()                                              *
*                                                               *
* Turns the RenBuggy to the left                                *
*                                                               *
* Inputs: A floating point value representing the length of     *
* time the buggy will turn for                                  *
*                                                               *
* Returns: none                                                 *
****************************************************************/
extern void left(float time)
{
    Rmotor = 1.0;
    Lmotor = 0.2;
    wait(time);
    stop();
}

/****************************************************************
* Function: right()                                             *
*                                                               *
* Turns the RenBuggy to the right                               *
*                                                               *
* Inputs: A floating point value representing the length of     *
* time the buggy will turn for                                  *
*                                                               *
* Returns: none                                                 *
****************************************************************/
extern void right(float time)
{
    Lmotor = 1.0;
    Rmotor = 0.2;
    wait(time);
    stop();
}

/****************************************************************
* Function: stop()                                              *
*                                                               *
* Brings the RenBuggy to a complete stop                        *
*                                                               *
* Inputs: none                                                  *
*                                                               *
* Returns: none                                                 *
****************************************************************/
void stop()
{
    Lmotor = Rmotor = 0;
}

#endif // TIMEDMOVEMENT_C