#ifndef MOTOR_H
#define MOTOR_H

#include "mbed.h"
#include "PCA9555.h"
#include "qed.h"
 /** Class for controlling motors trough PCA9555 */
class Motor {
protected:
    FunctionPointer stallChangeCallback;
    FunctionPointer stallEndCallback;
    FunctionPointer stallWarningCallback;
    FunctionPointer stallErrorCallback;
public:
    /** Create an instance of the motor connected to specfied pins, and IO-expander.
     *
     * @param PWMpin Pin for PWM output
     * @param *ioExt Pointer to IO-expander object
     * @param dir1Pin Direction pin 1 number (on IO-expander)
     * @param dir2Pin Direction pin 2 number (on IO-expander)
     * @param encA Encoder pin
     * @param encB Encoder pin
     */
    Motor(PinName PWMpin, PCA9555 *ioExt, unsigned int dirPin, PinName encA, PinName encB);
    
    /** Set speed setpoint
     *
     * @param newSpeed New setpoint
     */
    void setSpeed(int newSpeed);
    
    /** Get current speed setpoint value */
    int getSpeed();
    
    /**Method that calculates appropriate PWM values for keeping motor speed close to setpoint
    *     This method shoud be called periodically (60Hz)
    */
    void pid();
    
    int getStallLevel();
    
    void stallChange(void (*function)(void));
    
    template<typename T>
    void stallChange(T *object, void (T::*member)(void)) { 
        stallChangeCallback.attach(object, member); 
    }
    
    void stallEnd(void (*function)(void));
    
    template<typename T>
    void stallEnd(T *object, void (T::*member)(void)) { 
        stallEndCallback.attach(object, member); 
    }
    
    void stallWarning(void (*function)(void));
    
    template<typename T>
    void stallWarning(T *object, void (T::*member)(void)) { 
        stallWarningCallback.attach(object, member); 
    }
    
    void stallError(void (*function)(void));
    
    template<typename T>
    void stallError(T *object, void (T::*member)(void)) { 
        stallErrorCallback.attach(object, member); 
    }
 
private:
    PwmOut pwm;
    PCA9555 *extIO;
    unsigned int dir;
    QED qed;
    
    int currentSpeed;
    int getDecoderCount();
    
    void resetPID();

    /** Set pwm duty cycle
     *
     * @param newPWM Duty cycle
     */
    void setPWM(int newPWM);
    
    //void pid();

    int setPoint;
    int pMulti;
    int iMulti;
    int dMulti;
    int error;
    int prevError;
    int P;
    int I;
    int D;
    int minPwm;
    int pidMulti;
    int iMax;
    
    int pwmPeriod;
    
    int currentPWM;
    int stallCount;
    int prevStallCount;
    int stallWarningLimit;
    int stallErrorLimit;
    int stallLevel;
};

#endif