

#include "DHT.h"

#define DHT_DATA_BIT_COUNT 41

DHT::DHT(PinName pin,int DHTtype) {
    _pin = pin;
    _DHTtype = DHTtype;
    _firsttime=true;
}

DHT::~DHT() {
}

int DHT::readData() {
    int i, j, retryCount,b;
    unsigned int bitTimes[DHT_DATA_BIT_COUNT];

    eError err = ERROR_NONE;
    uint32_t currentTime = us_ticker_read() / 1000000;

    DigitalInOut DHT_io(_pin);

    //Clear the bit buffer. 
    for (i = 0; i < DHT_DATA_BIT_COUNT; i++) {
        bitTimes[i] = 0;
    }

    if (!_firsttime) {
        if (int(currentTime - _lastReadTime) < 2) {
            err = ERROR_NO_PATIENCE;
            return err;
        }
    } else {
        _firsttime = false;
        _lastReadTime = currentTime;
    }
    retryCount = 0;

    do {
        if (retryCount > 125) {
            err = BUS_BUSY;
            return err;
        }
        retryCount++;
        wait_us(2);
    } while (DHT_io == 0);


    //Set pin as output
    DHT_io.output(); 
    //Put the Line as High impedance 
    DHT_io = 1;
    wait_ms(250);
    
    //Start of message.
    DHT_io = 0;
    wait_ms(20);

    DHT_io = 1;
    wait_us(40);
    //Set line as input
    DHT_io.input();

    wait_us(10);
    
    retryCount = 0;

    do {
        if (retryCount > 40)  {
            err = ERROR_NOT_PRESENT;
            return err;
        }
        retryCount++;
        wait_us(1);
    } while ( DHT_io == 1 );

    if (err != ERROR_NONE) {
        return err;
    }

    for (i = 0; i < 5; i++) {
        for (j = 0; j < 8; j++) {
            retryCount = 0;
            do {
                if (retryCount > 78)  {
                    err = ERROR_DATA_TIMEOUT;
                    return err;
                }
                retryCount++;
                wait_us(1);
            } while (DHT_io == 0);
            wait_us(30);
            bitTimes[i*8 + j] = DHT_io;
            int count = 0;
            while (DHT_io == 1 && count < 100) {
                wait_us(1);
                count++;
            }
        }
    }
    DHT_io.output();
    DHT_io = 1;

    BuildRxBytes(bitTimes);
    if(ValidCheckSum()){
        _lastReadTime = currentTime;
    }
    else{
        err = ERROR_CHECKSUM;
    }

    return err;
}

float DHT::CalcTemperature() {
    int v;

    switch (_DHTtype) {
        case DHT11:
            v = DHT_data[2];
            return float(v);
        case DHT22:
            v = DHT_data[2] & 0x7F;
            v *= 256;
            v += DHT_data[3];
            v /= 10;
            if (DHT_data[2] & 0x80)
                v *= -1;
            return float(v);
    }
    return 0;
}

float DHT::ReadHumidity() {
    return _lastHumidity;
}

float DHT::ConvertCelciustoFarenheit(float celsius) {
    return celsius * 9 / 5 + 32;
}

float DHT::ConvertCelciustoKelvin(float celsius) {
    return celsius + 273.15;
}

// dewPoint function NOAA
// reference: http://wahiduddin.net/calc/density_algorithms.htm
float DHT::CalcdewPoint(float celsius, float humidity) {
    float A0= 373.15/(273.15 + celsius);
    float SUM = -7.90298 * (A0-1);
    SUM += 5.02808 * log10(A0);
    SUM += -1.3816e-7 * (pow(10, (11.344*(1-1/A0)))-1) ;
    SUM += 8.1328e-3 * (pow(10,(-3.49149*(A0-1)))-1) ;
    SUM += log10(1013.246);
    float VP = pow(10, SUM-3) * humidity;
    float T = log(VP/0.61078);   // temp var
    return (241.88 * T) / (17.558-T);
}

// delta max = 0.6544 wrt dewPoint()
// 5x faster than dewPoint()
// reference: http://en.wikipedia.org/wiki/Dew_point
float DHT::CalcdewPointFast(float celsius, float humidity)
{
        float a = 17.271;
        float b = 237.7;
        float temp = (a * celsius) / (b + celsius) + log(humidity/100);
        float Td = (b * temp) / (a - temp);
        return Td;
}

float DHT::ReadTemperature(eScale Scale) {
    if (Scale == FARENHEIT)
        return ConvertCelciustoFarenheit(_lastTemperature);
    else if (Scale == KELVIN)
        return ConvertCelciustoKelvin(_lastTemperature);
    else
        return _lastTemperature;
}

float DHT::CalcHumidity() {
    int v;

    switch (_DHTtype) {
        case DHT11:
            v = DHT_data[0];
            return float(v);
        case DHT22:
            v = DHT_data[0];
            v *= 256;
            v += DHT_data[1];
            v /= 10;
            return float(v);
    }
    return 0;
}

void  DHT::BuildRxBytes(unsigned int * bitTimes ){
    int byteValue = 0;
    for (int byteCount = 0; byteCount < 5; byteCount++) {
        byteValue = 0;
        for (int bit = 0; bit < 8; bit++) {
            if ( bitTimes[ (byteCount * 8)+ bit + 1] > 0) {
                byteValue |= ( 1 << (7 - bit) );
            }
        }
        DHT_data[byteCount] = byteValue;
    }
}

bool  DHT::ValidCheckSum(void){
    bool CorrectCheckSum = false;

    if (DHT_data[4] == ((DHT_data[0] + DHT_data[1] + DHT_data[2] + DHT_data[3]) & 0xFF)) {
        _lastTemperature = CalcTemperature();
        _lastHumidity = CalcHumidity();
        CorrectCheckSum = true;
    }

    return CorrectCheckSum;
}
