//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//                          Lots_of_DS18B20
// Demo on how to identify and read multiple DS18B20 connected to the same bus.
// Parasitic power is not used in this example.
// This program is based on the sample code from Maxim/Dallas application
// note 162 (http://www.maxim-ic.com/app-notes/index.mvp/id/162).
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// Program output should look like this:
// *** Test with multiple DS18B20 ***
// Memory allocated for 20 devices.
// Scanning for devices...
// ROM CODE =62:00:00:01:CB:25:CA:28 1
// ROM CODE =B6:00:00:01:CB:1B:9E:28 2
// ROM CODE =66:00:00:01:CB:28:59:28 3
// 3 devices found.
// Scanning completed.
// Temp: 022.6875 Device: 000001CB25CA 001
// Temp: 022.6875 Device: 000001CB1B9E 002
// Temp: 027.6250 Device: 000001CB2859 003
// Temp: 022.6250 Device: 000001CB25CA 001
// Temp: 022.6875 Device: 000001CB1B9E 002
// Temp: 025.3125 Device: 000001CB2859 003
// Temp: 022.8125 Device: 000001CB25CA 001
// Temp: 024.1875 Device: 000001CB1B9E 002
// Temp: 023.7500 Device: 000001CB2859 003
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// I have found that adding a small capacitor (100 nF or so) over the supply
// lines on each device on long cable runs significantly reduces the number of
// bad readings.
// If you make temperature conversions more often than every 4-5 seconds, the
// device(s) will self heat and produce a higher temperature reading.
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

#include <mbed.h>
Serial pc(USBTX, USBRX);

#define FALSE 0
#define TRUE 1
#define MaxROMs 20 // Defines how many devices space is allocated for.

DigitalInOut DQ(p30); // Attach the DQ pin of your sensors to this mbed pin.

unsigned char SPad[9]; // Scratchpad storage
unsigned char ROM[8];
unsigned char lastDiscrep = 0; // last discrepancy
unsigned char doneFlag = 0; // Done flag
unsigned char FoundROM[MaxROMs][8]; // table of found ROM codes
unsigned char numROMs; // Number of found devices.
unsigned char dowcrc;
unsigned char dscrc_table[] = {
0, 94,188,226, 97, 63,221,131,194,156,126, 32,163,253, 31, 65,
157,195, 33,127,252,162, 64, 30, 95, 1,227,189, 62, 96,130,220,
35,125,159,193, 66, 28,254,160,225,191, 93, 3,128,222, 60, 98,
190,224, 2, 92,223,129, 99, 61,124, 34,192,158, 29, 67,161,255,
70, 24,250,164, 39,121,155,197,132,218, 56,102,229,187, 89, 7,
219,133,103, 57,186,228, 6, 88, 25, 71,165,251,120, 38,196,154,
101, 59,217,135, 4, 90,184,230,167,249, 27, 69,198,152,122, 36,
248,166, 68, 26,153,199, 37,123, 58,100,134,216, 91, 5,231,185,
140,210, 48,110,237,179, 81, 15, 78, 16,242,172, 47,113,147,205,
17, 79,173,243,112, 46,204,146,211,141,111, 49,178,236, 14, 80,
175,241, 19, 77,206,144,114, 44,109, 51,209,143, 12, 82,176,238,
50,108,142,208, 83, 13,239,177,240,174, 76, 18,145,207, 45,115,
202,148,118, 40,171,245, 23, 73, 8, 86,180,234,105, 55,213,139,
87, 9,235,181, 54,104,138,212,149,203, 41,119,244,170, 72, 22,
233,183, 85, 11,136,214, 52,106, 43,117,151,201, 74, 20,246,168,
116, 42,200,150, 21, 75,169,247,182,232, 10, 84,215,137,107, 53};

//////////////////////////////////////////////////////////////////////////////
// OW_RESET - performs a reset on the 1-wire bus and returns the presence detect.
unsigned char ow_reset(void) {
    unsigned char presence;
    DQ.output();
    DQ = 0; //pull DQ line low
    wait_us(480); // leave it low for 480us
    DQ.input(); // allow line to return high
    wait_us(70); // wait for presence
    presence = DQ; // get presence signal
    wait_us(410); // wait for end of timeslot
    return(presence); // presence signal returned, 0=presence, 1 = no sensor found.
}

//////////////////////////////////////////////////////////////////////////////
// READ_BIT - reads a bit from the one-wire bus.
unsigned char read_bit(void) {
    unsigned char retval;
    wait_us(1); // Recovery time
    DQ.output();
    DQ = 0; // pull DQ low to start timeslot
    wait_us(2);
    DQ.input(); // Tristate line
    wait_us(10); // delay 10 us from start of timeslot
    retval=DQ;
    wait_us(48); // minimum Read time slot: 60 us.
    return(retval); // return value of DQ line
}

//////////////////////////////////////////////////////////////////////////////
// WRITE_BIT - writes a bit to the one-wire bus, passed in bitval.
void write_bit(unsigned char bitval) {
    wait_us(1); // Recovery time
    DQ.output();
    DQ = 0; // pull DQ low to start timeslot
    wait_us(10);
    if(bitval==1) DQ =1; // return DQ high if write 1
    wait_us(50); // hold value for remainder of timeslot
    DQ.input(); // Release line
}

//////////////////////////////////////////////////////////////////////////////
// READ_BYTE - reads a byte from the one-wire bus.
unsigned char read_byte(void) {
    unsigned char i, value=0;
    for (i=0;i<8;i++) {
        if(read_bit()) value|=0x01<<i; // reads byte in, one byte at a time and then shifts it left
    }
    return(value);
}

//////////////////////////////////////////////////////////////////////////////
// WRITE_BYTE - writes a byte to the one-wire bus.
void write_byte(char val) {
    unsigned char i;
    unsigned char temp;
    for (i=0; i<8; i++) {// writes byte, one bit at a time
        temp = val>>i; // shifts val right 'i' spaces
        temp &= 0x01; // copy that bit to temp
        write_bit(temp); // write bit
    }
}

//////////////////////////////////////////////////////////////////////////////
// ONE WIRE CRC
unsigned char ow_crc( unsigned char x) {
    dowcrc = dscrc_table[dowcrc^x];
    return dowcrc;
}

//////////////////////////////////////////////////////////////////////////////
// NEXT
// The Next function searches for the next device on the 1-Wire bus. If
// there are no more devices on the 1-Wire then false is returned.
unsigned char Next(void) {
    unsigned char m = 1; // ROM Bit index
    unsigned char n = 0; // ROM Byte index
    unsigned char k = 1; // bit mask
    unsigned char x = 0;
    unsigned char discrepMarker = 0; // discrepancy marker
    unsigned char g; // Output bit
    unsigned char nxt; // return value
    int flag;
    nxt = FALSE; // set the nxt flag to false
    dowcrc = 0; // reset the dowcrc
    flag = ow_reset(); // reset the 1-Wire
    if(flag||doneFlag) { // no parts -> return false
        lastDiscrep = 0; // reset the search
        return FALSE;
    }
    write_byte(0xF0); // send SearchROM command
    do { // for all eight bytes
        x = 0;
        if(read_bit()==1) x = 2;
        wait_us(120); 
        if(read_bit()==1) x |= 1; // and its complement
        if(x ==3) // there are no devices on the 1-Wire
            break;
        else {
            if(x>0) // all devices coupled have 0 or 1
                g = x>>1; // bit write value for search
            else {
                // if this discrepancy is before the last discrepancy on a previous Next then pick the same as last time
                if(m<lastDiscrep)
                    g = ((ROM[n]&k)>0);
                else // if equal to last pick 1
                    g = (m==lastDiscrep); // if not then pick 0
                // if 0 was picked then record position with mask k
                if (g==0) discrepMarker = m;
            }
            if(g==1) // isolate bit in ROM[n] with mask k
                ROM[n] |= k;
            else
                ROM[n] &= ~k;
            write_bit(g); // ROM search write
            m++; // increment bit counter m
            k = k<<1; // and shift the bit mask k
            if(k==0) { // if the mask is 0 then go to new ROM // byte n and reset mask
                ow_crc(ROM[n]); // accumulate the CRC
                n++; k++;
            }
        }
    }
    while(n<8); //loop until through all ROM bytes 0-7
    if(m<65||dowcrc) // if search was unsuccessful then
        lastDiscrep=0; // reset the last discrepancy to 0
    else { // search was successful, so set lastDiscrep, lastOne, nxt
        lastDiscrep = discrepMarker;
        doneFlag = (lastDiscrep==0);
        nxt = TRUE; // indicates search is not complete yet, more parts remain
    }
    return nxt;
}

//////////////////////////////////////////////////////////////////////////////
// FIRST
// The First function resets the current state of a ROM search and calls
// Next to find the first device on the 1-Wire bus.
unsigned char First(void) {
    lastDiscrep = 0; // reset the rom search last discrepancy global
    doneFlag = FALSE;
    return Next(); // call Next and return its return value
}

//////////////////////////////////////////////////////////////////////////////
// FIND DEVICES
void FindDevices(void) {
    unsigned char m;
    if(!ow_reset()) { //Begins when a presence is detected
        if(First()) { //Begins when at least one part is found
            numROMs=0;
            do {
                numROMs++;
                for(m=0;m<8;m++) {
                    FoundROM[numROMs][m]=ROM[m]; //Identifies ROM
                }
                pc.printf("ROM CODE =%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X %d\r\n",
                FoundROM[numROMs][7],FoundROM[numROMs][6],FoundROM[numROMs][5],FoundROM[numROMs][4],
                FoundROM[numROMs][3],FoundROM[numROMs][2],FoundROM[numROMs][1],FoundROM[numROMs][0],numROMs);
            }
            while (Next()&&(numROMs<MaxROMs)); //Continues until no additional devices are found
        }
    }
    pc.printf("\n%d devices found.\r\n\n",numROMs);
}
//////////////////////////////////////////////////////////////////////////////
void Read_ScratchPad(unsigned char n) { // Read the n first scratchpad bytes. Old data not wiped.
    n=n % 10; 
    write_byte(0xBE);
    for (int j=1;j<=n;j++){SPad[j-1]=read_byte();}
    //                                              CRC     ********reserved******* Config  Tl      Th      T MSB   T LSB
    pc.printf("\n ScratchPAD: %X%X%X%X%X%X%X%X%X\n",SPad[8],SPad[7],SPad[6],SPad[5],SPad[4],SPad[3],SPad[2],SPad[1],SPad[0]);
}

//////////////////////////////////////////////////////////////////////////////
// Perform Match ROM
unsigned char Send_MatchRom(unsigned char DeviceNo) {
    unsigned char i;
    if(ow_reset()) return false;
    write_byte(0x55); // match ROM
    for(i=0;i<8;i++) {
        write_byte(FoundROM[DeviceNo][i]); //send ROM code
    }
    return true;
}

//////////////////////////////////////////////////////////////////////////////
void ConvT() { // Make all devices on the bus start a temperature conversion.
    ow_reset();
    write_byte( 0xcc); // Skip ROM command. 
    write_byte( 0x44); // Convert T command.
}

//////////////////////////////////////////////////////////////////////////////
unsigned int ReadRawTemp(unsigned char device) {
    int HighByte, LowByte;
    Send_MatchRom(device); // Select device.
    write_byte( 0xbe); // Read Scratchpad command.
    LowByte=read_byte();
    HighByte=read_byte();
    return (HighByte << 8) + LowByte;
}

//////////////////////////////////////////////////////////////////////////////
float Get_Temp(unsigned char device) {
    int Raw = ReadRawTemp(device);
    if((Raw>>8) & 0x80) { // Check if temperature is negative.
        Raw = (Raw ^ 0xFFFF) + 1;
        Raw *= -1;
    }
    float temperature = (float)Raw / 16.0;
    return temperature;
}

//////////////////////////////////////////////////////////////////////////////
int main() {
    float temperature;
    DQ.output();
    DQ = 0;
    DQ.input();
    pc.baud(9600);
    pc.printf("\n\n*** Test with multiple DS18B20 ***\r\n\n");
    pc.printf("Memory allocated for %d devices.\r\n",MaxROMs);
    pc.printf("Scanning for devices...\r\n");
    ow_reset();
    FindDevices();
    pc.printf("Scanning completed.\r\n");

    while (1) {   
        ConvT(); // Issue Convert T command.
        wait_ms(750); // Minimum 12-bit conversion time.        
        for(int i=1;i<=numROMs;i++) { // Cycle through found devices.
            temperature = Get_Temp(i);
            pc.printf("Temp: %08.4f Device: %02X%02X%02X%02X%02X%02X %03d\r\n",temperature,FoundROM[i][6],FoundROM[i][5],FoundROM[i][4],FoundROM[i][3],FoundROM[i][2],FoundROM[i][1],i);      
        }
        pc.printf("\r\n");
        wait(5); // Doing conversions more often will make devices self heat and produce a higher temperature reading.
    }
}
