/*
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
 ******************************************************************************
 * @file:   adi_sense_api.h
 * @brief:  ADISENSE Host Library Application Programming Interface (API)
 *-----------------------------------------------------------------------------
 */

#ifndef __ADI_SENSE_API_H__
#define __ADI_SENSE_API_H__

#include "inc/adi_sense_types.h"
#include "inc/adi_sense_config_types.h"
#include "inc/adi_sense_platform.h"
#include "inc/adi_sense_gpio.h"
#include "inc/adi_sense_spi.h"
#include "inc/adi_sense_log.h"
#include "inc/adi_sense_time.h"

/*! @defgroup ADISENSE_Api ADISENSE Host Library API
 *  Host library API common to the ADISENSE product family.
 *  @{
 */

#ifdef __cplusplus
extern "C" {
#endif

/*! The maximum number of channels supported by this API
 * @note Specific ADISENSE products may implement a lower number of channels */
#define ADI_SENSE_MAX_CHANNELS 16

/*! A handle used in all API functions to identify the ADISENSE device. */
typedef void* ADI_SENSE_DEVICE_HANDLE;

/*! Supported connection types for communication with the ADISENSE device. */
typedef enum {
    ADI_SENSE_CONNECTION_TYPE_SPI = 1,
    /*!< Serial Peripheral Interface (SPI) connection type */
} ADI_SENSE_CONNECTION_TYPE;

/*! Connection details for communication with a ADISENSE device instance. */
typedef struct {
    ADI_SENSE_CONNECTION_TYPE      type;
    /*!< Connection type selection */
    ADI_SENSE_PLATFORM_SPI_CONFIG  spi;
    /*!< SPI connection parameters, required if SPI connection type is used */
    ADI_SENSE_PLATFORM_GPIO_CONFIG gpio;
    /*!< GPIO connection parameters, for device reset and status I/O signals */
    ADI_SENSE_PLATFORM_LOG_CONFIG  log;
    /*!< Log interface connection parameters, for display/routing of log messages */
} ADI_SENSE_CONNECTION;

/*! Bit masks (flags) for the different device status indicators. */
typedef enum {
    ADI_SENSE_DEVICE_STATUS_BUSY         = (1 << 0),
    /*!< Indicates that a command is currently running on the device */
    ADI_SENSE_DEVICE_STATUS_DATAREADY    = (1 << 1),
    /*!< Indicates the availability of measurement data for retrieval */
    ADI_SENSE_DEVICE_STATUS_ERROR        = (1 << 2),
    /*!< Indicates that an error condition has been detected by the device */
    ADI_SENSE_DEVICE_STATUS_ALERT        = (1 << 3),
    /*!< Indicates that an alert condition has been detected by the device */
    ADI_SENSE_DEVICE_STATUS_FIFO_ERROR   = (1 << 4),
    /*!< Indicates that a FIFO error condition has been detected by the device */
    ADI_SENSE_DEVICE_STATUS_CONFIG_ERROR = (1 << 5),
    /*!< Indicates that a configuration error condition has been detected by the device */
    ADI_SENSE_DEVICE_STATUS_LUT_ERROR    = (1 << 6),
    /*!< Indicates that a look-up table error condition has been detected by the device */
    ADI_SENSE_DEVICE_STATUS_EXT_FLASH_ERROR = (1 << 7),
    /*!< Indicates that an external flash memory error condition has been detected by the device */
} ADI_SENSE_DEVICE_STATUS_FLAGS;

/*! Bit masks (flags) for the different diagnostics status indicators. */
typedef enum {
    ADI_SENSE_DIAGNOSTICS_STATUS_CHECKSUM_ERROR       = (1 << 0),
    /*!< Indicates Error on Internal Checksum Calculations */
    ADI_SENSE_DIAGNOSTICS_STATUS_COMMS_ERROR          = (1 << 1),
    /*!< Indicates Error on Internal Device Communications */
    ADI_SENSE_DIAGNOSTICS_STATUS_SUPPLY_MONITOR_ERROR = (1 << 2),
    /*!< Indicates Low Voltage on Internal Supply Voltages */
    ADI_SENSE_DIAGNOSTICS_STATUS_SUPPLY_CAP_ERROR     = (1 << 3),
    /*!< Indicates Fault on Internal Supply Regulator Capacitor */
    ADI_SENSE_DIAGNOSTICS_STATUS_CONVERSION_ERROR     = (1 << 8),
    /*!< Indicates Error During Internal ADC Conversions */
    ADI_SENSE_DIAGNOSTICS_STATUS_CALIBRATION_ERROR    = (1 << 9),
    /*!< Indicates Error During Internal Device Calibrations */
} ADI_SENSE_DIAGNOSTICS_STATUS_FLAGS;

/*! Bit masks (flags) for the different channel alert indicators. */
typedef enum {
    ADI_SENSE_CHANNEL_ALERT_TIMEOUT          = (1 << 0),
    /*!< Indicates timeout condition detected on the channel */
    ADI_SENSE_CHANNEL_ALERT_UNDER_RANGE      = (1 << 1),
    /*!< Indicates raw sample under valid input range, possibly clamped */
    ADI_SENSE_CHANNEL_ALERT_OVER_RANGE       = (1 << 2),
    /*!< Indicates raw sample over valid input range, possibly clamped */
    ADI_SENSE_CHANNEL_ALERT_LOW_LIMIT        = (1 << 3),
    /*!< Indicates measurement result was below configured minimum threshold */
    ADI_SENSE_CHANNEL_ALERT_HIGH_LIMIT       = (1 << 4),
    /*!< Indicates measurement result was above configured maximum threshold */
    ADI_SENSE_CHANNEL_ALERT_SENSOR_OPEN      = (1 << 5),
    /*!< Indicates open circuit or mis-wire condition detected on the channel */
    ADI_SENSE_CHANNEL_ALERT_REF_DETECT       = (1 << 6),
    /*!< Indicates reference-detect error condition detected on the channel */
    ADI_SENSE_CHANNEL_ALERT_CONFIG_ERR       = (1 << 7),
    /*!< Indicates configuration error condition detected on the channel */
    ADI_SENSE_CHANNEL_ALERT_LUT_ERR          = (1 << 8),
    /*!< Indicates look-up table error condition detected on the channel */
    ADI_SENSE_CHANNEL_ALERT_SENSOR_NOT_READY = (1 << 9),
    /*!< Indicates digital sensor not-ready error condition detected on the channel */
    ADI_SENSE_CHANNEL_ALERT_COMP_NOT_READY   = (1 << 10),
    /*!< Indicates compensation channel not-ready error condition detected on the channel */
    ADI_SENSE_CHANNEL_ALERT_LUT_UNDER_RANGE  = (1 << 13),
    /*!< Indicates raw sample was under the available LUT/equation range */
    ADI_SENSE_CHANNEL_ALERT_LUT_OVER_RANGE   = (1 << 14),
    /*!< Indicates raw sample was over the available LUT/equation range */
} ADI_SENSE_CHANNEL_ALERT_FLAGS;

/*! Status details retreived from the ADISENSE device. */
typedef struct {
    ADI_SENSE_DEVICE_STATUS_FLAGS      deviceStatus;
    /*!< General summary status information from the device  */
    ADI_SENSE_DIAGNOSTICS_STATUS_FLAGS diagnosticsStatus;
    /*!< Diagnostic error status information from the device  */
    ADI_SENSE_CHANNEL_ALERT_FLAGS      channelAlerts[ADI_SENSE_MAX_CHANNELS];
    /*!< Per-channel alert status information from the device  */
    uint32_t                           errorCode;
    /*!< Code identifying the last error signalled by the device  */
    uint32_t                           alertCode;
    /*!< Code identifying the last alert signalled by the device  */
    uint32_t                           channelAlertCodes[ADI_SENSE_MAX_CHANNELS];
    /*!< Per-channel code identifying the last alert signalled for each channel */
} ADI_SENSE_STATUS;

/*! Data sample details retreived from the ADISENSE device. */
typedef struct {
    ADI_SENSE_DEVICE_STATUS_FLAGS      status;
    /*!< Device summary status snapshot when the sample was recorded */
    uint32_t                           channelId;
    /*!< The measurement channel from which this sample was obtained */
    uint32_t                           rawValue;
    /*!< The raw (unprocessed) value obtained directly from the measurement
     *   channel, if available
     */
    float32_t                          processedValue;
    /*!< The processed value obtained from the measurement channel, as a final
     *   measurement value, following calibration and linearisation correction,
     *   and conversion into an appropriate unit of measurement.
     */
} ADI_SENSE_DATA_SAMPLE;

/*! Measurement mode options for the ADISENSE device.
 *  @ref adi_sense_StartMeasurement
 */
typedef enum {
    ADI_SENSE_MEASUREMENT_MODE_HEALTHCHECK = 1,
    /*!< In this mode, a special health-check measurement cycle is executed,
     *   carrying out a single conversion per channel with measurement
     *   diagnostics enabled, intended for use as a system health check. */
    ADI_SENSE_MEASUREMENT_MODE_NORMAL,
    /*!< In this mode, normal measurement cycle(s) are executed and data samples
     *   are returned with raw measurement values included. */
    ADI_SENSE_MEASUREMENT_MODE_OMIT_RAW,
    /*!< In this mode, normal measurement cycle(s) are executed and data samples
     *   are returned with raw measurement values omitted for efficiency. */
    ADI_SENSE_MEASUREMENT_MODE_FFT,
    /*!< In this mode, FFT mode measurement cycle(s) are executed and data
     *   samples are returned based on FFT-specific configuration parameters. */
} ADI_SENSE_MEASUREMENT_MODE;

/*! Identifiers for the user configuration slots in persistent memory. */
typedef enum {
    ADI_SENSE_FLASH_CONFIG_1,
    ADI_SENSE_FLASH_CONFIG_2,
    ADI_SENSE_FLASH_CONFIG_3,
    ADI_SENSE_FLASH_CONFIG_4,
} ADI_SENSE_USER_CONFIG_SLOT;


/******************************************************************************
 * ADISENSE High-Level API function prototypes
 *****************************************************************************/

/*!
 * @brief Open ADISENSE device handle and set up communication interface.
 *
 * @param[in]  nDeviceIndex    Zero-based index number identifying this device
 *                             instance.  Note that this will be used to
 *                             retrieve a specific device configuration for
 *                             this device (see @ref adi_sense_SetConfig
 *                             and @ref ADI_SENSE_CONFIG)
 * @param[in]  pConnectionInfo Host-specific connection details (e.g. SPI, GPIO)
 * @param[out] phDevice        Pointer to return an ADISENSE device handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *         - #ADI_SENSE_NO_MEM  Failed to allocate memory resources.
 *         - #ADI_SENSE_INVALID_DEVICE_NUM Invalid device index specified
 *
 * @details Configure and initialise the Log interface and the SPI/GPIO
 *          communication interface to the ADISense module.
 */
ADI_SENSE_RESULT adi_sense_Open(
    unsigned                   const nDeviceIndex,
    ADI_SENSE_CONNECTION     * const pConnectionInfo,
    ADI_SENSE_DEVICE_HANDLE  * const phDevice);

/*!
 * @brief Close ADISENSE device context and free resources.
 *
 * @param[in] hDevice ADISENSE device context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 */
ADI_SENSE_RESULT adi_sense_Close(
    ADI_SENSE_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Get the current state of the specified GPIO input signal.
 *
 * @param[in]  hDevice    ADISENSE device context handle
 * @param[in]  ePinId     GPIO pin to query
 * @param[out] pbAsserted Pointer to return the state of the status signal GPIO pin
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *         - #ADI_SENSE_INVALID_DEVICE_NUM Invalid GPIO pin specified.
 *
 * @details Sets *pbAsserted to true if the status signal is asserted, or false
 *          otherwise.
 */
ADI_SENSE_RESULT adi_sense_GetGpioState(
    ADI_SENSE_DEVICE_HANDLE   const hDevice,
    ADI_SENSE_GPIO_PIN        const ePinId,
    bool_t                  * const pbAsserted);

/*!
 * @brief Register an application-defined callback function for GPIO interrupts
 *
 * @param[in] hDevice          ADISENSE context handle (@ref adi_sense_Open)
 * @param[in] ePinId           GPIO pin on which to enable/disable interrupts
 * @param[in] callbackFunction Function to be called when an interrupt occurs.
 *                             Specify NULL here to disable interrupts.
 * @param[in] pCallbackParam   Optional opaque parameter passed to the callback
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *         - #ADI_SENSE_INVALID_DEVICE_NUM Invalid GPIO pin specified.
 */
ADI_SENSE_RESULT adi_sense_RegisterGpioCallback(
    ADI_SENSE_DEVICE_HANDLE          const hDevice,
    ADI_SENSE_GPIO_PIN               const ePinId,
    ADI_SENSE_GPIO_CALLBACK          const callbackFunction,
    void                           * const pCallbackParam);

/*!
 * @brief Trigger a shut down of the device.
 *
 * @param[in] hDevice ADISENSE device context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADISENSE device to initiate a shut down,
 *          typically used to conserve power when the device is not
 *          in use.  The device may be restarted by calling
 *          @ref adi_sense_Reset().  Note that active configuration
 *          settings are not preserved during shutdown and must be
 *          reloaded after the device has become ready again.
 *
 * @note No other command must be running when this is called.
 */
ADI_SENSE_RESULT adi_sense_Shutdown(
    ADI_SENSE_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Reset the ADISENSE device.
 *
 * @param[in] hDevice ADISENSE device context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Trigger a hardware-reset of the ADISENSE device.
 *
 * @note The device may require several seconds before it is ready for use
 *       again. @ref adi_sense_GetDeviceReadyState may be used to check if
 *       the device is ready.
 */
ADI_SENSE_RESULT adi_sense_Reset(
    ADI_SENSE_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Check if the device is ready, following power-up or a reset.
 *
 * @param[in]  hDevice ADISENSE device context handle
 * @param[out] pbReady Pointer to return true if the device is ready, or false
 *                     otherwise
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details This function attempts to read a fixed-value device register via
 *           the communication interface.
 */
ADI_SENSE_RESULT adi_sense_GetDeviceReadyState(
    ADI_SENSE_DEVICE_HANDLE    const hDevice,
    bool_t                   * const pbReady);

/*!
 * @brief Obtain the product ID from the device.
 *
 * @param[in]  hDevice    ADISENSE device context handle
 * @param[out] pProductId Pointer to return the product ID value
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Reads the product ID registers on the device and returns the value.
 */
ADI_SENSE_RESULT adi_sense_GetProductID(
    ADI_SENSE_DEVICE_HANDLE    const hDevice,
    ADI_SENSE_PRODUCT_ID     * const pProductId);

/*!
 * @brief Write full configuration settings to the device registers.
 *
 * @param[in]  hDevice ADISENSE device context handle
 * @param[out] pConfig Pointer to the configuration data structure
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *
 * @note Settings are not applied until adi_sense_ApplyConfigUpdates() is called
 */
ADI_SENSE_RESULT adi_sense_SetConfig(
    ADI_SENSE_DEVICE_HANDLE   const hDevice,
    ADI_SENSE_CONFIG        * const pConfig);

/*!
 * @brief Apply the configuration settings currently stored in device registers
 *
 * @param[in] hDevice ADISENSE device context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADISENSE device to reload and apply configuration
 *          from the device configuration registers.  Changes to configuration
 *          registers are ignored by the device until this function is called.
 *
 * @note No other command must be running when this is called.
 */
ADI_SENSE_RESULT adi_sense_ApplyConfigUpdates(
    ADI_SENSE_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Store the configuration settings to persistent memory on the device.
 *
 * @param[in] hDevice ADISENSE device context handle
 * @param[in] eSlotId User configuration slot in persistent memory
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADISENSE device to save the current contents of its
 *          device configuration registers to non-volatile memory.
 *
 * @note No other command must be running when this is called.
 * @note Do not power down the device while this command is running.
 */
ADI_SENSE_RESULT adi_sense_SaveConfig(
    ADI_SENSE_DEVICE_HANDLE    const hDevice,
    ADI_SENSE_USER_CONFIG_SLOT const eSlotId);

/*!
 * @brief Restore configuration settings from persistent memory on the device.
 *
 * @param[in] hDevice ADISENSE device context handle
 * @param[in] eSlotId User configuration slot in persistent memory
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADISENSE device to restore the contents of its
 *          device configuration registers from non-volatile memory.
 *
 * @note No other command must be running when this is called.
 */
ADI_SENSE_RESULT adi_sense_RestoreConfig(
    ADI_SENSE_DEVICE_HANDLE    const hDevice,
    ADI_SENSE_USER_CONFIG_SLOT const eSlotId);

/*!
 * @brief Erases the external flash memory.
 *
 * @param[in] hDevice ADISENSE device context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Sends the bulk erase instruction to the external flash
 *          device. All stored samples are deleted.
 *          It is a blocking operation and takes tens of seconds to
 *          complete.
 *
 * @note No other command must be running when this is called.
 */
ADI_SENSE_RESULT adi_sense_EraseExternalFlash(
    ADI_SENSE_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Gets the number of samples stored in the external flash
 *        memory.
 *
 * @param[in] hDevice ADISENSE device context handle
 * @param[in] pSampleCount Address of the return value.
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @note No other command must be running when this is called.
 */
ADI_SENSE_RESULT adi_sense_GetExternalFlashSampleCount(
    ADI_SENSE_DEVICE_HANDLE    const hDevice,
    uint32_t * nSampleCount);

// DEBUG - TO BE DELETED
ADI_SENSE_RESULT adi_sense_SetExternalFlashIndex(
    ADI_SENSE_DEVICE_HANDLE    const hDevice,
    uint32_t nStartIndex);

/*!
 * @brief Read measurement samples stored in the the external flash memory.
 *
 * @param[in]  hDevice          ADISENSE device context handle
 * @param[out] pSamples         Pointer to return a set of requested data
 *                              samples.
 * @param[in]  nStartIndex      Index of first sample to retrieve.
 * @param[in]  nBytesPerSample  The size, in bytes, of each sample.
 * @param[in]  nRequested       Number of requested data samples.
 * @param[out] pnReturned       Number of valid data samples successfully
 *                              retrieved.
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Reads the status registers and extracts the relevant information
 *          to return to the caller.
 *
 */
ADI_SENSE_RESULT adi_sense_GetExternalFlashData(
    ADI_SENSE_DEVICE_HANDLE    const hDevice,
    ADI_SENSE_DATA_SAMPLE    * const pSamples,
    uint32_t                   const nIndex,
    uint32_t                   const nRequested,
    uint32_t                 * const pnReturned);

/*!
 * @brief Store the LUT data to persistent memory on the device.
 *
 * @param[in] hDevice ADISENSE device context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADISENSE device to save the current contents of its
 *          LUT data buffer, set using @ref adi_sense_SetLutData, to
 *          non-volatile memory.
 *
 * @note No other command must be running when this is called.
 * @note Do not power down the device while this command is running.
 */
ADI_SENSE_RESULT adi_sense_SaveLutData(
    ADI_SENSE_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Restore LUT data from persistent memory on the device.
 *
 * @param[in] hDevice ADISENSE device context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADISENSE device to restore the contents of its
 *          LUT data, previously stored with @ref adi_sense_SaveLutData, from
 *          non-volatile memory.
 *
 * @note No other command must be running when this is called.
 */
ADI_SENSE_RESULT adi_sense_RestoreLutData(
    ADI_SENSE_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Start the measurement cycles on the device.
 *
 * @param[in] hDevice          ADISENSE device context handle
 * @param[in] eMeasurementMode Allows a choice of special modes for the
 *                             measurement.  See @ref ADI_SENSE_MEASUREMENT_MODE
 *                             for further information.
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADISENSE device to start executing measurement cycles
 *          according to the current applied configuration settings.  The
 *          DATAREADY status signal will be asserted whenever new measurement
 *          data is published, according to selected settings.
 *          Measurement cycles may be stopped by calling @ref
 *          adi_sense_StopMeasurement.
 *
 * @note No other command must be running when this is called.
 */
ADI_SENSE_RESULT adi_sense_StartMeasurement(
    ADI_SENSE_DEVICE_HANDLE    const hDevice,
    ADI_SENSE_MEASUREMENT_MODE const eMeasurementMode);

/*!
 * @brief Stop the measurement cycles on the device.
 *
 * @param[in] hDevice ADISENSE device context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADISENSE device to stop executing measurement cycles.
 *          The command may be delayed until the current conversion, if any, has
 *          been completed and published.
 *
 * @note To be used only if a measurement command is currently running.
 */
ADI_SENSE_RESULT adi_sense_StopMeasurement(
    ADI_SENSE_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Run built-in diagnostic checks on the device.
 *
 * @param[in] hDevice ADISENSE device context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADISENSE device to execute its built-in diagnostic
 *          tests, on any enabled measurement channels, according to the current
 *          applied configuration settings.  Device status registers will be
 *          updated to indicate if any errors were detected by the diagnostics.
 *
 * @note No other command must be running when this is called.
 */
ADI_SENSE_RESULT adi_sense_RunDiagnostics(
    ADI_SENSE_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Run built-in calibration on the device.
 *
 * @param[in] hDevice ADISENSE device context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADISENSE device to execute its self-calibration
 *          routines, on any enabled measurement channels, according to the
 *          current applied configuration settings.  Device status registers
 *          will be updated to indicate if any errors were detected.
 *
 * @note No other command must be running when this is called.
 */
ADI_SENSE_RESULT adi_sense_RunCalibration(
    ADI_SENSE_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Run built-in digital calibration on the device.
 *
 * @param[in] hDevice ADISENSE device context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADISENSE device to execute its calibration
 *          routines, on any enabled digital channels, according to the
 *          current applied configuration settings.  Device status registers
 *          will be updated to indicate if any errors were detected.
 *
 * @note No other command must be running when this is called.
 */
ADI_SENSE_RESULT adi_sense_RunDigitalCalibration(
    ADI_SENSE_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Read the current status from the device registers.
 *
 * @param[in]  hDevice ADISENSE device context handle
 * @param[out] pStatus Pointer to return the status summary obtained from the
 *                     device.
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Reads the status registers and extracts the relevant information
 *          to return to the caller.
 *
 * @note This may be called at any time, assuming the device is ready.
 */
ADI_SENSE_RESULT adi_sense_GetStatus(
    ADI_SENSE_DEVICE_HANDLE    const hDevice,
    ADI_SENSE_STATUS         * const pStatus);

/*!
 * @brief Read measurement data samples from the device registers.
 *
 * @param[in]  hDevice          ADISENSE device context handle
 * @param[in]  eMeasurementMode Must be set to the same value used for @ref
 *                              adi_sense_StartMeasurement().
 * @param[out] pSamples         Pointer to return a set of requested data samples.
 * @param[in]  nBytesPerSample  The size, in bytes, of each sample.
 * @param[in]  nRequested       Number of requested data samples.
 * @param[out] pnReturned       Number of valid data samples successfully retrieved.
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Reads the status registers and extracts the relevant information
 *          to return to the caller.
 *
 * @note This is intended to be called only when the DATAREADY status signal
 *       is asserted.
 */
ADI_SENSE_RESULT adi_sense_GetData(
    ADI_SENSE_DEVICE_HANDLE    const hDevice,
    ADI_SENSE_MEASUREMENT_MODE const eMeasurementMode,
    ADI_SENSE_DATA_SAMPLE    * const pSamples,
    uint8_t                    const nBytesPerSample,
    uint32_t                   const nRequested,
    uint32_t                 * const pnReturned);

/*!
 * @brief Check if a command is currently running on the device.
 *
 * @param[in]  hDevice          ADISENSE device context handle
 * @param[out] pbCommandRunning Pointer to return the command running status
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully.
 *
 * @details Reads the device status register to check if a command is running.
 */
ADI_SENSE_RESULT adi_sense_GetCommandRunningState(
    ADI_SENSE_DEVICE_HANDLE hDevice,
    bool_t *pbCommandRunning);

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADI_SENSE_API_H__ */
