/*
CONFIDENTIAL AND PROPRIETARY INFORMATION

Copyright (c) 2018 Emutex Ltd. All rights reserved.
This software and documentation contain confidential and
proprietary information that is the property of
Emutex Ltd. The software and documentation are
furnished under a license agreement and may be used
or copied only in accordance with the terms of the license
agreement. No part of the software and documentation
may be reproduced, transmitted, or translated, in any
form or by any means, electronic, mechanical, manual,
optical, or otherwise, without prior written permission
of Emutex Ltd., or as expressly provided by the license agreement.
Reverse engineering is prohibited, and reproduction,
disclosure or use without specific written authorization
of Emutex Ltd. is strictly forbidden.
 */

/*
Copyright (c) 2017 Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
  - Modified versions of the software must be conspicuously marked as such.
  - This software is licensed solely and exclusively for use with processors
    manufactured by or for Analog Devices, Inc.
  - This software may not be combined or merged with other code in any manner
    that would cause the software to become subject to terms and conditions
    which differ from those listed here.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights of one
    or more patent holders.  This license does not release you from the
    requirement that you obtain separate licenses from these patent holders
    to use this software.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES, INC. AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
TITLE, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
NO EVENT SHALL ANALOG DEVICES, INC. OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, PUNITIVE OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, DAMAGES ARISING OUT OF CLAIMS OF INTELLECTUAL
PROPERTY RIGHTS INFRINGEMENT; PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
 ******************************************************************************
 * @file:   adi_sense_1000_config.h
 * @brief:  Configuration type definitions for ADSNS1000.
 *-----------------------------------------------------------------------------
 */

#ifndef __ADI_SENSE_1000_CONFIG_H__
#define __ADI_SENSE_1000_CONFIG_H__

#include "adi_sense_platform.h"
#include "adi_sense_1000_sensor_types.h"

/*! @addtogroup ADSNS1000_Api
 *  @{
 */

#ifdef __cplusplus
extern "C" {
#endif

/*! Maximum length allowed for a digital sensor command */
#define ADI_SENSE_1000_SENSOR_COMMAND_MAX_LENGTH 7

/*! ADSNS1000 channel priority options */
typedef enum {
    ADI_SENSE_1000_CHANNEL_PRIORITY_0 = 0,
    ADI_SENSE_1000_CHANNEL_PRIORITY_1,
    ADI_SENSE_1000_CHANNEL_PRIORITY_2,
    ADI_SENSE_1000_CHANNEL_PRIORITY_3,
    ADI_SENSE_1000_CHANNEL_PRIORITY_4,
    ADI_SENSE_1000_CHANNEL_PRIORITY_5,
    ADI_SENSE_1000_CHANNEL_PRIORITY_6,
    ADI_SENSE_1000_CHANNEL_PRIORITY_7,
    ADI_SENSE_1000_CHANNEL_PRIORITY_8,
    ADI_SENSE_1000_CHANNEL_PRIORITY_9,
    ADI_SENSE_1000_CHANNEL_PRIORITY_10,
    ADI_SENSE_1000_CHANNEL_PRIORITY_11,
    ADI_SENSE_1000_CHANNEL_PRIORITY_12,
    ADI_SENSE_1000_CHANNEL_PRIORITY_13,
    ADI_SENSE_1000_CHANNEL_PRIORITY_14,
    ADI_SENSE_1000_CHANNEL_PRIORITY_15,

    ADI_SENSE_1000_CHANNEL_PRIORITY_HIGHEST = ADI_SENSE_1000_CHANNEL_PRIORITY_0,
    ADI_SENSE_1000_CHANNEL_PRIORITY_LOWEST  = ADI_SENSE_1000_CHANNEL_PRIORITY_15,
} ADI_SENSE_1000_CHANNEL_PRIORITY;

/*! ADSNS1000 operating mode options */
typedef enum {
    ADI_SENSE_1000_OPERATING_MODE_SINGLECYCLE = 1,
    /*!< Executes a single measurement cycle and stops */
    ADI_SENSE_1000_OPERATING_MODE_CONTINUOUS,
    /*!< Continuously executes measurement cycles */
    ADI_SENSE_1000_OPERATING_MODE_MULTICYCLE,
    /*!< Executes a burst of measurement cycles, repeated at defined intervals */
} ADI_SENSE_1000_OPERATING_MODE;

/*! ADSNS1000 data ready mode options */
typedef enum {
    ADI_SENSE_1000_DATAREADY_PER_CONVERSION = 1,
    /*!< The DATAREADY signal is asserted after completion of each conversion
     *   - a single data sample only from the latest completed conversion is
     *     stored in this mode
     */
    ADI_SENSE_1000_DATAREADY_PER_CYCLE,
    /*!< The DATAREADY signal is asserted after completion of each measurement
     *   cycle
     *   - data samples only from the lastest completed measurement cycle are
     *     stored in this mode
     */
    ADI_SENSE_1000_DATAREADY_PER_MULTICYCLE_BURST,
    /*!< The DATAREADY signal is asserted after completion of each burst of
     *   measurement cycles
     *   - applicable only when @ref ADI_SENSE_1000_OPERATING_MODE_MULTICYCLE
     *     is also selected
     *   - data samples only from the lastest completed burst of measurement
     *     cycles are stored in this mode
     */
} ADI_SENSE_1000_DATAREADY_MODE;

/*! ADSNS1000 calibration mode options */
typedef enum {
    ADI_SENSE_1000_NO_CALIBRATION = 0,
    /*!< No full scale ADC calibration is performed after a latch command is sent.
     */
    ADI_SENSE_1000_DO_CALIBRATION,
    /*!< Full scale ADC Calibration is performed after a latch command is sent.
     */
} ADI_SENSE_1000_CALIBRATION_MODE;

/*! ADSNS1000 power mode options */
typedef enum {
    ADI_SENSE_1000_POWER_MODE_LOW = 1,
    /*!< Lowest ADC power consumption mode, with lowest conversion rate */
    ADI_SENSE_1000_POWER_MODE_MID,
    /*!< Medium ADC power consumption mode, with medium conversion rate */
    ADI_SENSE_1000_POWER_MODE_FULL,
    /*!< Highest ADC power consumption mode, with highest conversion rate */
} ADI_SENSE_1000_POWER_MODE;

/*! ADSNS1000 measurement cycle types */
typedef enum
{
    ADI_SENSE_1000_CYCLE_TYPE_SWITCH = 0,
    /*!< Switch channels after every conversion */
    ADI_SENSE_1000_CYCLE_TYPE_FULL   = 1
    /*!< Perform full number of requested conversions on a channel
     *   consecutively before switching to the next channel */
} ADI_SENSE_1000_CYCLE_TYPE;

/*! ADSNS1000 measurement unit options
 *
 *  Optionally select a measurement unit for final conversion results.
 *  Currently applicable only to specific temperature sensor types.
 */
typedef enum {
    ADI_SENSE_1000_MEASUREMENT_UNIT_UNSPECIFIED = 0,
    /*!< No measurement unit specified */
    ADI_SENSE_1000_MEASUREMENT_UNIT_CELSIUS,
    /*!< Celsius temperature unit - applicable to temperature sensors only */
    ADI_SENSE_1000_MEASUREMENT_UNIT_FAHRENHEIT,
    /*!< Fahrenheit temperature unit - applicable to temperature sensors only */
} ADI_SENSE_1000_MEASUREMENT_UNIT;

/*! ADSNS1000 Open-Sensor Diagnostics frequency
 *
 *  Select the per-cycle frequency at which open-sensor diagnostic
 *  checks should be performed.  Open-sensor diagnostic checks typically require
 *  specific or time-consuming processing which cannot be executed while a
 *  measurement cycle is running.
 *
 *  @note Open-sensor diagnostic checks, when performed, will add a delay to the
 *        start of the next measurement cycle.
 */
typedef enum {
    ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS_DISABLED = 0,
    /*!< No Open-Sensor Detection is performed */
    ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS_PER_CYCLE,
    /*!< No Open-Sensor Detection is performed prior to each cycle */
    ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS_PER_100_CYCLES,
    /*!< No Open-Sensor Detection is performed at intervals of 100 cycles */
    ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS_PER_1000_CYCLES,
    /*!< No Open-Sensor Detection is performed at intervals of 1000 cycles */
} ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS;

/*! ADSNS1000 analog input signal amplification gain options
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef enum {
    ADI_SENSE_1000_ADC_GAIN_1X = 0,
    /*!< no amplification gain */
    ADI_SENSE_1000_ADC_GAIN_2X,
    /*!< x2 amplification gain */
    ADI_SENSE_1000_ADC_GAIN_4X,
    /*!< x4 amplification gain */
    ADI_SENSE_1000_ADC_GAIN_8X,
    /*!< x8 amplification gain */
    ADI_SENSE_1000_ADC_GAIN_16X,
    /*!< x16 amplification gain */
    ADI_SENSE_1000_ADC_GAIN_32X,
    /*!< x32 amplification gain */
    ADI_SENSE_1000_ADC_GAIN_64X,
    /*!< x64 amplification gain */
    ADI_SENSE_1000_ADC_GAIN_128X,
    /*!< x128 amplification gain */
} ADI_SENSE_1000_ADC_GAIN;

/*! ADSNS1000 analog sensor excitation current output level options
 *
 *  @note applicable only to ADC analog sensor channels, and
 *        specific sensor types
 */
typedef enum {
    ADI_SENSE_1000_ADC_EXC_CURRENT_NONE = 0,
    /*!< Excitation current disabled */
    ADI_SENSE_1000_ADC_EXC_CURRENT_50uA,
    /*!< 50uA excitation current enabled */
    ADI_SENSE_1000_ADC_EXC_CURRENT_100uA,
    /*!< 100uA excitation current  */
    ADI_SENSE_1000_ADC_EXC_CURRENT_250uA,
    /*!< 250uA excitation current enabled */
    ADI_SENSE_1000_ADC_EXC_CURRENT_500uA,
    /*!< 500uA excitation current enabled */
    ADI_SENSE_1000_ADC_EXC_CURRENT_750uA,
    /*!< 750uA excitation current enabled */
    ADI_SENSE_1000_ADC_EXC_CURRENT_1000uA,
    /*!< 1mA excitation current enabled */
} ADI_SENSE_1000_ADC_EXC_CURRENT;

/*! ADSNS1000 analog sensor excitation current ratios used for diode sensor
 *
 *  @note applicable only to a diode sensor
 */
typedef enum {
    ADI_SENSE_1000_ADC_EXC_CURRENT_IOUT_DIODE_DEFAULT = 0,
    ADI_SENSE_1000_ADC_EXC_CURRENT_IOUT_DIODE_MAX,
} ADI_SENSE_1000_ADC_EXC_CURRENT_DIODE_RATIO;

/*! ADSNS1000 analog reference selection options
 *
 *  @note applicable only to ADC analog sensor channels, and
 *        specific sensor types
 */
typedef enum {
    ADI_SENSE_1000_ADC_REFERENCE_NONE = 0,
    /*!< No reference is selected */

    ADI_SENSE_1000_ADC_REFERENCE_RESISTOR_INTERNAL_1,
    /*!< Internal reference resistor #1 (4320 ohms) is selected */
    ADI_SENSE_1000_ADC_REFERENCE_RESISTOR_INTERNAL_2,
    /*!< Internal reference resistor #2 (10000 ohms) is selected */
    ADI_SENSE_1000_ADC_REFERENCE_VOLTAGE_INTERNAL,
    /*!< Internal ADC voltage reference (2.5V) is selected */
    ADI_SENSE_1000_ADC_REFERENCE_VOLTAGE_AVDD,
    /*!< Analag Supply Voltage AVDD reference (typically 3.3V) is selected */

    ADI_SENSE_1000_ADC_REFERENCE_RESISTOR_EXTERNAL_1,
    /*!< External user-supplied reference resistor #1 is selected
     *
     * @note reference resistor value externalRef1Value must be specified
     * (see @ref ADI_SENSE_1000_MEASUREMENT_CONFIG) */
    ADI_SENSE_1000_ADC_REFERENCE_RESISTOR_EXTERNAL_2,
    /*!< External user-supplied reference resistor #2 is selected
     *
     * @note reference resistor value externalRef2Value must be specified
     * (see @ref ADI_SENSE_1000_MEASUREMENT_CONFIG) */
    ADI_SENSE_1000_ADC_REFERENCE_VOLTAGE_EXTERNAL_1,
    /*!< External user-supplied reference voltage #1 is selected
     *
     * @note reference voltage value externalRef1Value must be specified
     * (see @ref ADI_SENSE_1000_MEASUREMENT_CONFIG) */
    ADI_SENSE_1000_ADC_REFERENCE_VOLTAGE_EXTERNAL_2,
    /*!< External user-supplied reference voltage #2 is selected
     *
     * @note reference voltage value externalRef2Value must be specified
     * (see @ref ADI_SENSE_1000_MEASUREMENT_CONFIG) */
    ADI_SENSE_1000_ADC_REFERENCE_BRIDGE_EXCITATION,
    /*!< Bridge Excition Voltage is selected as reference
     *
     * @note this reference MUST be selected for 4/6-wire bridge sensor types
     * (see @ref ADI_SENSE_1000_ADC_SENSOR_TYPE) */
} ADI_SENSE_1000_ADC_REFERENCE_TYPE;

/*! ADSNS1000 analog filter selection options
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef enum {
    ADI_SENSE_1000_ADC_FILTER_SINC4 = 1,
    /*!< SINC4 filter
     *
     * @note The number of input samples to be averaged by the filter
     *       must be specified by fs (see @ref ADI_SENSE_1000_ADC_FILTER_CONFIG).
     *       The fs value affects the output sample rate and settling times
     *       of the ADC */
    ADI_SENSE_1000_ADC_FILTER_FIR_20SPS,
    /*!< FIR post filter, producing a 20sps output sample rate */
    ADI_SENSE_1000_ADC_FILTER_FIR_25SPS,
    /*!< FIR post filter, producing a 25sps output sample rate */
} ADI_SENSE_1000_ADC_FILTER_TYPE;

/*! ADSNS1000 FFT sequence mode options
 *
 *  @note applicable only for FFT measurement modes
 */
typedef enum {
    ADI_SENSE_1000_FFT_MODE_SINGLE = 0,
    /*!< Performs a single sequence of FFTs on selected channels and stops */
    ADI_SENSE_1000_FFT_MODE_CONTINUOUS,
    /*!< Performs continuous sequences of FFTs on selected channels */
} ADI_SENSE_1000_FFT_MODE;

/*! ADSNS1000 FFT size options (number of bins)
 *
 *  @note applicable only for FFT measurement modes
 */
typedef enum {
    ADI_SENSE_1000_FFT_SIZE_256 = 0,
    /*!< 256 bins */
    ADI_SENSE_1000_FFT_SIZE_512,
    /*!< 512 bins */
    ADI_SENSE_1000_FFT_SIZE_1024,
    /*!< 1024 bins */
    ADI_SENSE_1000_FFT_SIZE_2048,
    /*!< 2048 bins */
} ADI_SENSE_1000_FFT_SIZE;

/*! ADSNS1000 FFT window type options
 *
 *  @note applicable only for FFT measurement modes
 */
typedef enum {
    ADI_SENSE_1000_FFT_WINDOW_NONE = 0,
    /*!< No Window */
    ADI_SENSE_1000_FFT_WINDOW_HANN,
    /*!< Hann Window */
    ADI_SENSE_1000_FFT_WINDOW_BLACKMAN_HARRIS,
    /*!< Blackman-Harris Window */
} ADI_SENSE_1000_FFT_WINDOW;

/*! ADSNS1000 FFT output format options
 *
 *  @note applicable only for FFT measurement modes
 */
typedef enum {
    ADI_SENSE_1000_FFT_OUTPUT_FULL = 0,
    /*!< N/2-Term Amplitude Response */
    ADI_SENSE_1000_FFT_OUTPUT_FULL_WITH_RAW,
    /**< N/2-Term Amplitude Response Plus N Raw ADC Samples */
    ADI_SENSE_1000_FFT_OUTPUT_MAX16,
    /*!< Bin-Number and Amplitude of 16 Highest Peaks of Amplitude Response */
} ADI_SENSE_1000_FFT_OUTPUT;

/*! ADSNS1000 I2C clock speed options
 *
 *  @note applicable only for I2C sensors
 */
typedef enum
{
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_I2C_CLOCK_SPEED_100K = 0,
    /*!< 100kHz I2C clock speed */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_I2C_CLOCK_SPEED_400K,
    /*!< 400kHz I2C clock speed */
} ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_I2C_CLOCK_SPEED;

/*! ADSNS1000 UART baud rate options
 *
 *  @note applicable only for UART sensors
 */
typedef enum
{
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_BAUD_RATE_115200 = 0,
    /*!< 115200 UART baud rate */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_BAUD_RATE_57600,
    /*!< 57600 UART baud rate */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_BAUD_RATE_38400,
    /*!< 38400 UART baud rate */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_BAUD_RATE_19200,
    /*!< 19200 UART baud rate */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_BAUD_RATE_9600,
    /*!< 9600 UART baud rate */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_BAUD_RATE_4800,
    /*!< 4800 UART baud rate */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_BAUD_RATE_2400,
    /*!< 2400 UART baud rate */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_BAUD_RATE_1200,
    /*!< 1200 UART baud rate */
} ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_BAUD_RATE;

/*! ADSNS1000 UART line config options
 *
 *  @note applicable only for UART sensors
 */
typedef enum
{
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_LINE_CONFIG_8N1 = 0,
    /*!< UART line control 8 data bits / no parity / 1 stop bit  */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_LINE_CONFIG_8N2,
    /*!< UART line control 8 data bits / no parity / 2 stop bits  */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_LINE_CONFIG_8N3,
    /*!< UART line control 8 data bits / no parity / 3 stop bits  */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_LINE_CONFIG_8E1,
    /*!< UART line control 8 data bits / even parity / 1 stop bit  */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_LINE_CONFIG_8E2,
    /*!< UART line control 8 data bits / even parity / 2 stop bits  */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_LINE_CONFIG_8E3,
    /*!< UART line control 8 data bits / even parity / 3 stop bits  */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_LINE_CONFIG_8O1,
    /*!< UART line control 8 data bits / odd parity / 1 stop bit  */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_LINE_CONFIG_8O2,
    /*!< UART line control 8 data bits / odd parity / 2 stop bits  */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_LINE_CONFIG_8O3,
    /*!< UART line control 8 data bits / odd parity / 3 stop bits  */
} ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_LINE_CONFIG;

/*! ADSNS1000 SPI mode options
 *
 *  @note applicable only for SPI sensors
 */
typedef enum
{
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_MODE_0 = 0,
    /*!< SPI mode 0 Clock Polarity = 0 Clock Phase = 0 */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_MODE_1,
    /*!< SPI mode 0 Clock Polarity = 0 Clock Phase = 1 */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_MODE_2,
    /*!< SPI mode 0 Clock Polarity = 1 Clock Phase = 0 */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_MODE_3,
    /*!< SPI mode 0 Clock Polarity = 1 Clock Phase = 1 */
} ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_MODE;

/*! ADSNS1000 SPI clock speed options
 *
 *  @note applicable only for SPI sensors
 */
typedef enum
{
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_13MHZ = 0,
    /*!< SPI Clock Speed configured to 13MHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_6_5MHZ,
    /*!< SPI Clock Speed configured to 6.5MHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_3_25MHZ,
    /*!< SPI Clock Speed configured to 3.25MHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_1_625MHZ,
    /*!< SPI Clock Speed configured to 1.625MHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_812KHZ,
    /*!< SPI Clock Speed configured to 812kHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_406KHZ,
    /*!< SPI Clock Speed configured to 406kHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_203KHZ,
    /*!< SPI Clock Speed configured to 203kHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_101KHZ,
    /*!< SPI Clock Speed configured to 101kHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_50KHZ,
    /*!< SPI Clock Speed configured to 50kHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_25KHZ,
    /*!< SPI Clock Speed configured to 25kHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_12KHZ,
    /*!< SPI Clock Speed configured to 12kHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_6KHZ,
    /*!< SPI Clock Speed configured to 6kHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_3KHZ,
    /*!< SPI Clock Speed configured to 3kHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_1_5KHZ,
    /*!< SPI Clock Speed configured to 1.5kHz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_793HZ,
    /*!< SPI Clock Speed configured to 793hz */
  ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK_396HZ,
    /*!< SPI Clock Speed configured to 396hz */
} ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK;

/*! ADSNS1000 Power Configuration options */
typedef struct {
    ADI_SENSE_1000_POWER_MODE        powerMode;
    /*!< Power mode selection */
} ADI_SENSE_1000_POWER_CONFIG;

/*! ADSNS1000 Multi-Cycle Configuration options
 *
 * @note required only when ADI_SENSE_1000_OPERATING_MODE_MULTICYCLE is selected
 *       as the operatingMode (@ref ADI_SENSE_1000_MEASUREMENT_CONFIG)
 */
typedef struct {
    uint32_t                         cyclesPerBurst;
    /*!< Number of cycles to complete for a single burst */
    uint32_t                         burstInterval;
    /*!< Interval, in seconds, between each successive burst of cycles */
} ADI_SENSE_1000_MULTICYCLE_CONFIG;

/*! ADSNS1000 Measurement Configuration options */
typedef struct {
    ADI_SENSE_1000_OPERATING_MODE    operatingMode;
    /*!< Operating mode - specifies how measurement cycles are scheduled */
    ADI_SENSE_1000_DATAREADY_MODE    dataReadyMode;
    /*!< Data read mode - specifies how output samples are stored for reading */
    ADI_SENSE_1000_CALIBRATION_MODE calibrationMode;
    /*!< Calibration mode - specifies if adc calibration is performed after a latch command */
    ADI_SENSE_1000_MULTICYCLE_CONFIG multiCycleConfig;
    /*!< Multi-Cycle configuration - specifies how bursts of measurement cycles
     *   are scheduled.  Applicable only when operatingMode is
     *   ADI_SENSE_1000_OPERATING_MODE_MULTICYCLE
     */
    uint32_t                         cycleInterval;
    /*!< Cycle interval - specifies the time interval between the start of each
     *   successive measurement cycle.  Applicable only when operatingMode is
     *   not ADI_SENSE_1000_OPERATING_MODE_SINGLECYCLE
     */
    ADI_SENSE_1000_CYCLE_TYPE        cycleType;
    /*!< Cycle type - specifies how the channel list is traversed with each
     *   conversion during the cycle.
     */
    float32_t                        externalRef1Value;
    /*!< Resistance/voltage value connected to external reference input #1.
     *   Applicable only if the selected reference type is
     *   ADI_SENSE_1000_ADC_REFERENCE_RESISTOR_EXTERNAL_1 or
     *   ADI_SENSE_1000_ADC_REFERENCE_VOLTAGE_EXTERNAL_1
     *   (see @ref ADI_SENSE_1000_ADC_REFERENCE_TYPE)
     */
    float32_t                        externalRef2Value;
    /*!< Resistance/voltage value connected to external reference input #2.
     *   Applicable only if the selected reference type is
     *   ADI_SENSE_1000_ADC_REFERENCE_RESISTOR_EXTERNAL_2 or
     *   ADI_SENSE_1000_ADC_REFERENCE_VOLTAGE_EXTERNAL_2
     *   (see @ref ADI_SENSE_1000_ADC_REFERENCE_TYPE)
     */    
    bool                             enableExternalFlash;
    /*!< Enable the use of external flash memory for storage of samples.
     */
} ADI_SENSE_1000_MEASUREMENT_CONFIG;

/*! ADSNS1000 ADC Excitation Current output configuration
 *
 *  @note applicable only to ADC analog sensor channels, and
 *        specific sensor types
 */
typedef struct {
    ADI_SENSE_1000_ADC_EXC_CURRENT              outputLevel;
    /*!< Excitation current output level */
    ADI_SENSE_1000_ADC_EXC_CURRENT_DIODE_RATIO  diodeRatio;
    /*!< Excitation current output diode ratio */
} ADI_SENSE_1000_ADC_EXC_CURRENT_CONFIG;

/*! ADSNS1000 ADC Filter configuration
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef struct {
    ADI_SENSE_1000_ADC_FILTER_TYPE   type;
    /*!< Filter type selection */
    uint32_t                         fs;
    /*!< Number of input samples to accumulate and average in the filter.
     *   Applicable only when type is ADI_SENSE_1000_ADC_FILTER_SINC4 */
} ADI_SENSE_1000_ADC_FILTER_CONFIG;

/*! ADSNS1000 ADC Referemce configuration
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef struct {
    ADI_SENSE_1000_ADC_REFERENCE_TYPE type;
    /*!< Filter type selection */
    bool_t                            disableBuffer;
    /*!< Option to disable internal ADC reference buffer */
} ADI_SENSE_1000_ADC_REFERENCE_CONFIG;

/*! ADSNS1000 ADC analog channel configuration details
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef struct {
    ADI_SENSE_1000_ADC_SENSOR_TYPE        sensor;
    /*!< Sensor type selection */
    ADI_SENSE_1000_ADC_GAIN               gain;
    /*!< ADC Gain selection */
    ADI_SENSE_1000_ADC_EXC_CURRENT_CONFIG current;
    /*!< ADC Excitation Current configuration */
    ADI_SENSE_1000_ADC_FILTER_CONFIG      filter;
    /*!< ADC Filter configuration */
    ADI_SENSE_1000_ADC_REFERENCE_CONFIG   reference;
    /*!< ADC Reference configuration */
    bool_t                                enableVbias;
    /*!< Option to enable internal ADC voltage bias */
} ADI_SENSE_1000_ADC_CHANNEL_CONFIG;

/*! ADSNS1000 digital sensor data encoding
 *
 *  @note applicable only to SPI and I2C digital sensor channels
 */
typedef enum
{
    ADI_SENSE_1000_DIGITAL_SENSOR_DATA_CODING_NONE,
    /**< None/Invalid - data format is ignored if coding is set to this value */
    ADI_SENSE_1000_DIGITAL_SENSOR_DATA_CODING_UNIPOLAR,
    /**< Unipolar - unsigned integer values */
    ADI_SENSE_1000_DIGITAL_SENSOR_DATA_CODING_TWOS_COMPLEMENT,
    /**< Twos-complement - signed integer values */
    ADI_SENSE_1000_DIGITAL_SENSOR_DATA_CODING_OFFSET_BINARY,
    /**< Offset Binary - used to represent signed values with unsigned integers,
     *   with the mid-range value representing 0 */
} ADI_SENSE_1000_DIGITAL_SENSOR_DATA_CODING;

/*! ADSNS1000 digital sensor data format configuration
 *
 *  @note applicable only to SPI and I2C digital sensor channels
 */
typedef struct {
    ADI_SENSE_1000_DIGITAL_SENSOR_DATA_CODING coding;
    /**< Data Encoding of Sensor Result */
    bool_t littleEndian;
    /**< Set as true if data format is little-endian, false otherwise */
    bool_t leftJustified;
    /**< Set as true if data is left-justified in the data frame, false otherwise */
    uint8_t frameLength;
    /**< Data frame length (number of bytes to read from the sensor) */
    uint8_t numDataBits;
    /**< Number of relevant data bits to extract from the data frame */
    uint8_t bitOffset;
    /**< Data bit offset, relative to data alignment within the data frame */
} ADI_SENSE_1000_DIGITAL_SENSOR_DATA_FORMAT;

/*! ADSNS1000 digital sensor command
 *
 *  @note applicable only to SPI and I2C digital sensor channels
 */
typedef struct {
    uint8_t command[ADI_SENSE_1000_SENSOR_COMMAND_MAX_LENGTH];
    /*!< Optional command bytes to send to the device */
    uint8_t commandLength;
    /*!< Number of valid command bytes.  Set to 0 if unused */
} ADI_SENSE_1000_DIGITAL_SENSOR_COMMAND;

/*! ADSNS1000 digital sensor calibration param
 *
 *  @note applicable only to digital sensor channels
 */
typedef struct {
    uint32_t calibrationParam;
    /*!< Independently established environmental variable used during calibration
     *   of a digital sensor. Used only if the sensor supports calibration
     *   and expects an environmental parameter
     */
    bool_t enableCalibrationParam;
    /*!< Allow Calibration_Parameter to be used during calibration of any digital sensor */
} ADI_SENSE_1000_DIGITAL_CALIBRATION_COMMAND;

/*! ADSNS1000 digital sensor communication config
 *
 *  @note applicable only to digital sensor channels
 */
typedef struct {
    bool_t  useCustomCommsConfig;
    /*!< Optional parameter to enable user digital communication settings */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_I2C_CLOCK_SPEED     i2cClockSpeed;
    /*!< Optional parameter to configure specific i2c speed for i2c sensor */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_MODE            spiMode;
    /*!< Optional parameter to configure specific spi mode for spi sensor */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_SPI_CLOCK           spiClock;
    /*!< Optional parameter to configure specific spi clock for spi sensor */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_BAUD_RATE      uartBaudRate;
    /*!< Optional parameter to configure specific uart baud rate for uart sensor */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMS_UART_LINE_CONFIG    uartLineConfig;
    /*!< Optional parameter to configure specific uart line control for uart sensor */
} ADI_SENSE_1000_DIGITAL_SENSOR_COMMS;

/*! ADSNS1000 I2C digital channel configuration details
 *
 *  @note applicable only to I2C digital sensor channels
 */
typedef struct {
    ADI_SENSE_1000_I2C_SENSOR_TYPE        sensor;
    /*!< Sensor type selection */
    uint32_t                              deviceAddress;
    /*!< I2C device address (7-bit) */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMAND configurationCommand;
    /*!< Optional configuration command to send to the device at start-up.
     *   A default configuration command will be used if this is not specified.
     *   Applicable only to specific I2C sensor types.
     */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMAND dataRequestCommand;
    /*!< Optional data request command to send to the device for each sample.
     *   A default data request command will be used if this is not specified.
     *   Applicable only to specific I2C sensor types.
     */
    ADI_SENSE_1000_DIGITAL_SENSOR_DATA_FORMAT dataFormat;
    /*!< Optional data format configuration to parse/extract data from the device.
     *   A default data format will be used if this is not specified.
     *   Applicable only to specific I2C sensor types
     */
    ADI_SENSE_1000_DIGITAL_CALIBRATION_COMMAND digitalCalibrationParam;
    /*!< This is used to supply an independently established environmental variable
     *   that must be used during calibration of a [digital] sensor which a)
     *   supports calibration and b) expects an environmental parameter.
     *   An example is a CO2 sensor, which may require the CO2 concentration level
     *   when performing a calibration
     */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMS configureComms;
    /*!< Optional configuration to setup a user communication config.
     *   A default configuration  will be used if this is not specified.
     *   Applicable only to specific I2C sensor types.
     */
} ADI_SENSE_1000_I2C_CHANNEL_CONFIG;

/*! ADSNS1000 SPI digital channel configuration details
 *
 *  @note applicable only to SPI digital sensor channels
 */
typedef struct {
    ADI_SENSE_1000_SPI_SENSOR_TYPE        sensor;
    /*!< Sensor type selection */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMAND configurationCommand;
    /*!< Optional configuration command to send to the device at start-up.
     *   A default configuration command will be used if this is not specified.
     *   Applicable only to specific SPI sensor types.
     */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMAND dataRequestCommand;
    /*!< Optional data request command to send to the device for each sample.
     *   A default data request command will be used if this is not specified.
     *   Applicable only to specific SPI sensor types.
     */
    ADI_SENSE_1000_DIGITAL_SENSOR_DATA_FORMAT dataFormat;
    /*!< Optional data format configuration to parse/extract data from the device.
     *   A default data format will be used if this is not specified.
     *   Applicable only to specific SPI sensor types
     */
    ADI_SENSE_1000_DIGITAL_CALIBRATION_COMMAND digitalCalibrationParam;
    /*!< This is used to supply an independently established environmental variable
     *   that must be used during calibration of a [digital] sensor which a)
     *   supports calibration and b) expects an environmental parameter.
     *   An example is a CO2 sensor, which may require the CO2 concentration level
     *   when performing a calibration
     */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMS configureComms;
    /*!< Optional configuration to setup a user communication config.
     *   A default configuration  will be used if this is not specified.
     *   Applicable only to specific SPI sensor types.
     */
} ADI_SENSE_1000_SPI_CHANNEL_CONFIG;

/*! ADSNS1000 UART digital channel configuration details
 *
 *  @note applicable only to UART digital sensor channels
 */
typedef struct {
    ADI_SENSE_1000_UART_SENSOR_TYPE        sensor;
    /*!< Sensor type selection */
    ADI_SENSE_1000_DIGITAL_CALIBRATION_COMMAND digitalCalibrationParam;
    /*!< This is used to supply an independently established environmental variable
     *   that must be used during calibration of a [digital] sensor which a)
     *   supports calibration and b) expects an environmental parameter.
     *   An example is a CO2 sensor, which may require the CO2 concentration level
     *   when performing a calibration
     */
    ADI_SENSE_1000_DIGITAL_SENSOR_COMMS configureComms;
    /*!< Optional configuration to setup a user communication config.
     *   A default configuration  will be used if this is not specified.
     *   Applicable only to specific UART sensor types.
     */
} ADI_SENSE_1000_UART_CHANNEL_CONFIG;

/*! ADSNS1000 Measurement Channel configuration details */
typedef struct {
    bool_t                               enableChannel;
    /*!< Option to enable this channel.  If set to false, all other fields
     *   are ignored and this channel will be omitted from measurement cycles
     */
    bool_t                               enableFFT;
    /*!< Option to include this channel in FFT mode measurements */
    bool_t                               disablePublishing;
    /*!< Option to disable publishing of data samples from this channel.  The
     *   channel may still be included in measurement cycles, but data samples
     *   obtained from this channel will not be published.  This is typically
     *   used for channels which are required only as a compensation reference
     *   for another channel (e.g. Cold-Junction Compensation channels).
     */
    ADI_SENSE_1000_CHANNEL_ID            compensationChannel;
    /*!< Optional compensation channel.  Set to ADI_SENSE_1000_CHANNEL_ID_NONE if not
     *   required.  Typically used for thermocouple sensors that require a
     *   separate measurement of the "cold-junction" temperature, which can be
     *   be provided by an RTD temperature sensor connected on a separate
     *   "compensation channel" */
    bool_t                               enableUnityLut;
    /*!< Option to choose the LUT for calculations, this implies that the
     * fundamental measurement for the sensor (typically mV or Ohms)
     * would be returned as the measurement result.
     */
    ADI_SENSE_1000_MEASUREMENT_UNIT      measurementUnit;
    /*!< Optional measurement unit selection for conversion results.  Applicable
     *   only for certain sensor types.  Set to
     *   ADI_SENSE_1000_MEASUREMENT_UNIT_DEFAULT if not applicable.
     */
    float32_t                            lowThreshold;
    /*!< Optional minimum threshold value for each processed sample, to be
     *   checked prior to publishing.  A channel ALERT condition is raised
     *   if the processed value is lower than this threshold.  Set to NaN
     *   if not required.
     */
    float32_t                            highThreshold;
    /*!< Optional maximum threshold value for each processed sample, to be
     *   checked prior to publishing.  A channel ALERT condition is raised
     *   if the processed value is higher than this threshold.  Set to NaN
     *   if not required.
     */
    float32_t                            offsetAdjustment;
    /*!< Optional offset adjustment value applied to each processed sample.
     *   Set to NaN or 0.0 if not required.
     */
    float32_t                            gainAdjustment;
    /*!< Optional gain adjustment value applied to each processed sample.
     *   Set to NaN or 1.0 if not required.
     */
    float32_t                            sensorParameter;
    /*!< Optional sensor parameter adjustment.
     *   Set to NaN or 0 if not required.
     */
    uint32_t                             measurementsPerCycle;
    /*!< The number of measurements to obtain from this channel within each
     *   cycle.  Each enabled channel is measured in turn, until the number of
     *   measurements requested for the channel has been reached.  A different
     *   number of measurements-per-cycle may be specified for each channel.
     */
    uint32_t                             cycleSkipCount;
    /*!< Optional number of cycles to skip, such that this channel is included
     *   in the sequence in only one of every (cycleSkipCount + 1) cycles that
     *   occur.  If set to 0 (default), this channel is included in every cycle;
     *   if set to 1, this channel is included in every 2nd cycle; if set to 2,
     *   this channel is included in every 3rd cycle, and so on.
     */
    uint32_t                             extraSettlingTime;
    /*!< A minimum settling time is applied internally for each channel, based
     *   on the sensor type.  However, additional settling time (microseconds)
     *   can optionally be specified.  Set to 0 if not required.
     */
    ADI_SENSE_1000_CHANNEL_PRIORITY      priority;
    /*!< By default, channels are arranged in the measurement sequence based on
     *   ascending order of channel ID.  However, a priority-level may be
     *   specified per channel to force a different ordering of the channels,
     *   with higher-priority channels appearing before lower-priority channels.
     *   Channels with equal priority are ordered by ascending order of channel
     *   ID.  Lower numbers indicate higher priority, with 0 being the highest.
     *   Set to 0 if not required.
     */
    union {
        ADI_SENSE_1000_ADC_CHANNEL_CONFIG adcChannelConfig;
        /*!< ADC channel configuration - applicable only to ADC channels */
        ADI_SENSE_1000_I2C_CHANNEL_CONFIG i2cChannelConfig;
        /*!< I2C channel configuration - applicable only to I2C channels */
        ADI_SENSE_1000_SPI_CHANNEL_CONFIG spiChannelConfig;
        /*!< SPI channel configuration - applicable only to SPI channels */
        ADI_SENSE_1000_UART_CHANNEL_CONFIG uartChannelConfig;
        /*!< UART channel configuration - applicable only to UART channels */
    };
    /*!< Only one of adcChannelConfig, i2cChannelConfig, spiChannelConfig or
     *   uartChannelConfig is required, depending on the channel designation
     *   (analog, I2C, SPI or UART)
     */
} ADI_SENSE_1000_CHANNEL_CONFIG;

/*! ADSNS1000 Diagnostics configuration details */
typedef struct {
    bool_t                                 disableGlobalDiag;
    /*!< Option to disable the following diagnostic checks on the ADC:
     *   - Reference Detection errors
     *   - Input under-/over-voltage errors
     *   - Calibration, Conversion and Saturation errors
     */
    bool_t                                 disableMeasurementDiag;
    /*!< Option to disable additional checks per measurement channel:
     *   - High/low threshold limit violation
     */
    ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS osdFrequency;
    /*!< Option to enable Open-Circuit Detection at a selected cycle interval */
} ADI_SENSE_1000_DIAGNOSTICS_CONFIG;

/*! ADSNS1000 FFT Measurement Mode configuration options */
typedef struct {
    ADI_SENSE_1000_FFT_MODE               mode;
    /*!< FFT sequence mode */
    ADI_SENSE_1000_FFT_SIZE               size;
    /*!< FFT size selection */
    ADI_SENSE_1000_FFT_WINDOW             window;
    /*!< FFT window type selection */
    ADI_SENSE_1000_FFT_OUTPUT             output;
    /*!< FFT output type selection */
} ADI_SENSE_1000_FFT_CONFIG;

/*! ADSNS1000 Device configuration details */
typedef struct {
    ADI_SENSE_1000_POWER_CONFIG           power;
    /*!< Power configuration details */
    ADI_SENSE_1000_MEASUREMENT_CONFIG     measurement;
    /*!< Measurement configuration details */
    ADI_SENSE_1000_DIAGNOSTICS_CONFIG     diagnostics;
    /*!< Diagnostics configuration details */
    ADI_SENSE_1000_FFT_CONFIG             fft;
    /*!< FFT configuration details */
    ADI_SENSE_1000_CHANNEL_CONFIG         channels[ADI_SENSE_1000_MAX_CHANNELS];
    /*!< Channel-specific configuration details */
} ADI_SENSE_1000_CONFIG;

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADI_SENSE_1000_CONFIG_H__ */
