// This program measures drain water level

#include "mbed.h"
#include "mDot.h"
//#include "WakeUp.h"

mDot* dot;
Ticker batteryvoltageMonitorTick;
 
AnalogIn batt_voltage(PA_1);
DigitalIn mt_pwron(PA_4);
DigitalIn mt_caught(PA_5);
DigitalOut transmitLED(LED1);

DigitalOut range(PB_1);     
AnalogIn ain(PB_0);


// Configuration variables
static std::string config_network_name = "SIGMA_LoRa";
static std::string config_network_pass = "sigma2013";
static uint8_t config_frequency_sub_band = 1;

//Global Variables
static volatile bool timeToReadBatteryVoltage = true;
static volatile bool dataChanged = true;

//Function prototypes
void batteryvoltageMonitorTock();
void printError(mDot* dot, int32_t returnCode);
void printVersion();
bool setFrequencySubBand(uint8_t subBand);
bool setNetworkName(const std::string name);
bool setNetworkPassphrase(const std::string passphrase);
bool setPower(uint8_t power);
bool setAck(uint8_t retries);
bool joinNetwork();
bool send(const std::string text);

int main()
{
//The low-power oscillator can be quite inaccurate on some targets
//this function calibrates it against the main clock
//WakeUp::calibrate();


    printf("\r\n\r\n");
    printf("=====================================\r\n");
    printf("Drain water sensor\r\n");
    printf("=====================================\r\n");
    printVersion();

    // get the mDot handle
    dot = mDot::getInstance();

    // reset to default config so we know what state we're in
    dot->resetNetworkSession();
    dot->resetConfig();
    if (dot->getPublicNetwork() == false) {
        dot->setPublicNetwork(true);
        printf("Public network = FLASE \r\n");
        }

    // set up the mDot with our network information
    setNetworkName(config_network_name);
    setNetworkPassphrase(config_network_pass);
    setFrequencySubBand(config_frequency_sub_band);
    setPower(14);    // Reduce latency for 868 units
    setAck(0);      // Disable ack for less latency
    
    float adc, volts, cm;
    
    while (!joinNetwork()) { wait(2); dot->resetNetworkSession(); }


    
    while (1) {
       char latestData[100];      

        range=true;                // turn sonic sensor on
        wait(0.02);                 // wait for it to turn on
        
        adc = ain.read();           // read sonic sensor as a float
        range=false;               // tuen sonic sensor off
        
        volts = adc * 3.3466f;      // convert to volts
        cm = volts / 0.00326f;      // 3.3V/1024=V/cm

    
         
 //      batteryvoltageMonitorTick.attach(&batteryvoltageMonitorTock, 0.15);   
 //       sprintf(latestData, "power on: %d, kill status: %d, battery voltage: %f", old_mt_pwron, old_mt_caught, (batt_voltage*3.3));
 //       printf("%s\r\n", latestData);
    
        printf("%8.4f adc %8.2fV %4.1f cm Battery: %8.1f V\r\n", adc, volts, cm, (batt_voltage*3.3));
        sprintf(latestData, "Distance:%8.0f, Battery: %8.0f", cm,(batt_voltage*3.3));
        send(latestData);
        
        
        wait(3); 
  
        //Set wakeup time for 10 seconds
        //WakeUp::set_ms(1000);
        
        //Enter deepsleep, the program won't go beyond this point until it is woken up
        //deepsleep();
    
    
    
    }
    
}
 
    


void printVersion()
{
    printf("%s\r\n\r\n", dot->getId().c_str());
}

bool setFrequencySubBand(uint8_t subBand)
{
    int32_t returnCode;
    printf("Setting frequency sub band to '%d'...\r\n", subBand);
    if ((returnCode = dot->setFrequencySubBand(subBand)) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setNetworkName(const std::string name)
{
    int32_t returnCode;
    printf("Setting network name to '%s'...\r\n", name.c_str());
    if ((returnCode = dot->setNetworkName(name)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setNetworkPassphrase(const std::string passphrase)
{
    int32_t returnCode;
    printf("Setting passphrase to '%s'...\r\n", passphrase.c_str());
    if ((returnCode = dot->setNetworkPassphrase(passphrase)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setPower(uint8_t power)
{
    int32_t returnCode;
    printf("Setting tx power to '%d'...\r\n", power);
    if ((returnCode = dot->setTxPower(power)) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    return true;
}


bool joinNetwork()
{
    int32_t returnCode;
    printf("\r\nJoining network...\r\n");
    if ((returnCode = dot->joinNetworkOnce()) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    printf("Network Joined!\r\n");
    return true;
}

bool setAck(uint8_t retries)
{
    int32_t returnCode;
    printf("Setting ack to '%d'...\r\n", retries);
    if ((returnCode = dot->setAck(retries)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool send(const std::string text)
{
    int32_t returnCode;
    uint32_t timeTillSend = dot->getNextTxMs();
    if (timeTillSend != 0) {
        printf("waiting %lu ms to send\r\n", timeTillSend);
        return false;
    }

    printf("Sending data...  ");
    std::vector<uint8_t> data(text.begin(), text.end());
    if ((returnCode = dot->send(data, 1)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    printf("Data sent!\r\n");
    return true;
}

void printError(mDot* dot, int32_t returnCode)
{
    std::string error = mDot::getReturnCodeString(returnCode) + " - " + dot->getLastError();
    printf("%s\r\n", error.c_str());
}

