// MUNIN Client library library
// mbed Team     -   15 April 2014
// Author: Przemyslaw Wirkus

#ifndef MUNIN_CIENT_LIBRARY_H
#define MUNIN_CIENT_LIBRARY_H

#include <string>
#include <vector>
#include <sstream>
#include "mbed.h"
#include "EthernetInterface.h"

namespace {
// Munin command(s):
const int MUNIN_BUFFER_SIZE = 256;
}

/** MuninClient class, used to fetch counters from Munin
*
* Example:
* @code

#include "munincli.h"
#include "dispBoB.h"

namespace
{
const char* MUNIN_SERVER_ADDRESS = "172.28.22.45";
const int MUNIN_SERVER_PORT = 4949;
}

namespace {
// Munin command(s):
char CMD_FETCH_USERS[] = "fetch users\n";
}


int main()
{
    MuninClient munin_lcd;
    munin_lcd.dhcp_connection();    // DHCP connection
    dispBoB bob_display(p28, p27, p26);

    while(true) {
        while (munin_lcd.socket_connection(MUNIN_SERVER_ADDRESS, MUNIN_SERVER_PORT) < 0) {
            wait(1);
        }

        munin_lcd.socket_recv();
        munin_lcd.socket_send(CMD_FETCH_USERS, sizeof(CMD_FETCH_USERS));
        munin_lcd.socket_recv_all();
        {
            const char *param = "usercount.value";
            const int counter_value = munin_lcd.get_param_value(param);
            bob_display.cls();
            bob_display.printf("% 6d\n", counter_value);
        }

        munin_lcd.socket_close();
        wait(3);
    }
}


* @endcode
*/
class MuninClient
{
public:
    /** Connect to DHCP
    */
    void dhcp_connection() {
        printf("DHCP: Connecting ... ");
        eth.init(); //Use DHCP
        eth.connect();
        printf("%s\r\n", eth.getIPAddress());
    }

    /** Create a dispBoB object defined on the I2C master bus
    *
    * @param server Munin server IP / name
    * @param port Munin server port
    */
    int socket_connection(const char *server, const int port) {
        printf("TCP: Connecting ... ");
        int ret = socket.connect(server, port) < 0;
        printf("[OK]\r\n");
        return ret;
    }

    /** Closes socket conenction to Munin server
    */
    void socket_close() {
        socket.close();
    }

    /** Recv data from Munin to buffer 'buf'
    */
    int socket_recv() {
        const int n = socket.receive(buf, sizeof(buf) - 1);
        if (n > 0) {
            buf[n] = '\0';
        }
        return n;
    }

    /** Recv data from Munin to buffer 'buf'. Used to fetch reply for Munin command (given counter name)
    */
    int socket_recv_all() {
        const int n = socket.receive_all(buf, sizeof(buf) - 1);
        if (n > 0) {
            buf[n] = '\0';
        }
        return n;
    }

    /** Send data using socket to Munin server. Used to send command to Munin server.
    * @param cmd Munin command string, must end with \n
    * @param cmd_size Munin command buffer size
    */
    int socket_send(char *cmd, const int cmd_size) {
        const int n = socket.send(cmd, cmd_size - 1);
        return n;
    }

    /** Explodes Munin command reply into individual counters divided by '\n'.
    * Function extract counter value from conunter wirh name param_name.
    * @param param_name counter name
    */
    int get_param_value(std::string param_name) {
        // Explode Munin command and creaste separate strings with counters
        std::vector<std::string> s = explode(std::string(buf), '\n');

        // For each counter name get its name and if name matches param_name extract and return counter value
        for (std::vector<std::string>::iterator it = s.begin(); it != s.end(); ++it) {
            std::string &param = *it;
            if (!param.compare(0, param_name.size(), param_name)) {
                std::vector<std::string> name_value = explode(param, ' ');
                if (name_value.size() == 2) {
                    const int val = atoi(name_value[1].c_str());
                    return val;
                }
            }
        }
        return 0;
    }

protected:
    EthernetInterface eth;
    TCPSocketConnection socket;
    char buf[MUNIN_BUFFER_SIZE];

    // String explode function
    std::vector<std::string> explode(std::string const &s, char delim) {
        std::vector<std::string> result;
        std::istringstream iss(s);
        for (std::string token; std::getline(iss, token, delim); ) {
            result.push_back(token);
        }
        return result;
    }
};

#endif
