#include "mbed.h"

// Trigger PIN
DigitalOut trigger(D9);
// Echo PIN
InterruptIn echo(D8);
// Ticker object to periodicly start the measurement
Ticker execute;
// Timeout to write LOW to Trigger pin after 10 microseconds
Timeout go_low;
// Timer to measure the time between the sonic burst goes out and in again
Timer get_time;
// Serial Interface to print data
Serial pc(USBTX, USBRX, 115200);

// Global Variables for the update status and the the distance in microseconds 
volatile float distance_us;
volatile bool update;

// Prototyping
void sendHigh(void);
void sendLow(void);
void startTimer(void);
void stopTimer(void);
float getDistance(void);
bool checkUpdate(void);
void startSensor(int freq);
void stopSensor(void);

// Function to start sending HIGH in periodic intervalls to TRIGGER PIN via a TICKER Object
void startSensor(int freq)
{
    // Check if frequency is in the allowed range
    if(freq > 16)   freq = 16;
    if(freq < 0)    freq = 0;
    // Calculate the time in seconds for the choosen frequency
    float freq_calc = 1.0f / freq;
    // Start the ticker and attach the function "sendHigh" to periodic write a HIGH on the trigger output
    execute.attach(&sendHigh, freq_calc);
}

// Function to stop sending HIGH to TRIGGER PIN
void stopSensor(void)
{
    // Detach the ticker function
    execute.detach();
}

// Check for an updatet status for the measurement
bool checkUpdate(void)
{
    if(update)
    {
        update = false;
        return true;
    }
    return false;
}

// Function to return the distance, should be called if "checkUpdate()" returns true
float getDistance(void)
{
    float distance_return = distance_us * 0.03432f / 2.0f;
    return distance_return;
}

// Called by interrupt - rising flank on ECHO PIN
void startTimer(void)
{
    // Start measure the time
    get_time.reset();
    get_time.start();
}

// Called by interrupt - falling flank on ECHO PIN
void stopTimer(void)
{
    // Stop measure the time
    get_time.stop();
    // Save time to variable
    distance_us = get_time.read_us();
    // Change state of update flag to true
    update = true;
}

// Called by TICKER object to periodicly write HIGH to TRIGGER PIN
void sendHigh(void)
{
    // Write HIGH to TRIGGER PIN
    trigger.write(1);
    // Attach function "sendLow()" to TIMEOUT [Interrupt called after x us] to go low again after 10 microseconds
    go_low.attach_us(&sendLow, 10);
}

// Called by TIMEOUT Object to set TRIPPER PIN LOW again
void sendLow(void)
{
    trigger.write(0);
}




int main(void)
{
    // Start condition for the update flag
    update = false;
    // If echo goes HIGH make a callback and call the function "startTimer()"
    echo.rise(&startTimer);
    // If echo goes LOW make a callback and call the function "stopTimer()"
    echo.fall(&stopTimer);
    // Start condition for Trigger Output - LOW
    trigger.write(0);
    // Welcome message
    pc.printf("\nHCSR04 - Interrupt\n");
    
    // Start measurement with 2 Hz [2times a second]
    startSensor(2);
    
    while(1)
    {
        // Check if there is anything to display
        if(checkUpdate())
        {
            // Print the actual distance to the Serial Interface
            pc.printf("Distance: %.2f \n", getDistance());
        }
    }
    
    
    return 0;
}