///////////////////////////////////////////////////////////////////////////////////////////////////////
// MAX32625PICO FIRMWARE FOR MAXIM AE11 EVBs
// Revision:    1.3
// Date:        09/18/2019
// Author:      Bill Smith
///////////////////////////////////////////////////////////////////////////////////////////////////////

#include "mbed.h"
#include "USBHID.h"
#include "max32625pico.h"
#include "i2c.h"

// configure VDDIOH to local 3.3V supply, set dipVio and swdVio to 1.8V supply
MAX32625PICO pico(MAX32625PICO::IOH_3V3, MAX32625PICO::VIO_1V8, MAX32625PICO::VIO_1V8);

// We declare a USBHID device.
// HID In/Out Reports are 64 Bytes long
// Vendor ID (VID):     0x0B6A  // Maxim Integrated
// Product ID (PID):    0x0001  // 
// Serial Number:       0x0001  //
USBHID hid(64, 64, 0x0B6A, 0x0003, 0x0001, true);

InterruptIn alertb1(P3_2);             // Interrupt PIN (INPUT)
InterruptIn alertb2(P3_3);             // Interrupt PIN (INPUT)

DigitalOut red_led(LED1);
DigitalOut green_led(LED2);
DigitalOut blue_led(LED3);

Timeout to_alertb1;                    // ALERTB1 timeout
Timeout to_alertb2;                    // ALERTB2 timeout

HID_REPORT send_report;
HID_REPORT recv_report;

//-----------------------------------------------------------------------------------------------------
// USB COMMANDS
//-----------------------------------------------------------------------------------------------------
// LED Command
#define WRITE_LED                   0x20    // Send data to the RGB LED

// I2C1 Commands
#define QUICK_WRITE_I2C1            0x30    // Send Device Address on I2C Master 1, look for ACK
#define READ_BYTE_I2C1              0x31    // Read Data Byte (8-bits) from Register of I2C Device
#define WRITE_BYTE_I2C1             0x32    // Write Data Byte (8-bits) to Register of I2C Device
#define READ_WORD_I2C1              0x33    // Read Data Word (16-bits) from Register of I2C Device
#define WRITE_WORD_I2C1             0x34    // Write Data Word (16-bits) to Register of I2C Device
#define WRITE_BLOCK_I2C1            0x3A    // Write Data Block to Register of I2C Device
#define SET_I2C_ADDR                0x3E    // Set I2C Slave Address
#define GET_I2C_ADDR                0x3F    // Get I2C Slave Address

// I2C2 Commands
//#define QUICK_WRITE_I2C2            0x35    // Send Device Address on I2C Master 2, look for ACK
//#define READ_BYTE_I2C2              0x36    // Read Data Byte (8-bits) from Register of I2C Device
//#define WRITE_BYTE_I2C2             0x37    // Write Data Byte (8-bits) to Register of I2C Device
//#define READ_WORD_I2C2              0x38    // Read Data Word (16-bits) from Register of I2C Device
//#define WRITE_WORD_I2C2             0x39    // Write Data Word (16-bits) to Register of I2C Device
//#define WRITE_BLOCK_I2C2            0x3B    // Write Data Block to Register of I2C Device

// AE11 Commands
#define GET_FUNC_REGS               0x40    // Return
#define GET_VENDOR_REGS             0x41    // Return
#define GET_INT_REGS                0x42    // Return
#define GET_MASK_REGS               0x43    // Return
#define GET_CTRL_REGS               0x44    // Return
#define GET_TAB1_REGS               0x45    // Return
#define GET_OTP_DATA_REGS           0x46    // Return
#define GET_OTP_TEST_CMD_REGS       0x47    // Return
#define GET_OTP_TEST_MODE_REGS      0x48    // Return
#define GET_OTP_TEST_OVRD_REGS      0x49    // Return
#define OTP_REGS_UNLOCK             0x4A    // Return 
#define OTP_REGS_LOCK               0x4B    // Return     

// INTERRUPT Commands
#define ALERTB1_ASSERTED            0xF0    // ALERTB1 Interrupt
#define CLEAR_INTS                  0xF2    // Clear All Interrupts

// LED CONSTANTS
#define LEDS_OFF        0x00
#define RED             0x01
#define GREEN           0x02
#define BLUE            0x03
#define YELLOW          0x04

// LOGICAL CONSTANTS
#define OFF             0x00
#define ON              0x01

char FUNC_REGS[63] = {  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
                        0x0A, 0x0B, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 
                        0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2E, 0x2F, 
                        0x30, 0x50, 0x51, 0x70, 0x71, 0x74, 0x75, 0x76, 0x77, 0x78, 
                        0x79, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 
                        0x89, 0x8A, 0x8B };
                        
char OTP_DATA_REGS[36] = {  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
                            0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F, 0xA0, 0xA1, 0xA2, 0xA3,
                            0xA4, 0xA5, 0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD,
                            0xAE, 0xAF, 0xB0, 0xB1, 0xB2, 0xB3 };

char OTP_TEST_CMD_REGS[16] = {  0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 
                                0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF };
                                
char OTP_TEST_MODE_REGS[23] = { 0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9,
                                0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0, 0xE1, 0xE2, 0xE3,
                                0xE4, 0xE5, 0xE6 };
                                
char OTP_TEST_OVRD_REGS[13] = { 0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9,
                                0xFA, 0xFB, 0xFC };
                     
char VENDOR_REGS[12] = {    0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
                            0x8A, 0x8B };

char INT_REGS[8] = {    0x10, 0x11, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x8B };

char MASK_REGS[5] = {   0x12, 0x13, 0x14, 0x15, 0x16 };

char CTRL_REGS[5] = {   0x18, 0x19, 0x1A, 0x1B, 0x1C };

char TAB1_REGS[29] = {  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 
                        0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x80, 0x81, 0x82, 
                        0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A, 0x8B };

//-----------------------------------------------------------------------------------------------------
// VARIABLES
//-----------------------------------------------------------------------------------------------------
char I2C1_RXDATA[40];
//char I2C2_RXDATA[40];
char I2C1_TXDATA[40];
//char I2C2_TXDATA[40];
static char ALERTB1_STATE = 0x00;              // ALERTB1 PIN STATE
static bool ALERTB1_CHANGED = false;
static char I2C_ADDR = 0x00;


//-----------------------------------------------------------------------------------------------------
// ALERTB1 INTERRUPT SERVICE ROUTINE
//-----------------------------------------------------------------------------------------------------
void test_alertb1_low()
{
    if(alertb1.read() == 0)
    {
        ALERTB1_STATE = 0x01;  // Capture the state of the ALERTB1 pin
        ALERTB1_CHANGED = true;
    }
} 

void test_alertb1_high()
{
    if(alertb1.read() == 1)
    {
        ALERTB1_STATE = 0x00;  // Capture the state of the ALERTB1 pin
        ALERTB1_CHANGED = true;
    }
} 

void ALERTB1_FALL_ISR()
{
    to_alertb1.attach(test_alertb1_low, 0.1);
}

void ALERTB1_RISE_ISR()
{
    to_alertb1.attach(test_alertb1_high, 0.1);
}

void StartUp_LEDs()
{
    red_led = 0;    // Turn ON RED LED
    wait(0.3);
    red_led = 1;    // Turn OFF RED LED
    green_led = 0;  // Turn ON GREEN LED
    wait(0.3);
    green_led = 1;  // Turn OFF GREEN LED
    blue_led = 0;   // Turn ON BLUE LED 
    wait(0.3);   
    blue_led = 1;   // Turn OFF BLUE LED   
}


// main() runs in its own thread in the OS
int main()
{
    send_report.length = 64;
    recv_report.length = 64;
    
    StartUp_LEDs(); // Cycle LEDs to show startup
    
    Init_I2C();     // Make sure we can talk
    
    alertb1.fall(&ALERTB1_FALL_ISR);          // Attaches ALERTB1 to the ISR
    alertb1.rise(&ALERTB1_RISE_ISR);
    
    while (true) 
    {
        if(ALERTB1_CHANGED)
        {
            send_report.data[0] = ALERTB1_ASSERTED;        // Echo Command
            send_report.data[1] = ALERTB1_STATE;           // Echo INTB State
            
            for(int i=0;i<8;i++)
            {
                send_report.data[2 + i] = I2C_Read_Byte(0x00, I2C_ADDR, INT_REGS[i]);       // Slave Address must be set!              
            }
                    
            ALERTB1_CHANGED = false;                    // Clear the INTB State Changed Flag
            hid.sendNB(&send_report);                   // Send the report (Non-blocking)
        }
        
        if(hid.readNB(&recv_report)) 
        {
            switch(recv_report.data[0])
            {
            //-----------------------------------------------------------------------------------------------------------------
                case WRITE_LED:             // COMMAND [0x20] - WRITE TO RGB LED
            //-----------------------------------------------------------------------------------------------------------------
                    switch(recv_report.data[1])
                    {
                        case LEDS_OFF:
                            red_led = 1;    // Turn OFF RED LED
                            green_led = 1;  // Turn OFF GREEN LED
                            blue_led = 1;   // Turn OFF BLUE LED
                            break;
                        case RED:
                            if(recv_report.data[2] == 1){red_led = 0;} else {red_led = 1;}
                            break;
                        case GREEN:
                            if(recv_report.data[2] == 1){green_led = 0;} else {green_led = 1;}
                            break; 
                        case BLUE:
                            if(recv_report.data[2] == 1){blue_led = 0;} else {blue_led = 1;}
                            break;
                        default:
                            break;
                    }// End Switch (RD1)
                        
                    send_report.data[0] = recv_report.data[0];                              // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Subcommand 1
                    send_report.data[2] = recv_report.data[2];                              // Echo Subcommand 2
                    break;
            //-----------------------------------------------------------------------------------------------------------------
              case QUICK_WRITE_I2C1:        // COMMAND [0x30] - QUICK WRITE TO I2C1
            //-----------------------------------------------------------------------------------------------------------------                    
                    send_report.data[0] = QUICK_WRITE_I2C1;                                 // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Subcommand                   
                    send_report.data[2] = I2C_Quick_Write(0x00, recv_report.data[1]);       // Echo ACK/NACK Status
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case READ_BYTE_I2C1:        // COMMAND [0x31] - BYTE READ FROM I2C1
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = READ_BYTE_I2C1;                                   // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Device Address
                    send_report.data[2] = recv_report.data[2];                              // Echo Register 
                    send_report.data[3] = I2C_Read_Byte(0x00, recv_report.data[1], recv_report.data[2]);   // Return Data  
                    break;
            //-----------------------------------------------------------------------------------------------------------------
               case WRITE_BYTE_I2C1:        // COMMAND [0x32] - BYTE WRITE TO I2C1 
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = WRITE_BYTE_I2C1;                                  // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Device Address                   
                    send_report.data[2] = recv_report.data[2];                              // Echo Register
                    send_report.data[3] = recv_report.data[3];                              // Echo Data
                    send_report.data[4] = I2C_Write_Register(0x00, recv_report.data[1], recv_report.data[2], recv_report.data[3]);                   
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case READ_WORD_I2C1:        // COMMAND [0x33] - READ WORD FROM I2C1
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = READ_WORD_I2C1;                                   // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Device Address
                    send_report.data[2] = recv_report.data[2];                              // Echo Register 
                    
                    unsigned short RX1 = I2C_Read_Word(0x00, recv_report.data[1], recv_report.data[2]);   // Read Device
                    
                    send_report.data[3] = (char)((RX1 & 0xFF00) >> 8);                       // Return Data High Byte
                    send_report.data[4] = (char)(RX1 & 0x00FF);                              // Return Data Low Byte   
                    break;
            //-----------------------------------------------------------------------------------------------------------------
               case WRITE_WORD_I2C1:        // COMMAND [0x34] - WORD WRITE TO I2C1 
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = WRITE_WORD_I2C1;                                  // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Device Address                   
                    send_report.data[2] = recv_report.data[2];                              // Echo Register
                    send_report.data[3] = recv_report.data[3];                              // Echo HBData
                    send_report.data[4] = recv_report.data[4];                              // Echo LBData
                    send_report.data[5] = I2C_Write_Word(0x00, recv_report.data[1], recv_report.data[2], recv_report.data[3], recv_report.data[4]);                    
                    break;

            //-----------------------------------------------------------------------------------------------------------------
               case WRITE_BLOCK_I2C1:        // COMMAND [0x3A] - WRITE A BLOCK of DATA (1 to 10 bytes) TO I2C1 
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = WRITE_BLOCK_I2C1;                                 // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Device Address                   
                    send_report.data[2] = recv_report.data[2];                              // Echo Register
                    send_report.data[3] = recv_report.data[3];                              // Echo Number of Data Bytes

                    I2C1_TXDATA[0] = recv_report.data[2];                                   // Register Address
                    
                    for(int n = 0; n < recv_report.data[3]; n++)
                    {
                        I2C1_TXDATA[1 + n] = recv_report.data[4 + n];                       // Load TX Buffer with Data
                    }
 
                    I2C_Write_Block(0x00, recv_report.data[1], recv_report.data[3]);
                 
                    break;

            //-----------------------------------------------------------------------------------------------------------------
              case SET_I2C_ADDR:           // COMMAND [0x3E] - ASSIGN I2C DEVICE ADDRESS FOR SLAVE OPERATIONS
            //-----------------------------------------------------------------------------------------------------------------                    
                    send_report.data[0] = SET_I2C_ADDR;                                     // Echo Command
                    I2C_ADDR = recv_report.data[1];                                         // Assign Slave Address
                    send_report.data[1] = recv_report.data[1];                              // Echo Subcommand                   
                    break;
            //-----------------------------------------------------------------------------------------------------------------
              case GET_I2C_ADDR:           // COMMAND [0x3F] - ASSIGN I2C DEVICE ADDRESS FOR SLAVE OPERATIONS
            //-----------------------------------------------------------------------------------------------------------------                    
                    send_report.data[0] = GET_I2C_ADDR;                                     // Echo Command
                    send_report.data[1] = I2C_ADDR;                                         // Return Slave Address                  
                    break;
/*
            //-----------------------------------------------------------------------------------------------------------------
              case QUICK_WRITE_I2C2:        // COMMAND [0x35] - QUICK WRITE TO I2C2
            //-----------------------------------------------------------------------------------------------------------------                    
                    send_report.data[0] = QUICK_WRITE_I2C2;                                 // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Subcommand                   
                    send_report.data[2] = I2C_Quick_Write(0x01, recv_report.data[1]);       // Echo ACK/NACK Status
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case READ_BYTE_I2C2:        // COMMAND [0x36] - BYTE READ FROM I2C1
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = READ_BYTE_I2C2;                                   // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Device Address
                    send_report.data[2] = recv_report.data[2];                              // Echo Register 
                    send_report.data[3] = I2C_Read_Byte(0x01, recv_report.data[1], recv_report.data[2]);   // Return Data  
                    break;
            //-----------------------------------------------------------------------------------------------------------------
               case WRITE_BYTE_I2C2:        // COMMAND [0x37] - BYTE WRITE TO I2C2 
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = WRITE_BYTE_I2C2;                                  // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Device Address                   
                    send_report.data[2] = recv_report.data[2];                              // Echo Register
                    send_report.data[3] = recv_report.data[3];                              // Echo Data
                    send_report.data[4] = I2C_Write_Register(0x01, recv_report.data[1], recv_report.data[2], recv_report.data[3]);                   
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case READ_WORD_I2C2:        // COMMAND [0x38] - READ WORD FROM I2C2
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = READ_WORD_I2C2;                                   // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Device Address
                    send_report.data[2] = recv_report.data[2];                              // Echo Register 
                    
                    unsigned short RX2 = I2C_Read_Word(0x01, recv_report.data[1], recv_report.data[2]);   // Read Device
                    
                    send_report.data[3] = (char)((RX2 & 0xFF00) >> 8);                       // Return Data High Byte
                    send_report.data[4] = (char)(RX2 & 0x00FF);                              // Return Data Low Byte   
                    break;
            //-----------------------------------------------------------------------------------------------------------------
               case WRITE_WORD_I2C2:        // COMMAND [0x39] - WORD WRITE TO I2C2 
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = WRITE_WORD_I2C2;                                  // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Device Address                   
                    send_report.data[2] = recv_report.data[2];                              // Echo Register
                    send_report.data[3] = recv_report.data[3];                              // Echo HBData
                    send_report.data[4] = recv_report.data[4];                              // Echo LBData
                    send_report.data[5] = I2C_Write_Word(0x01, recv_report.data[1], recv_report.data[2], recv_report.data[3], recv_report.data[4]);                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
               case WRITE_BLOCK_I2C2:        // COMMAND [0x3B] - WRITE A BLOCK of DATA (1 to 10 bytes) TO I2C2 
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = WRITE_BLOCK_I2C2;                                 // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Device Address                   
                    send_report.data[2] = recv_report.data[2];                              // Echo Register
                    send_report.data[3] = recv_report.data[3];                              // Echo Number of Data Bytes

                    I2C2_TXDATA[0] = recv_report.data[2];                                   // Register Address
                    
                    for(int n = 0; n < recv_report.data[3]; n++)
                    {
                        I2C2_TXDATA[1 + n] = recv_report.data[4 + n];                       // Load TX Buffer with Data
                    }
                    
                    I2C_Write_Block(0x01, recv_report.data[1], recv_report.data[3]);
                 
                    break;
*/
            //-----------------------------------------------------------------------------------------------------------------
                case GET_FUNC_REGS:             // COMMAND [0x40] - BLOCK READ OF MAX25430 FUNCTIONAL REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = GET_FUNC_REGS;                                    // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    for(int i=0;i<63;i++)
                    {
                        send_report.data[2 + i] = I2C_Read_Byte(0x00, recv_report.data[1], FUNC_REGS[i]);              
                    }
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case GET_VENDOR_REGS:          // COMMAND [0x41] - BLOCK READ OF MAX25430 VENDOR REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = GET_VENDOR_REGS;                                  // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    for(int i=0;i<12;i++)
                    {
                        send_report.data[2 + i] = I2C_Read_Byte(0x00, recv_report.data[1], VENDOR_REGS[i]);              
                    }
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case GET_INT_REGS:          // COMMAND [0x42] - BLOCK READ OF MAX25430 INTERRUPT REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = GET_INT_REGS;                                     // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    for(int i=0;i<8;i++)
                    {
                        send_report.data[2 + i] = I2C_Read_Byte(0x00, recv_report.data[1], INT_REGS[i]);              
                    }
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case GET_MASK_REGS:          // COMMAND [0x43] - BLOCK READ OF MAX25430 MASK REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = GET_MASK_REGS;                                    // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    for(int i=0;i<5;i++)
                    {
                        send_report.data[2 + i] = I2C_Read_Byte(0x00, recv_report.data[1], MASK_REGS[i]);              
                    }
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case GET_CTRL_REGS:          // COMMAND [0x44] - BLOCK READ OF MAX25430 TCPC CONTROL REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = GET_CTRL_REGS;                                    // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    for(int i=0;i<5;i++)
                    {
                        send_report.data[2 + i] = I2C_Read_Byte(0x00, recv_report.data[1], CTRL_REGS[i]);              
                    }
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case GET_TAB1_REGS:          // COMMAND [0x45] - BLOCK READ OF MAX25430 TAB1 REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = GET_TAB1_REGS;                                    // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    for(int i=0;i<29;i++)
                    {
                        send_report.data[2 + i] = I2C_Read_Byte(0x00, recv_report.data[1], TAB1_REGS[i]);              
                    }
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case GET_OTP_DATA_REGS:     // COMMAND [0x46] - BLOCK READ OF MAX25430 OTP DATA REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = GET_OTP_DATA_REGS;                                // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    for(int i=0;i<36;i++)
                    {
                        send_report.data[2 + i] = I2C_Read_Byte(0x00, recv_report.data[1], OTP_DATA_REGS[i]);              
                    }
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case GET_OTP_TEST_CMD_REGS:     // COMMAND [0x47] - BLOCK READ OF MAX25430 OTP TEST COMMAND REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = GET_OTP_TEST_CMD_REGS;                            // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    for(int i=0;i<16;i++)
                    {
                        send_report.data[2 + i] = I2C_Read_Byte(0x00, recv_report.data[1], OTP_TEST_CMD_REGS[i]);              
                    }
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case GET_OTP_TEST_MODE_REGS:    // COMMAND [0x48] - BLOCK READ OF MAX25430 OTP TEST MODE REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = GET_OTP_TEST_MODE_REGS;                           // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    for(int i=0;i<23;i++)
                    {
                        send_report.data[2 + i] = I2C_Read_Byte(0x00, recv_report.data[1], OTP_TEST_MODE_REGS[i]);              
                    }
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case GET_OTP_TEST_OVRD_REGS:    // COMMAND [0x49] - BLOCK READ OF MAX25430 OTP TEST OVERRIDE REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = GET_OTP_TEST_OVRD_REGS;                           // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    for(int i=0;i<13;i++)
                    {
                        send_report.data[2 + i] = I2C_Read_Byte(0x00, recv_report.data[1], OTP_TEST_OVRD_REGS[i]);              
                    }
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case OTP_REGS_UNLOCK:          // COMMAND [0x4A] - UNLOCK MAX25430 OTP REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = OTP_REGS_UNLOCK;                                  // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    I2C_Write_Register(0x00, recv_report.data[1], 0xC0, 0x12);
                    I2C_Write_Register(0x00, recv_report.data[1], 0xC0, 0x34);
                    I2C_Write_Register(0x00, recv_report.data[1], 0xC0, 0xFA);
                    I2C_Write_Register(0x00, recv_report.data[1], 0xC0, 0xDB);              
                   
                    send_report.data[2] = 0x01;                                             // Task Complete
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case OTP_REGS_LOCK:          // COMMAND [0x4B] - LOCK MAX25430 OTP REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = OTP_REGS_LOCK;                                    // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    I2C_Write_Register(0x00, recv_report.data[1], 0xC0, 0x00);
                    I2C_Write_Register(0x00, recv_report.data[1], 0xC0, 0x00);
                    I2C_Write_Register(0x00, recv_report.data[1], 0xC0, 0x00);
                    I2C_Write_Register(0x00, recv_report.data[1], 0xC0, 0x00);              
                   
                    send_report.data[2] = 0x01;                                             // Task Complete
                    
                    break;    
            //-----------------------------------------------------------------------------------------------------------------
                case CLEAR_INTS:          // COMMAND [0xF2] - CLEAR ALL MAX25430 INTERRUPT REGISTERS
            //-----------------------------------------------------------------------------------------------------------------
                    send_report.data[0] = CLEAR_INTS;                                       // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo I2C Device Address
                    
                    I2C_Write_Register(0x00, recv_report.data[1], 0x10, 0xFF);              // ALERT STATUS LB              
                    I2C_Write_Register(0x00, recv_report.data[1], 0x11, 0xA7);              // ALERT STATUS HB
                    I2C_Write_Register(0x00, recv_report.data[1], 0x1F, 0xFF);              // FAULT STATUS
                    
                    send_report.data[2] = 0x55;                                             // Task Complete
                    
                    break;
                          
                default:
                    break;
            }// End Switch - Command Interpretor
 
            //Send the report
            hid.send(&send_report);
        }// End If - HID Message
    }// End Main While Loop
}// End Main