///////////////////////////////////////////////////////////////////////////////////////////////////////
// MAX32630FTHR FIRMWARE FOR TESTING I/Os
// Revision:    1.0.0
// Date:        11/05/2019
// Author:      Bill Smith
//
// 11/05/2019: Initial Code
///////////////////////////////////////////////////////////////////////////////////////////////////////

#include "mbed.h"
#include "max32630fthr.h"
#include "USBHID.h"


MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3);

// We declare a USBHID device.
// HID In/Out Reports are 64 Bytes long
// Vendor ID (VID):     0x0B6A  // Maxim Integrated
// Product ID (PID):    0x0001  // 
// Serial Number:       0x0001  //
USBHID hid(64, 64, 0x0B6A, 0x0002, 0x0001, true);

DigitalOut red_led(LED1);
DigitalOut green_led(LED2);
DigitalOut blue_led(LED3);

DigitalOut PIN27(P3_5);     // SCL_1
DigitalOut PIN28(P3_4);     // SDA_1
DigitalOut PIN9(P5_7);      // SDA_2
DigitalOut PIN10(P6_0);     // SCL_2

HID_REPORT send_report;
HID_REPORT recv_report;

//-----------------------------------------------------------------------------------------------------
// USB COMMANDS
//-----------------------------------------------------------------------------------------------------

// LED Command
#define WRITE_LED                   0x20    // Send data to the RGB LED


// Commands
#define TOGGLE_OFF                  0x30    // 
#define TOGGLE_SCL1                 0x31    // 
#define TOGGLE_SDA1                 0x32    // 
#define TOGGLE_SCL2                 0x33    // 
#define TOGGLE_SDA2                 0x34    // 
#define TOGGLE_ALERTB1              0x35    // 
#define TOGGLE_ALERTB2              0x36    // 

// LED CONSTANTS
#define LEDS_OFF        0x00
#define RED             0x01
#define GREEN           0x02
#define BLUE            0x03
#define YELLOW          0x04

// LOGICAL CONSTANTS
#define OFF             0x00
#define SCL1            0x01
#define SDA1            0x02
#define SCL2            0x03
#define SDA2            0x04


//-----------------------------------------------------------------------------------------------------
// VARIABLES
//-----------------------------------------------------------------------------------------------------
static char TOGGLE_STATE = 0x00;


void StartUp_LEDs()
{
    red_led = 0;    // Turn ON RED LED
    wait(0.3);
    red_led = 1;    // Turn OFF RED LED
    green_led = 0;  // Turn ON GREEN LED
    wait(0.3);
    green_led = 1;  // Turn OFF GREEN LED
    blue_led = 0;   // Turn ON BLUE LED 
    wait(0.3);   
    blue_led = 1;   // Turn OFF BLUE LED   
}


// main() runs in its own thread in the OS
int main()
{
    send_report.length = 64;
    recv_report.length = 64;
    
    StartUp_LEDs(); // Cycle LEDs to show startup
    
    while (true) 
    {
        switch(TOGGLE_STATE)
        {
            case OFF:
                PIN27 = 0;  // SCL_1 = OFF
                PIN28 = 0;  // SDA_1 = OFF
                PIN10 = 0;  // SCL_2 = OFF
                PIN9 = 0;   // SDA_2 = OFF
            case SCL1:
                PIN27 = !PIN27;
                wait(0.1); // TOGGLE SCL_1 EACH PASS
                break;
            case SDA1:
                PIN28 = !PIN28;
                wait(0.1); // TOGGLE SDA_1 EACH PASS
                break;
            case SCL2:
                PIN10 = !PIN10;
                wait(0.1); // TOGGLE SCL_2 EACH PASS
                break;
            case SDA2:
                PIN9 = !PIN9;
                wait(0.1); // TOGGLE SDA_2 EACH PASS
                break;
            default:
                break;
        }// END SWITCH
        
        if(hid.readNB(&recv_report)) 
        {
            switch(recv_report.data[0])
            {
            //-----------------------------------------------------------------------------------------------------------------
                case WRITE_LED:             // COMMAND [0x20] - WRITE TO RGB LED
            //-----------------------------------------------------------------------------------------------------------------
                    switch(recv_report.data[1])
                    {
                        case LEDS_OFF:
                            red_led = 1;    // Turn OFF RED LED
                            green_led = 1;  // Turn OFF GREEN LED
                            blue_led = 1;   // Turn OFF BLUE LED
                            break;
                        case RED:
                            if(recv_report.data[2] == 1){red_led = 0;} else {red_led = 1;}
                            break;
                        case GREEN:
                            if(recv_report.data[2] == 1){green_led = 0;} else {green_led = 1;}
                            break; 
                        case BLUE:
                            if(recv_report.data[2] == 1){blue_led = 0;} else {blue_led = 1;}
                            break;
                        default:
                            break;
                    }// End Switch (RD1)
                        
                    send_report.data[0] = recv_report.data[0];                              // Echo Command
                    send_report.data[1] = recv_report.data[1];                              // Echo Subcommand 1
                    send_report.data[2] = recv_report.data[2];                              // Echo Subcommand 2
                    break;
            //-----------------------------------------------------------------------------------------------------------------
              case TOGGLE_OFF:          // COMMAND [0x30] - TOGGLE OFF
            //-----------------------------------------------------------------------------------------------------------------                    
                    send_report.data[0] = TOGGLE_OFF;                                       // Echo Command
                    TOGGLE_STATE = OFF;
                    
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case TOGGLE_SCL1:       // COMMAND [0x31] - TOGGLE SCL1
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = TOGGLE_SCL1;                                      // Echo Command
                    TOGGLE_STATE = SCL1;
                     
                    break;
            //-----------------------------------------------------------------------------------------------------------------
               case TOGGLE_SDA1:        // COMMAND [0x32] - TOGGLE SDA1 
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = TOGGLE_SDA1;                                      // Echo Command
                    TOGGLE_STATE = SDA1;
                                                        
                    break;
            //-----------------------------------------------------------------------------------------------------------------
                case TOGGLE_SCL2:       // COMMAND [0x33] - TOGGLE SCL2
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = TOGGLE_SCL2;                                      // Echo Command
                    TOGGLE_STATE = SCL2;
                                       
                    break;
            //-----------------------------------------------------------------------------------------------------------------
               case TOGGLE_SDA2:        // COMMAND [0x34] - TOGGLE SDA2 
            //-----------------------------------------------------------------------------------------------------------------                     
                    send_report.data[0] = TOGGLE_SDA2;                                      // Echo Command
                    TOGGLE_STATE = SDA2;                   
                                       
                    break;

            }// End Switch - Command Interpretor
 
            //Send the report
            hid.send(&send_report);
        }// End If - HID Message
    }// End Main While Loop
}// End Main