#ifndef MBED_I2S_H
#define MBED_I2S_H

#include "mbed.h"

/** An I2S class
 * It uses the I2S peripheral of the LPC1768
 * @author Pinski1
 * More details about the function goes here
 * 
 * An example:
 * @code
 * #include "I2S.h"
 * #include "mbed.h"
 * 
 * I2S audioInterface (p7, p6, p5, p30, p29, p8);
 *
 * void main()
 * {
 * 		audioInterface.setClocks(,,,);
 * 		audioInterface.setTx(32,9600000,true);
 * 		audioInterface.setRx(32,9600000,true);
 *
 * 		// now print out registers out to prove it's set up correctly
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * }
 * @endcode
 */
class I2S
{
	private:
		uint32_t masterClock;
		uint32_t peripheralClock;
		// fifoRx 0x400A800C
		// LPC_I2S->I2SRXFIFO
        // fifoTx 0x400A8008
		// LPC_I2S->I2STXFIFO
		
    public:
        /** Creates an I2S object to control the I2S peripheral.
         *
         * @param bitTx The bit clock pin used for transmitting I2S.
         * @param wdTx The word clock pin used for transmitting I2S.
         * @param daTx The data pin used for transmitting I2S.
         * @param bitRx The bit clock pin used for recieving I2S.
         * @param wdRx The word clock pin used for recieving I2S.
         * @param daRx The data clock pin used for recieving I2S.
         */
        I2S(PinName, PinName, PinName, PinName, PinName, PinName);
        
        /** Destructs the I2S object and turns off the I2S peripheral. */
        ~I2S();
        
        /** Sets up the clocks for the I2S peripheral.
         * Currents sets up both recieve and transmit channels identically.
		 *
         * @param x The X divider for the perphieral clock which sets the I2S master clock.
         * @param y The Y divider for the perphieral clock which sets the I2S master clock.
         * @param divider The CPU clock divider that gives the peripheral clock.
         */
        void setClocks(uint8_t x, uint8_t y, uint8_t divider);
        
        /** Sets the bit clock and word clocks for the channels.
		 *
         * @param resolution The bits per sample (8, 16 or 32).
         * @param rate The samples per second (16kHz to 96kHz).
         * @param stereo Whether the stream was stereo or mono.
         */
        void setTx(uint16_t resolution, uint16_t rate, bool stereo);
        void setRx(uint16_t resolution, uint16_t rate, bool stereo);

		/** Mutes or unmutes the transmit I2S channel. */
        void muteTx(void);
		/** Mutes or unmutes the receive I2S channel. */
        void muteRx(void);
		
		/** Resets the transmit I2S channel. */
        void resetTx(void);
		/** Resets the receive I2S channel. */
        void resetRx(void);
		
		/** Sets the I2S mode of the transmit channel.
		 *
		 * @param mode Sets the typical mode.
		 */
		void setTxMode(uint8_t mode);
		/** Sets the I2S mode of the receive channel.
		 *
		 * @param mode Sets the typical mode.
		 */
		void setRxMode(uint8_t mode);
        
		/** Sets up the Interrupt Requests
		 *
		 * @param rxInterrupt
		 * @param txInterrupt
		 * @param rxDepth
		 * @param txDepth
		 */
		void setIRQ(bool rxInterrupt, bool txInterrupt, uint8_t rxDepth, uint8_t txDepth);
		
		/** Sets up the DMA requests
		 *
		 * @param rxDMA
		 * @param txDMA
		 * @param rxDepth
		 * @param txDepth
		 */
		void setDMA1(bool rxDMA, bool txDMA, uint8_t rxDepth, uint8_t txDepth);
		void setDMA2(bool rxDMA, bool txDMA, uint8_t rxDepth, uint8_t txDepth);
};
#endif