#include <cstdlib>
#include <string>
#include "main.h"
#include "Functions.h"
#include "Definitions.h"
#include "Boolean.h"
#include "NextionLCD.h"
#include "mbed.h"
#include "platform/CircularBuffer.h"
#include "rtos.h"
#include "Threads.h"
#include "Languages.h"
#include "Ser25lcxxx.h"
#include "NVM.h"
#include "Watchdog.h"
#include "NextionPages.h"
#include "Controls.h"
#include "FastPWM.h"
#include "MotorControl.h"

/*

  _/      _/  _/_/_/_/  _/_/_/    _/_/_/    _/_/_/_/  _/_/_/        _/_/_/      _/_/      _/_/_/   
 _/      _/  _/        _/    _/  _/    _/  _/        _/    _/      _/    _/  _/    _/  _/          
_/      _/  _/_/_/    _/_/_/    _/    _/  _/_/_/    _/_/_/        _/    _/  _/    _/    _/_/       
 _/  _/    _/        _/    _/  _/    _/  _/        _/    _/      _/    _/  _/    _/        _/      
  _/      _/_/_/_/  _/    _/  _/_/_/    _/_/_/_/  _/    _/      _/_/_/      _/_/    _/_/_/    

*/

//================================================================================================================================
// COMPANY                :  Verderflex
//
// PROJECT                :  Verder DOS Peristaltic Pump
// DATE                   :  3rd March 2020   

string HARDWARE =           "Hardware Version : HVDOS X1.00";                                                
string SOFTWARE =           "Software Version : SVDOS X1.00";
string AUTHOR   =           "Author : Jim Lowe";

/*
 *   Firmware Numbering Scheme, V Major.Minor.Bugfix
 *
 *   Major  : Major Changes to function
 *   Minor  : Minor Changes to function
 *   Bugfix : Firmware fixes to function
 *
 */

// HARDWARE              :   NUCLEO-F746ZG + Nextion 4.3" Display
//
// COMPILER              :   Mbed Studio

//================================================================================================================================
//************************************************  Code Status / Changes ********************************************************
//================================================================================================================================

/*
 *
 *  NOTES on Nextion Displays:
 *  Nextion Push event is generated when the touch screen is PRESSED
 *  Nextion Pop event is generated when the touch screen is RELEASED

 *  Firmware Log, Initial Software Development Phase

    09/05/19    itoa() now working
    09/05/19_E  Nextion now running at 115200 Baud
    13/05/19_D  Ported over to Nucleo-F746ZG
    15/05/19_A  First version working on Mbed Studio
    15/05/19_F  Nextion RX Interrupt message decoder started
    16/05/19_A  Increment / Decrement arrow work but project need to be cleaned up before going further
    16/05/19_A  Push & Pop functions for touch buttons are working. 
    17/05/19_A  Adding debugging functions
    17/05/19_A  Created a debug thread seems to work ok  
    17/05/19_B  Success Nextion slider is working   
    17/05/19_B  Flow rate / RPM selection now working
    17/05/19_C  PWM now working, needs scaling factors to represent 100% for 1250 RPM etc
    17/05/19_C  Progress bar working
    20/05/19_A  Peripheral Pin List added
    20/05/19_D  PWM frequency changed to 24kHz, this was found to be the frequency used on the DRV8306EVM BLDC Demo board
    20/05/19_D  BLDC Motor working under PWM control, now reaches the same build level as QAD with Mbed 
    28/05/19_A  STM32 does not have any EEPROM, need EEPROM for NVM capability, adding EEPROM capabilities to project
    28/05/19_A  Add digital motor control
    28/05/19_A  Motor direction control working, other I/O need to be done when schematics arrive from TI
    28/05/19_B  Nextion back light dim now working, it is fussy about where it is executed
    29/05/19_B  EEPROM testing thread proves EEPROM can be accessed and read correctly, removing EEPROM result in all 00, with EEPROM pattern is repeatable.
    29/05/19_E  Good demo code for NVM, needs tidying up
    30/05/19_A  Can read full NVM, can write single bytes to NVM
    30/05/19_B  This EEPROM library cannot write a single byte to a memory location, it does not work, you have to write the entire NVM which is not good
    31/05/19_A  Other EEPROM library no good, will try this again
    31/05/19_A now working in a way to develop the NVM feature properly, this will be pursued after my holiday   
    10/06/19_A  Continue with NVM development  
    10/06/19_B  NVM ready for Nextion menu to be taken to next level to feature NVM settings
    12/06/19_A  Rotation arrow on menu was not controlling motor rotation, it had been broken by adding other menu buttons, and this is something I am aware of now and to watch out for
    13/06/19_B  Culled the amount of project in Mbed Studio as this was too big for the resources for Mbed Studio causing it to not recognize the target as an Mbed enabled device
    13/06/19_B  Though I could get away with doing the NVM update in the RX ISR but this causes a run time error crashing the processor, I will place the NVM update in a thread
    14/06/19_A  This code is broken trying to transition to better way to implement metering units selection, use next code or previous code
    14/06/19_C  Taken most of the day but now NVM RAM is being written to from EEPROM, due to char* pointer this mechanism not working as thought, I have a working method now.
    17/06/18_A  First implementation of working NVM, needs some refinement
    17/06/19_D  Good refinement achieved for Metering Units NVM
    17/06/19_E  memcpy() replaces for loop, implemented free() to free up dynamically created pointer memory after use as eepro.read() uses malloc()
    27/06/19_A  Pre work ONLY for 4-20mA, setup of button membrane, 4-20mA control inputs
    01/07/19_A  Pre work ONLY for 4-20mA, Open Drain outputs setup and default to off at power up
    02/07/19_A  Adding map function
    12/07/19_A  Assigned new I/O for start/stop buttons and start/stop LEDS as I/O I has assigned might be required later on as they are part of USB and ST_LINK stuff.
    15/07/19_A  Grey Box build all working correctly, phew
    15/07/19_A  Need to implement start / stop button control, added state machine
    15/07/19_A  Motor thread commented out, motor control part of state machine
    15/07/19_B  Not sure how text box b2 is displaying RPM when stopped, I have overridden this for now
    15/07/19_C  Stop/Start operational
    16/07/19_A  Scaling for RPM, ml/min and % working perfectly
    16/07/19_A_TEXT This is the first version that uses TEXT for displaying metering digits using printf
    17/07/19_C  Issues with nextion, this version look ok    
    18/07/19_A  Page component IDs definitions had changed on Nextion HMI code again when I created  16/07/19_A_TEXT version, its a really crappy and stops your working code from working
    19/07/19_A  3 Second Watchdog implemented and working   
    19/07/19_A  Nextion power control added and working 
    19/07/19_A  1ms timer implemented and working, not uses at present
    22/07/19_A  Fit 4-20mA Module and connect up to system
    22/07/19_A  DAC pins already use I will have to re-assign pins to be able to use DAC
    22/07/19_A  ADC is now measuring mV, ADC is 12bit so more noise than 10bit ADC
    23/07/19_B  4-20mA is outputting to debug screen and is scaled using mapF, look fairly stable and no averaging has been applied yet.
    24/07/19_A  Look at ways to make analogue reading  more stable 
    24/07/19_A  Speed control using LoopMate achieved, although values from ain.read_u16() do not seem to make any sense at this time, was expecting 0 - 4095?
    25/07/19_A  Good results displayed on debug screen without any software smoothing applied.
    25/07/19_B  Working really well now I have add the moving average algorithm, this adds a nice ramp rate to the product
    26/07/19_A  Full motor speed does not seem to be there, need to fix this and create a function for motor control via 4-20mA
    26/07/19_A  Motor speed is ok now, used mapF() to make sure motor has stopped at 4.0mA, previously it would only stop at 0.0mA
    26/07/19_A  Wow, showing excellent stability at 10uA resolution
    26/07/19_A  Metering unit displayed against mA, cool
    29/07/19_A  Rough ramp rate applied when change of motor direction, this code is only an experiment and is blocking and needs to be refined in to non-blocking code   
    01/08/19_A  Create functions for manual and 4-20mA
    01/08/19_A  Nextion definition need changing for updated display GUI featuring QDOS type menu system
    02/08/19_A  Created Timers.cpp to hold all timer functions and structures
    05/08/19_A  Trying to implement manual control with dual speed inc/dec functionality
    05/08/19_A  I have inc/dec functionality working, tidy code up and move forward
    05/08/19_C  meter val changed to int16_t data type from uint16_t so abs() can be used for zeroing negative numbers
    05/08/19_C  Nextion screen initializes on page 33 wher no buttons are, page 1 is called when start button is press, stop button calls page 33
    06/08/19_A  Threads remove until required, all processing done in main and nextion RX ISR, all working.
    07/08/19_A  Can now display flow units retrieval from NVM
    08/08/19_A  Implement flow unit selection and storage to EEPROM
    08/08/19_A  flow unit can not be see using extern!
    08/08/19_B  Flow unit can be written and read from EEPROM
    08/08/19_C  Implement manual and analogue operating mode stored in EEPROM     
    12/08/19_A  Analogue function need to support all flow units
    12/08/19_C  This version can demo flow units change, analogue and manual mode selected from menu system
    12/08/19_D  meterVal, minScale, maxScale are now floats to give resolution required, this version is not full working at this time
    13/08/19_A  displayed info in p1left not correct, need to implement incDecSpeed variable so when the touch key is held the inc/dec speed increases
    14/08/19_A  Beware EEPROM value had somehow been changed for run.mode stopping normal program from running, suspect power glitch, as a consequence 
                it has taken all morning to figure out the problem.
    14/08/19_B  System can now be re-booted by pressing and holding down the ON button for 3 seconds         
    15/08/19_A  Add some diagnostics to check EEPROM contents    
    16/08/19_A  Change state machine structure so meter screen is correctly updated on return from Nextion pages
    27/08/19_B  Inc/Dec indexer needs improvement
    28/08/19_B  Inc/Dec control has been improved, speed can be set manually some tweaking may still be required at later point, moving on to analogue out now.
    28/08/19_C  Start of DAC implementation to provide analogue output
    28/08/19_C  SPI moved from SPI1 on CN7 to SPI14 on CN9, this is required to allow connection to DACs required for analogue output
    28/08/19_C  I have verified that the SPI NVM is working after moving to other microcontroller pins for SPI14
    28/08/19_C  I am able to set analogue output currents for test purposes.
    28/08/19_E  Analogue output development started
    29/08/19_B  Manual speed adjustment improved further now.
    29/08/19_B  Stop LED used to prompt user to number indexer when setting flow rates
    02/09/19_A  Accurate analogue output when in manual mode  can be demonstrated
    02/09/19_B  Two dimensional array created for ALL VDOS flow rate types   
    02/09/19_B  Can store and retrieve floats to EEPROM as an experiment, need to implement this in analogue cal routine  
    04/09/19_A  Able to switch in and out of ANA_CAL state read for analogue menus
    05/09/19_A  WARNING page 1 component ID have changed
    06/09/19_B  Almost finished improved flow unit manual inc/dec control
    09/09/19_A  I need to be moving forward with analogue calibration so I have put on hold development o inc/dec control for now
    12/09/19_A  Floats can be written to and read from EEPROM for mA high, flow high, mA low, flow low calibrations
    12/09/19_B  % / mA graph working as initial demo
    16/09/19_A  Calibration graph line generation now done on Nextion by timer so no resources are required for this micro.
    17/09/19_A  IncDecControl function created that can be used for all inc/dec controls on all pages
    17/09/19_A  Clear out functions that are not used general tidy up of code
    18/09/19_A  Working on calibration graph, good progress made here
    19/09/19_A  struct settingsNVM has been updated to include calibration variables
    20/09/19_A  Hysteresis add to 4-20 pump motor start threshold
    20/09/19_A  Re-declaration of constants to give accurate rage #define ADC_MIN_V 0.013, #define ADC_MAX_V 2.9012//V   
    23/09/19_B  full NVM RAM loaded into NVM structure in READ_NVM
    23/09/19_B  USB ISOLATOR TO BE USED FROM NOW ON TO STOP USB GND EFFECTING THE UNIT
    23/09/19_B  Calibration is working, checking all aspects of calibration
    24/09/19_A  anaOut_mA(mapF(ana_mA, I_3MA, I_20MA, nvm.mA_low, nvm.mA_high)); APPEARS TO BE WORKING NOW, TESTING REQUIRED
    25/09/19_B  Calibarion, analogue output, displayed flow  units all appear to be match well to QDOS
    26/09/19_B  Works like QDOS for calibration now and analogue  in/out, issue when leaving calibration normal running does not resume correctly
    27/09/19_A  Calibration graph has refresh problem when adjusting flow rate %
    30/09/19_A  I hope the issue above in now fixed, this was done by erasing the RED line before going to the next calibration screen
    30/09/19_B  Start looking at speed feedback from tacho output on motor controller
    30/09/19_B  Tacko from fGout processed by isr test working
    01/10/19_A  General code tidy, removal of unused variables, add ISR module
    03/10/19_A  Implement motor stall detection
    07/10/19_A  Motor development put on hold as present motor controller cannot produce high torque at low speed as QDOS 20 does
    07/10/19_A  Remote Start / Stop implemented
    07/10/19_A  Remote Fluid Recover, reverses motor in analogue mode, will check more when we receive new QDOS with this feature
    07/10/19_A  Open Collector outputs now function, they have to be allocated to status and alarm software functions yet
    10/10/19_A  Remote run/stop operational, code tidy now required
    10/10/19_A  Need to store and retrieve the speed setting
    11/10/19_A  configure Neptune motor for shaft speed range of 100 - 2500 RPM
    14/10/19_A  Motor controller is on hold for now, I will concentrate of delivering the features of QDOS 20
    14/10/19_A  Implement Auto-restart feature    
    14/10/19_A  Flow units in manual mode saved and retreived from NVM after power loss
    16/10/19_B  State machine does not work as intended causing problem for auto-restart and flow unit change over, re-do state machine
    18/10/19_A  flow rate can be switched from one to another matching the percentage as QDOS does
    21/10/19_A  Auto-Restart, remote inputs and O/C outputs working, read for checking with QDOS 30
    22/10/19_A  Motor rotation is CCW but QDOS arrow is CW, Neptune will be set the same
    22/10/19_A  FastPWM library added, leave PWM prescaler at 1 so that that the smallest speed inc/dec results in a measureable PWM change,
                resolution is 12 bit (4096), measured bit width is ~20ns/bit, pwm period is 1/10000 = 100us, so 100us / 20ns = 5000
    22/10/19_B  Implement MAX prime function   
    23/10/19_A  To allow the touch button to be active at the stop screen as with QDOS the nexKeyScan() need to be call in the main loop   
    23/10/19_A  New motor ramp time added to motor function, need to connect up nBrake to motor controller         
    24/10/19_A  New readNVM() created to hide complexity and improve readability, working on motor ramp time parameter
    25/10/19_A  Nextion display has failed, as extra protection tristate TX & RX line until Nextion has powered up
    30/10/19_A  New Nextion display fitted, 1K resistor mod added to RX line
    30/10/19_B  QDOS 30 flow units added to replace QDOS 20 flow units
    31/10/19_A  Removed ramNVM so epprom data is transferred to and from direct to NVM data structure
    01/11/19_A  NVM memory working good and more easy to understand and use now
    01/11/19_B  PWM is now incremented / decremented at 0.1 resolution, so a change of 0.1% alter the PWM    
    01/11/19_B  Cannot find any problems with this version, best version so far       
    05/11/19_A  Calibration Menu cannot be accessed after initial calibration.  
    11/11/19_A  Work required for remote control inputs 
    11/11/19_A  Adding output1&2 settings
    12/11/19_A  Discovery that SPI frequency was set to 1000 Hz, I forgot about this    
    18/11/19_A  Various icon indicators added to page 1 top line
    20/11/19_A  Implementation of control settings
    21/11/19_A  Provide a version of software to use USB USART1 (PA9 & PA10) on product ONLY as product will not have ST-LINK 
    02/12/19_A  Method of programming via USB has been designed in hardware, trap software in while loop and it will re-boot via watchdog timer with boot0 connected to vdd
    03/12/19_A  #ifdef BOOTLOADER setup for production bootloader without USB/ST-LINK
    05/12/19_A  Microcontroller pin assignment checked with Kevin Moon's work, all ok.
    06/12/19_A  Continue with control settings menu, EEPROM variables for control settings
    11/12/19_A  Output 1 NVM working
    02/01/20_A  Output 1 & 2 appears to be working, NVM setting is being saved and retrieved correctly for displayed string names
    06/01/20_A  Trying to fix return from analogue cal routine, the nvm.flowUnitVal is being set to minium value
    06/01/20_B  The above is fixed, reset() was clearing nvm.flowUnitVal, this has been commeted out for now.    
    08/01/20_A  Code refactoring 
    13/01/20_A  Control Setting output 1&2, tick boxes for High and Low logic under development
    13/01/20_A  High & Low output logic setting now stored in nvm, develop function for remote inputs and O/C outputs
    16/01/20_A  Start/Stop input Pump Stop logic NVM tick boxes working.
    17/02/20_A  4-20mA Scale NVM tick boxes working.
    03/02/20_A  Remote pump Ster/Stop working same as QDOS
    04/02/20_A  Improve Nextion driver so that only updated data is sent to Nextion, if the data is the same nothing is done.
    04/02/20_B  Demonstrated code working for alarm outputs, continue this tomorrow
    10/02/20_A  Rotation arrow issues now resolved
    12/02/20_A  Control setting, inputs, o/c outputs, mA output scale all working
    13/02/20_A  DEBUG_LCD added to improve loop time further by being able to see if any LCD functions are being called continuoisly.
    24/02/20_A  Problem with with inc/dec controls, this is related to the speedLimit variable introduced
    24/02/20_A  SpeedLimit now operates from nvm.speedLimit, speed limit menu need to be implemented
    28/02/20_A  Improvements made to pause/stop symbol fucntion and output1 & 2 logic, test concur operation same as QDOS
    28/02/20_B  Staring investigatory work for ED motor controller controler support, setup an serial interface for rs232 motor control    
    28/02/20_C  Communication established on usart6 for motor control
    28/02/20_C  Speeded up tremendously the serial communiucations on all at usarts, they all work at 921600 Baud, Nextion seem not to 
                have a problem with this unimate fast baud, if I start to see problems I will slow the baud down.

*/

/*

    ***** TARGET CAN BE LOST FROM USB FOR NO REASON, ONLY WAY TO GET IT BACK IS TO RE-BOOT PC AND SELECT ACTIVE PROGRAM AGAIN *****

    Nextion text for mA readings will have to be set txt_maxl = 4 to limit precision

    Analogue read is ~15us
        
    Find relationship between RPM and PWM, make sure we have 4000 RPM, measure with tachometer
    Ramping need to be implemented so there is no abrupt stop and start at high speed
    Hide away complexity in functions
    Review all function, trim out useless functions
    
    Measured 4426 RPM unloaded

    Neptune requirement is 120 Litres/Hour = 2000ml/min

    Meeting today 16/8/19

    US Flow Units, DB
    Screen RED for alarm
    Screen saver
    Cooling fan as standard
    Digital Dosing Pump 
    Heads Left & Right fitment, rotation direction fixed based on which head is selected
    Sep 2020 exhibition

    QDOS 20 & 30 are the initial target

    PA_4 and PA_5 are DAC1 & DAC2 outputs, these are presently used for SPI, I am going to have to move the SPI in order to use the DACs

*/

//PERIPHERAL PINS
//https://github.com/ARMmbed/mbed-os/blob/master/targets/TARGET_STM/TARGET_STM32F7/TARGET_STM32F746xG/TARGET_NUCLEO_F746ZG/PeripheralPins.capabilities

/*
    25LC010AT

    1K SPI EEPROM (128 x 8)

    EEPROM 3V3 ok
         ---------                
    SCK |1       6| VDD
    VSS |2       5| /CS
    SI  |3       4| SO
         ---------

    PA_7 MOSI// Connected to RMII_CRS_DV [LAN8742A-CZ-TR_CRS_DV]
    PA_6 MISO  
    PA_5 SCLK    
    PA_4 SSEL  
*/

//Objects


/*USART

USART1 = VERDERBUS
USART2 = NEXTION DISPLAY
USART3 = BOOTLOADER
USART6 = MOTOR CONTROL


*/

#ifdef BOOTLOADER//USB TO UART BOOTLOADER
    /* 
        Physical connections PB10, PB11 USART3 required for Bootloader
        RXPC <- PB10 OUTPUT TXN        
        TXPC -> PB11 INPUT RXN
        These become Virtual Com Port PD_8, PD_9
    */
    Serial pc(SERIAL_TX, SERIAL_RX);//Bootloader port
#else//MBED VIA ST-LINK
    /* 
        Physical connections PD_8, PD_9 {USART3} debug port
        RXPC <- PD_8 OUTPUT TXN        
        TXPC -> PD_9 INPUT RXN
    */
    Serial pc(USBTX, USBRX);//Virtual Com Port PD_8, PD_9 Debug port required for MBED
#endif

NextionLCD lcd(USART2_TX, USART2_RX);//Tx, Rx, Nextion Port {USART2}

Serial usart6(USART6_TX, USART6_RX);//Tx, Rx Motor Controller {USART6}

SPI spi(MOSI,MISO,SCLK);//MOSI, MISO, SCLK, SSEL
Ser25LCxxx eeprom(&spi,SSEL,BYTE_SIZE,PAGE_SIZE);
Watchdog wd;

Timer tPop;//touch button release timer
Timer tPush;//touch button press timer
Timer tRst;//reset timer
Timer tSpeed;//numerical update rate of flow units

volatile float ml = 0;
bool maxBtnPress = false;//Prime buttOn (MAX)
uint8_t storedTask = false;//Remember the previous state machine state
bool rmtFlag = false;//Remote input flag
char *readbuf;
bool maxFlag = true;
bool arrowFlag = false;   

///////////////////////////////////////////////////////////////////////////////
//VARIABLES
///////////////////////////////////////////////////////////////////////////////
float minScale = 0;
float maxScale = 0;//Max Scale variable to stop max RPM, max ml/min and max %

///////////////////////////////////////////////////////////////////////////////
// ARRAYS
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// CONSTANTS
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// STRUCTURE DEFINITIONS
///////////////////////////////////////////////////////////////////////////////
struct settingsNVM nvm;

///////////////////////////////////////////////////////////////////////////////
// DEBUG
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// MAIN
///////////////////////////////////////////////////////////////////////////////
int main(){

    uint8_t task;

    task = INITIALISE;
 
    while(1){
    
        switch(task){
        
            case INITIALISE:    
                                sysInit();                                                                         
                                task = STOP_SCRN;
                                //task = DIAGS;
                                dbgStates(task); 
                                break;

            case STOP_SCRN:    
                                ml=ZERO;            
                                lcd.nexChgPage(PAGE_1_HOME);       
/*                                                            
                                lcd.nexDispSymbol("key", ON);
                                lcd.nexDispSymbol("leak", ON);
                                lcd.nexDispSymbol("update", ON);
                                lcd.nexDispSymbol("error", ON);
                                lcd.nexDispSymbol("wifi", ON);        
                                lcd.nexDispSymbol("locked", ON);
                                lcd.nexDispSymbol("unlocked", ON);
                                lcd.nexDispSymbol("temp", ON);
                                lcd.nexDispSymbol("stop", ON);
                                lcd.nexDispSymbol("pause", ON);                                                                                                                                                            
*/
                                lcd.nexDispSymbol("stop", ON);
                                lcd.nexDispSymbol("pause", OFF);                                  
                                lcd.nexDispSymbol("unlocked", ON);
                                decodeByteNVM(nvm.flowUnits, &precision);     
                               
                                lcd.nexSendTxt("p1left","VDOS");//page 1 middle text box    
                                lcd.nexSetFontCol("p1mid",RED);                                                                                          
                                lcd.nexSendTxt("p1mid","Manual Mode");//page 1 middle text box  

                                lcd.nexSendFloat("meter", nvm.flowUnitVal, precision);//display the flow units                                                             
                                stopBtn(BTN_BYPASS_TRUE);//button disasbled LEDs only      
                                motor(ZERO, CCW, BRAKE_OFF, MOTOR_DISABLE, 1);                      

                                storedTask = STOP_SCRN;//this state   
                                task = START;
                                dbgStates(task); 
                                break;
                                                                                                 
            case START:         
                                anaOut_mA(I_3MA);                                               
                                task = startUpMode(nvm.runMode);                         
                                storedTask = START;
                                                             
                                dbgStates(task); 
                                break;      
                                                            
            case MANUAL_SET:    //Setup for Manual running mode             
                                lcd.nexChgPage(PAGE_1_HOME);   
                                lcd.nexDispSymbol("unlocked", ON);
                                //lcd.nexSetFontCol("p1mid",BLACK);                                                                                           
                                //lcd.nexSendTxt("p1mid","Manual Mode");//page 1 middle text box                                                                                                                                                                                                                                                                
                        
                                lcd.nexSetCrop("inc",PAGE_1_HOME);//restore inc button image required for manual mode    
                                lcd.nexSetCrop("dec",PAGE_1_HOME);//restore dec button image required for manual mode                                                                                
                                decodeByteNVM(nvm.flowUnits, &precision);//display the flow unit symbol                                  
                                lcd.nexSendFloat("meter", nvm.flowUnitVal, precision);//display the flow unit value           
                                
                                storedTask = MANUAL_SET;//this state                                                                         
                                task = RUNNING;//next state  
                                dbgStates(task);                                
                                break;

            case ANALOGUE_SET:  //Setup for Analogue running mode                                    
                                lcd.nexChgPage(PAGE_1_HOME); 
                                lcd.nexDispSymbol("unlocked", ON);                                
                                lcd.nexSetFontCol("p1mid",RED);   
                                
                                lcd.nexSendTxt("p1mid","4-20mA Input Mode");//page 1 middle text box                                                                                                                                                                                               
                                lcd.nexSetCrop("inc",CLEAR_INC_DEC);//remove inc button image not required for analogue mode   
                                lcd.nexSetCrop("dec",CLEAR_INC_DEC);//remove dec button image not required for analogue mode                                                                                                                                                                            
                                decodeByteNVM(nvm.flowUnits, &precision); 
 
                                storedTask = ANALOGUE_SET;
                                task = RUNNING;  
                                dbgStates(task);                                                                                      
                                break;           
     
            case RUNNING:                   
                                switch(runModeNVM(nvm.runMode)){                                                                                                                        
                                    case MANUAL_SET:    task = manual();    break;
                                    case ANALOGUE_SET:  task = anaIn();       break;                                    
                                    //case FLOW_CAL:      task = flowCalMode();   break;                                    
                                    //case CONTACT_SET:   task = contactMode();   break;
                                    //case FLUIDREC_SET:  task = fluidRecMode();  break;  
                                    
                                    default:;
                                }   
                                  
                                task = STOP;   
                                dbgStates(task);                       
                                break;

            case STOP:                                       
                                if(((rmtRunStop == LO)&&(rmtFlag == true))||stopBtn(BTN_BYPASS_FALSE)){
                                                         
                                    lcd.nexDispSymbol("stop", OFF);                                       
                            
                                    task = STOP_SCRN;                                                                                                          
                                }
                                else
                                    task = RUNNING;
                                                
                                dbgStates(task); 
                                break;

            case ANA_CAL:            
                                task = anaCal();     
                                dbgStates(task);                                                         
                                break;

            case STALL:         while(1);
            
                                break;

            case FAULT:
                                break;     

            case DIAGS:         
             
                                break;                                                       

            default:            pc.printf("Error Main : Last task = %d",task);  
                             
                                //////////////////////////////////////while(1)
                                    //////////////////////////////////////////////////////wd.Service();//Kick the dog
        }

        #ifdef LOOP_TIME  
            loopTimeMeasure();
        #endif    
 
        #ifndef WDT_TEST
            if(!startBtn(OFF))//holding the start button will re-boot the system
                wd.Service();//Kick the dog    
        #endif        

        outPuts(leakIn, &nvm); 
        task = nexKeyScan(task, storedTask);//read nextion touch keys  
    }
}
