#include <cstdlib>
#include <string>
#include "main.h"
#include "Functions.h"
#include "Definitions.h"
#include "Boolean.h"
#include "NextionLCD.h"
#include "mbed.h"
#include "platform/CircularBuffer.h"
#include "rtos.h"
#include "Threads.h"
#include "Languages.h"
#include "Ser25lcxxx.h"
#include "NVM.h"
#include "FastPWM.h"

extern Ser25LCxxx eeprom;
extern Serial pc;//Debug Port via USB
extern SPI spi;//SPI

const char romNVM[NVM_SIZE]=    {   //Deafault NVM Parameters written to empty EEPROM                    
                                    RPM,//Flow Units
                                    MANUAL,//Run Mode
                                    AUTO_RESTART_OFF,
                                    PUMP_HEAD_OK,
                                    ALARM_OFF,
                                    HEAD_LEFT,
                                    PUMP_OFF,
                                    0x00,
                                    0x00,//8 MAX speed limit, 125.0
                                    0x00,
                                    0xFA,
                                    0x42,
                                    0xCD,//12 Flow Unit Value, 0.1                                                                                                                                                
                                    0xCC,
                                    0xCC,
                                    0x3D,                                                                                                            
                                    0x33,//16 Analogue low cal float value loc 0, 4.10
                                    0x33,
                                    0x84,
                                    0x40,
                                    0x0A,//20 Analogue high cal float loc 0, 19.80
                                    0xD7,     
                                    0x9F,
                                    0x41,
                                    0x00,//24 Flow % low cal float loc 0, 0.00
                                    0x00,     
                                    0x00,
                                    0x00,
                                    0x00,//28 Flow % high cal float loc 0, 100.00
                                    0x00,     
                                    0xC8,
                                    0x42,  
                                    0xCD,//32 ADC min V 0.1360V
                                    0xCC,
                                    0x4C,
                                    0x3C,
                                    0xCF,//36 ADC max V 3.31200V
                                    0xF7,
                                    0x53,
                                    0x40,
                                    0x03,//40 DAC V low 0.1348 = (0.3030*0.445)//445mV
                                    0x09,
                                    0x0A,
                                    0x3E,
                                    0xA6,//44 DAC V high 0.90597 = (0.3030*2.99)//2.99V     
                                    0xED,
                                    0x67,
                                    0x3F,
                                    RMT_STOP_PUMP_HIGH,//48
                                    OUT1_GENERAL_ALARM,
                                    OUT1_LOGIC_LO,                                                                          
                                    OUT2_RUN_STATUS,                         
                                    OUT2_LOGIC_LO,                                      
                                    _4_20MA_OUT_FULL_SCALE//53
                                };

                                
           
void clearNVM(void){
/*
    Clears NVM EEPROM space to 0xFF
*/
    #ifdef DEBUG_NVM
        pc.printf("\r\n\r\nClear All NMV settings (page 0 - 1)");//Display the buffer
    #endif

    eeprom.clearMem();
}

void wrtDefNVM(struct settingsNVM *data){

    #ifdef DEBUG_NVM
        pc.printf("\r\n\nWriting default NVM Settings from NVM ROM to NVM data structure and EEPROM");
    #endif    
    
    memcpy(data,romNVM,NVM_SIZE);

    writeNVM(NVM_START,NVM_SIZE,data);//Write NVM structure to EEPROM     
}

void loadNVM(struct settingsNVM *data, bool loadDefaults)
{
    char* tmp;

    if((blkChkEEPROM() == true)||(loadDefaults == true))//Virgin EEPROM     
    {
        #ifdef DEBUG_NVM
            pc.printf("\r\n\nVirgin EEPROM so load the default NVM ROM data");         
        #endif

        clearNVM();
        wrtDefNVM(data);
    }
    else
    {
        tmp = eeprom.read(NVM_START,NVM_SIZE);

        memcpy(data,tmp,NVM_SIZE);
        
        #ifdef DEBUG_NVM        
            pc.printf("\r\n\nEEPROM Contents");                 
            for(int i=0; i < NVM_SIZE; i++)      
                pc.printf("\r\n|(%d) \t 0x%02X |",i,tmp[i]);
    
            pc.printf("\r\n");
        #endif             

        free(tmp);           
    }
}

bool writeNVM(uint16_t start_addr,uint8_t len,struct settingsNVM *data){
/*
    Writes All contents of data structure to EEPROM
*/
    bool writeOK = true;
 
    if(!eeprom.write(start_addr,len,(char*)data)){
        #ifdef DEBUG_NVM
            pc.printf("\r\n\nWritting All NMV Settings\r\n");//Display the buffer            
        #endif
        writeOK = false;          
    }
    else
    {
        #ifdef DEBUG_NVM
            pc.printf("\r\n\nNMV Write FAILED\r\n");//Display the buffer     
        #endif
    }

    return(writeOK);
}
 
bool writeNVMfloat(uint16_t addr,float val){

    bool writeOK = true;

    #ifdef DEBUG_NVM
        pc.printf("\r\n\r\nWriting Float %f to EEPROM at base Addr %d", val, addr);//Display the buffer
    #endif    
        
    if(!eeprom.write(addr, sizeof(val), (char*)&val)){
            #ifdef DEBUG_NVM
                pc.printf("\r\n\r\nEEPROM Write Fail");
            #endif   
            writeOK = false;     
    }

    return(writeOK);
}

bool writeNVMByte(uint16_t addr,uint8_t val){

    static uint16_t lastAddr = 0;
    static uint8_t lastVal = 0;
    bool writeOK = true;

    if((addr != lastAddr)||(val != lastVal)){
        #ifdef DEBUG_NVM
            pc.printf("\r\n\r\nWriting Byte to 0x%02X to EEPROM at Addr %d", val, addr);
        #endif    
        
        if(!eeprom.write(addr, sizeof(val), (char*)&val)){
            #ifdef DEBUG_NVM
                pc.printf("\r\n\r\nEEPROM Write Fail");
            #endif   
            writeOK = false;     
        }
        else
            #ifdef DEBUG_NVM
                pc.printf("\tWrite OK");
            #endif    

        lastAddr = addr;
        lastVal = val;
    }

    return(writeOK);
}

bool blkChkEEPROM(void)
{
    /*
       Checks if ANY NVM location in the EEPROM is 0xFF, if so the EEPROM is deemed to be a virgin EEPROM
    */

    bool blank = true;
    uint8_t i;

    char* store = eeprom.read(NVM_START,NVM_SIZE);

    for (i=0;i<NVM_SIZE;i++)
    {
        #ifdef DEBUG_NVM
            pc.printf("\r\n\nBlank Check EEPROM = 0x%02X, Index = %d",store[i],i); 
        #endif    

        if(store[i] != BLANK)
        {            
            #ifdef DEBUG_NVM        
                pc.printf("\r\n\nEEPROM has NVM settings so load up the NVM data into NVM RAM");                 
                for(int i=0; i < NVM_SIZE; i++)      
                    pc.printf("\r\n|(%d) \t 0x%02X |",i,store[i]);
        
                pc.printf("\r\n");
            #endif

            blank = false;
            break;
        }
    }

    if(i==NVM_SIZE)
    {
        #ifdef DEBUG_NVM
            pc.printf("\r\n\nNVM EEPROM IS BLANK :-)"); 
        #endif    
    }

    return(blank);    
}

void initNVM(void){
    spi.format(8,3);
    spi.frequency(SPI_FREQ);
    Ser25LCxxx eeprom(&spi,SSEL,BYTE_SIZE,PAGE_SIZE);//CS SSEL
}
