/* Copyright (c) <2012> <P. Patel>, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 // ----------------------------- LED Ticker Interrupt Flasher -----------------------------------
 
#ifndef TOGGLE
#define TOGGLE

#include "mbed.h"

/** LED Toggle-Flashing (50% duty) library based on Ticker interrupts instead of wait
 *
 * Example
 * @code
 * Flasher myflasher(p20, 1.5);   // A flasher object attached to pin 20 with 1.5 second timing
 *
 * int main() {
 *     while (1) {
 *         myflasher.enable();          // Turn on flasher
 *     
 *         wait(3);                     // More code, simulated by wait
 *         myflasher.disableTo(1);      // Hold pin high
           myflasher.setTime(2);        // Set new timing
           wait(3);
           myflasher.enable();          // Resume flashing
           wait(3);
           myflasher = 0;               // Hold pin low (shorthand for disableTo(int state))
 *     }
 * }
 * @endcode
 */
class ToggleFlash {
public:
    /** Create a Flasherer object connected to a pin
     * @param pin The pin to which the Flasherer is attached
     */
    ToggleFlash(PinName pin);
    /** Create a Flasherer object connected to a pin and set the on-off time
     * @param pin The pin to which the Flasherer is attached
     * @param time The specified on-off time in seconds
     */
    ToggleFlash(PinName pin, float time);
    /** Set the on-off time ex. 1.5 means on for 1.5 seconds off for 1.5 seconds
     * @param time The specified on-off time in seconds
     */
    void setTime(float time);
    /** Retrieve the current time setting of the flasher object
     * @returns The current on-off time of the Flasher object
     */
    float getTime();
    /** Enable the flasher to start toggling pin
     *
     */
    void enable();
    /** Disable the flasher and give the pin a constant state
     * @param state 1 or 0 for constant on or constant off
     */
    void disableTo(int state);
    /** Return the current status of the flasher
     * @returns 1 or 0 for flasher enabled or flasher disabled
     */
    int readFlasher();
    /** Return the current state of the pin
     * @returns 1 or 0 for currently set high or low
     */
    int read();
#ifdef MBED_OPERATORS
    /** An operator shorthand for disableTo(int state) to give the pin a constant state
     */
    ToggleFlash& operator= (int state) {
        disableTo(state);
        return *this;
    }
    /** An operator shorthand for read() to read pin's current state
     */
    operator int() {
        return read();
    }
 #endif
private:  
    DigitalOut _pin;
    int _flasherState;
    float _time;
    Ticker _flasher;
    void toggle();
};
 
#endif