/* Copyright (c) <2012> <P. Patel>, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
 // ------------------------------ MAX518 Interfacing Library ------------------------------------
 
#ifndef MAX518
#define MAX518
#include "mbed.h"

/** MAX518 2-channel 8-bit DAC interfacing class to send I2C write commands
 *
 * Example:
 * @code
 * // Setting both buffered outputs to Vdd (5V)
 * #include "mbed.h"
 * #include "MAX518.h"
 * 
 * // DAC on I2C pins p9, p10 with address 0x58, bus frequency of 400kHz, 
 * // and expecting acknowledge bit in response.
 * DAC myDAC(p9, p10, 0x58, 400000, true);
 * 
 * int main() {
 *     while(1) {
 *         setDAC(255, 255);          // Set both channels to full output
 *       
 *         // Alternatively ...
 *         setCh1(255);
 *         setCh2(255);
 *     }
 * }
 * @endcode
 */
 
class DAC {
public:
    /** Create a MAX518 object connected to the specified I2C pins.
     *
     * @param sda PinName of I2C data pin, required
     * @param scl PinName of I2C clock pin, required
     * @param address 8-bit device address where LSB is forced to write state (zero), required
     * @param freq Optional I2C bus clock frequency, optional - defaults 100kHz
     * @param ack = true Set true if ACK bits are expected, optional - defaults true
     */
    DAC(PinName sda, PinName scl, int address, int freq, bool ack);
    DAC(PinName sda, PinName scl, int address, bool ack);
    DAC(PinName sda, PinName scl, int address, int freq);
    DAC(PinName sda, PinName scl, int address);

    /** Set both outputs with the same message.
     *
     * @param a Integer from 0 - 255 to set output of Channel 1
     * @param b Integer from 0 - 255 to set output of Channel 2
     * @returns True for success, false if no ACK's received as specified in constructor or invalid input)
     */
    bool setDAC(int a, int b);
    /** Set Channel 1 output.
     *
     * @param n Integer from 0 - 255 to set output of Channel 1
     * @returns True for success, false if no ACK's received as specified in constructor or invalid input)
     */
    bool setCh1(int n);
    /** Set Channel 2 output.
     *
     * @param n Integer from 0 - 255 to set output of Channel 2
     * @returns True for success, false if no ACK's received as specified in constructor or invalid input)
     */
    bool setCh2(int n);
    /** Reset device (set both outputs to 0).
     * @returns True for success, false if no ACK's received as specified in constructor)
     */
    bool reset();
    /** Power down device into silent monitoring mode.
     * @returns True for success, false if no ACK's received as specified in constructor)
     */
    bool powerDown();
private:  
    I2C _i2c;
    int _address;
    bool _ack;
};
 
#endif