// I2C device class (I2Cdev) demonstration Arduino sketch for MPU6050 class using DMP (MotionApps v2.0)
// 6/21/2012 by Jeff Rowberg <jeff@rowberg.net>
// Updates should (hopefully) always be available at https://github.com/jrowberg/i2cdevlib
//
// Changelog:
//     2012-06-21 - added note about Arduino 1.0.1 + Leonardo compatibility error
//     2012-06-20 - improved FIFO overflow handling and simplified read process
//     2012-06-19 - completely rearranged DMP initialization code and simplification
//     2012-06-13 - pull gyro and accel data from FIFO packet instead of reading directly
//     2012-06-09 - fix broken FIFO read sequence and change interrupt detection to RISING
//     2012-06-05 - add gravity-compensated initial reference frame acceleration output
//                - add 3D math helper file to DMP6 example sketch
//                - add Euler output and Yaw/Pitch/Roll output formats
//     2012-06-04 - remove accel offset clearing for better results (thanks Sungon Lee)
//     2012-06-01 - fixed gyro sensitivity to be 2000 deg/sec instead of 250
//     2012-05-30 - basic DMP initialization working

/* ============================================
I2Cdev device library code is placed under the MIT license
Copyright (c) 2012 Jeff Rowberg

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
===============================================
*/

#include "I2Cdev.h"
#include "MPU6050_6Axis_MotionApps20.h"

MPU6050 mpu;

const float M_PI = 3.14159265;

/* 四元数出力 (DMP生データ) */
// #define OUTPUT_READABLE_QUATERNION
// Memo:四元数とは、任意の方向を軸とした、任意角の回転を表す概念。

/* オイラー角出力 (四元数->オイラー角算出、ジンバルロック発生に注意) */
// #define OUTPUT_READABLE_EULER
// Memo:初期姿勢を基準にした座標

/* ロール/ピッチ/ヨー出力 (四元数->重力加速度->RPY算出、ジンバルロック発生に注意) */
#define OUTPUT_READABLE_YAWPITCHROLL
// Memo:センサーを基準にした座標

/* 重力加速度を除いた加速度(センサ基準座標) */
// #define OUTPUT_READABLE_REALACCEL

/* 重力加速度を除いた加速度(初期姿勢と重力加速度を基準にした座標) */
// #define OUTPUT_READABLE_WORLDACCEL


// MPU control/status vars
bool dmpReady = false;  // DMPの初期化が成功した場合はtrueに設定
uint8_t mpuIntStatus;   // センサの割り込みステータスを保持
uint8_t devStatus;      // デバイス動作後の状態 (0 = success, !0 = error)
uint16_t packetSize;    // expected DMP packet size (default is 42 bytes)
uint16_t fifoCount;     // count of all bytes currently in FIFO
uint8_t fifoBuffer[64]; // FIFO storage buffer

// orientation/motion vars
Quaternion q;           // [w, x, y, z]         quaternion container
VectorInt16 aa;         // [x, y, z]            accel sensor measurements
VectorInt16 aaReal;     // [x, y, z]            gravity-free accel sensor measurements
VectorInt16 aaWorld;    // [x, y, z]            world-frame accel sensor measurements
VectorFloat gravity;    // [x, y, z]            gravity vector
float euler[3];         // [psi, theta, phi]    Euler angle container
float ypr[3];           // [yaw, pitch, roll]   yaw/pitch/roll container and gravity vector


DigitalOut led1(LED1);
InterruptIn checkpin(p29);
Serial pc(USBTX, USBRX);

// ================================================================
// ===                     割り込み検出ルーチン                    ===
// ================================================================

volatile bool mpuInterrupt = false;     // センサの割り込みピンがHighになったかどうかを示します
void dmpDataReady() {
    mpuInterrupt = true;
}


// ================================================================
// ===                        メインルーチン                      ===
// ================================================================
void setup();
void loop();

int main() {
    setup();
    while(1) {
        loop();
    }
}

// ================================================================
// ===                      INITIAL SETUP                       ===
// ================================================================

void setup() {

    // initialize device
    pc.printf("Initializing I2C devices...\r\n");
    mpu.initialize();

    // verify connection
    pc.printf("Testing device connections...\r\n");
    if (mpu.testConnection()) pc.printf("MPU6050 connection successful\r\n");
    else pc.printf("MPU6050 connection failed\r\n");
    
    // wait for ready
    //Serial.println(F("\nSend any character to begin DMP programming and demo: "));
    //while (Serial.available() && Serial.read()); // empty buffer
    //while (!Serial.available());                 // wait for data
    //while (Serial.available() && Serial.read()); // empty buffer again

    // load and configure the DMP
    pc.printf("Initializing DMP...\r\n");
    devStatus = mpu.dmpInitialize();
    
    // make sure it worked (returns 0 if so)
    if (devStatus == 0) {
        // turn on the DMP, now that it's ready
        pc.printf("Enabling DMP...\r\n");
        mpu.setDMPEnabled(true);

        // enable Arduino interrupt detection
        pc.printf("Enabling interrupt detection (Arduino external interrupt 0)...\r\n");
        checkpin.rise(&dmpDataReady);
        
        mpuIntStatus = mpu.getIntStatus();

        // set our DMP Ready flag so the main loop() function knows it's okay to use it
        pc.printf("DMP ready! Waiting for first interrupt...\r\n");
        dmpReady = true;

        // get expected DMP packet size for later comparison
        packetSize = mpu.dmpGetFIFOPacketSize();
    } else {
        // ERROR!
        // 1 = initial memory load failed
        // 2 = DMP configuration updates failed
        // (if it's going to break, usually the code will be 1)
        
        pc.printf("DDMP Initialization failed (code ");
        pc.printf("%d", devStatus);
        pc.printf(")\r\n");
    }

}



// ================================================================
// ===                    MAIN PROGRAM LOOP                     ===
// ================================================================

void loop() {
    // DMPの初期化に失敗した場合、何もしない
    if (!dmpReady) return;

    // センサー割り込み or fifoオーバーフロー待ち
    while (!mpuInterrupt && fifoCount < packetSize) {
        // other program behavior stuff here
        // .
        // .
        // .
        // if you are really paranoid you can frequently test in between other
        // stuff to see if mpuInterrupt is true, and if so, "break;" from the
        // while() loop to immediately process the MPU data
        // .
        // .
        // .
    }

    // reset interrupt flag and get INT_STATUS byte
    mpuInterrupt = false;
    mpuIntStatus = mpu.getIntStatus();

    // get current FIFO count
    fifoCount = mpu.getFIFOCount();

    // オーバーフローをチェック (我々のコードがあまりにも非効率的でない限り、これが起こることはありません)
    if ((mpuIntStatus & 0x10) || fifoCount == 1024) {
        // reset so we can continue cleanly
        mpu.resetFIFO();
        //Serial.println(F("FIFO overflow!"));

    // オーバーフローが無ければ、DMPデータ・レディ割り込みをチェック
    } else if (mpuIntStatus & 0x02) {
        // wait for correct available data length, should be a VERY short wait
        while (fifoCount < packetSize) fifoCount = mpu.getFIFOCount();

        // read a packet from FIFO
        mpu.getFIFOBytes(fifoBuffer, packetSize);
        
        // track FIFO count here in case there is > 1 packet available
        // (this lets us immediately read more without waiting for an interrupt)
        fifoCount -= packetSize;

        #ifdef OUTPUT_READABLE_QUATERNION
            // display quaternion values in easy matrix form: w x y z
            mpu.dmpGetQuaternion(&q, fifoBuffer);
            printf("quat\t");
            printf("%f\t", q.w);
            printf("%f\t", q.x);
            printf("%f\t", q.y);
            printf("%f\t\r\n", q.z);
        #endif

        #ifdef OUTPUT_READABLE_EULER
            // display Euler angles in degrees
            mpu.dmpGetQuaternion(&q, fifoBuffer);
            mpu.dmpGetEuler(euler, &q);
            printf("euler\t");
            printf("%f\t", euler[0] * 180/M_PI);
            printf("%f\t", euler[1] * 180/M_PI);
            printf("%f\t\r\n", euler[2] * 180/M_PI);
        #endif

        #ifdef OUTPUT_READABLE_YAWPITCHROLL
            // display Euler angles in degrees
            mpu.dmpGetQuaternion(&q, fifoBuffer);
            mpu.dmpGetGravity(&gravity, &q);
            mpu.dmpGetYawPitchRoll(ypr, &q, &gravity);
            printf("ypr\t");
            printf("%f\t", ypr[0] * 180/M_PI);
            printf("%f\t", ypr[1] * 180/M_PI);
            printf("%f\t\r\n", ypr[2] * 180/M_PI);
        #endif

        #ifdef OUTPUT_READABLE_REALACCEL
            // display real acceleration, adjusted to remove gravity
            mpu.dmpGetQuaternion(&q, fifoBuffer);
            mpu.dmpGetAccel(&aa, fifoBuffer);
            mpu.dmpGetGravity(&gravity, &q);
            mpu.dmpGetLinearAccel(&aaReal, &aa, &gravity);
            printf("areal\t");
            printf("%d\t", aaReal.x);
            printf("%d\t", aaReal.y);
            printf("%d\t\r\n", aaReal.z);
        #endif

        #ifdef OUTPUT_READABLE_WORLDACCEL
            // display initial world-frame acceleration, adjusted to remove gravity
            // and rotated based on known orientation from quaternion
            mpu.dmpGetQuaternion(&q, fifoBuffer);
            mpu.dmpGetAccel(&aa, fifoBuffer);
            mpu.dmpGetGravity(&gravity, &q);
            mpu.dmpGetLinearAccelInWorld(&aaWorld, &aaReal, &q);
            printf("aworld\t");
            printf("%d\t", aaWorld.x);
            printf("%d\t", aaWorld.y);
            printf("%d\t\r\n", aaWorld.z);
        #endif

        // blink LED to indicate activity
        led1 = !led1;
    }
}