/**
 * @file    PPMout.h
 * @brief   PPMout - PPM signal generator for K64F
 * @author  Patrick Thomas
 * @version 1.0
 * @see
 *
 * Copyright (c) 2016
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef PPMOUT_H
#define PPMOUT_H

#include "mbed.h"

#define PIN(n,min,max) ((n) > (max) ? max : ((n) < (min) ? (min) : (n)))

// System clock
#define BUS_CLOCK 60000000
#define CLOCK_DIVIDER 12
#define CMT_PPS_VAL (CLOCK_DIVIDER - 1)

// Step array
#define CHANNELS 8
#define ARRAY_SIZE (CHANNELS + 1)

// Time parameters
#define STEPS_PER_MS (((BUS_CLOCK/CLOCK_DIVIDER)/8)/1000)
#define GAP_LENGTH_MS 0.3
#define FRAME_LENGTH_MS 22.5

// PPM parameters
#define PPM_RANGE_MS 1
#define PPM_ZERO_MS 0.7

// Derived step parameters
#define RANGE_STEPS (PPM_RANGE_MS*STEPS_PER_MS)
#define ZERO_STEPS (PPM_ZERO_MS*STEPS_PER_MS)
#define GAP_STEPS (int) (GAP_LENGTH_MS*STEPS_PER_MS)
#define TOTAL_GAP_STEPS (GAP_STEPS*ARRAY_SIZE)
#define FRAME_STEPS (FRAME_LENGTH_MS*STEPS_PER_MS)
#define MAX_NON_GAP_STEPS (FRAME_STEPS - TOTAL_GAP_STEPS)

/** A library for producing PPM output on the K64F (pin PTD7) using the CMT (Carrier/Modulator Transmitter) module
 *
 * Example:
 * @code
 *  #include "mbed.h"
 *  #include "PPMout.h"
 *
 *  PPMout myPPM;
 *  Serial pc(USBTX, USBRX);
 *
 *  int main()
 *  {
 *      while(1) {
 *
 *          char buffer[128];
 *          float f;
 *       
 *          pc.gets(buffer, 5);
 *          f = (float) atof(buffer);
 *      
 *          myPPM.update_channel(0, f);
 *      }
 *  }
 * @endcode
 */

class PPMout {
    
private:

    static int count_array[ARRAY_SIZE];
    static int counter;

    static void CMT_IRQHandler();
    void init();
    void set_sync_length();
    
public:
    
    /** Create PPMout instance
     */
    PPMout();
    
    /** Update a PPM channel value
     *  @param channel The channel to update (0-7)
     *  @param value The new value for the channel (0-1)
     */
    void update_channel(int channel, float value);
};

#endif