/**
 * @file    MIDIparser.h
 * @brief   MIDI parser - converts bytes into queued MIDI messages
 * @author  Patrick Thomas
 * @version 1.0
 * @see     
 *
 * Copyright (c) 2016
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef MIDIPARSER_MIDIPARSER_H_
#define MIDIPARSER_MIDIPARSER_H_

#include "MIDIMessage.h"
#include "MyBuffer.h"
#include <queue>

/** A parser that accepts bytes and outputs MIDI messages
 *
 * Example:
 * @code
 *  #include "mbed.h"
 *  #include "MIDIparser.h"
 *
 *  MIDIparser myParser;
 *  Serial pc(USBTX, USBRX);
 *
 *  int main()
 *  {
 *      while(1) {
 *       
 *      while(pc.readable()) {
 *       
 *          myParser.parse(pc.getc());
 *      }
 *       
 *      while (myParser.available()) {
 *           
 *          MIDIMessage myMessage = myParser.grab();
 *          pc.printf("%d %d %d %d\n", myMessage.type(), myMessage.channel(), myMessage.key(), myMessage.velocity());      
 *      } 
 *          
 *      }
 *  }
 * @endcode
 */

class MIDIparser {

private:

    MyBuffer<uint8_t> input_vector;
    queue<MIDIMessage> output_queue;

    bool sysex;
    bool dual;

public:

    /** Create a parser and allocate input/output buffers
     */
    MIDIparser();
    
    /** Input a byte to the parser
     *  @param data The byte to parse
     */
    void parse(uint8_t data);
    
     /** Determine if a MIDI message is in the queue
     *  @return 1 if something is ready, 0 otherwise
     */
    uint32_t available();
    
    /** Grab the next MIDI message
     *  @return The next MIDI message
     */
    MIDIMessage grab();
};

#endif /* MIDIPARSER_MIDIPARSER_H_ */
