/**
 * @file    MIDIparser.cpp
 * @brief   MIDI parser - converts bytes into queued MIDI messages
 * @author  Patrick Thomas
 * @version 1.0
 * @see     
 *
 * Copyright (c) 2016
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "MIDIparser.h"

MIDIparser::MIDIparser() {

}

void MIDIparser::parse(uint8_t this_byte) {

    // Data byte
    if (this_byte < 128) {

        // We expect the output buffer to be non-empty here
        if (input_vector.available()) {

            input_vector.put(this_byte);

            // Dual condition - perform one more read after this
            if (!sysex && dual) dual = false;

            // Single condition - send two bytes now
            else if (!sysex && !dual) {

                output_queue.push(MIDIMessage(input_vector.head(), input_vector.size()));
                input_vector.clear();
            }
        }
    }

    // System realtime byte
    else if (this_byte > 243) {

        switch (this_byte & 0xF) {

        // Undefined
        case 4:
        case 5:
            break;

            // Sysex end
        case 7:
            // Only output valid Sysex data
            if (sysex && !dual) output_queue.push(MIDIMessage(input_vector.head(), input_vector.size()));
            input_vector.clear();
            break;

            // Realtime
        default:
            output_queue.push(MIDIMessage(&this_byte, 1));
            break;
        }
    }

    // Status byte
    else {

        // We expect the output buffer to be empty here
        if (!input_vector.available()) {

            input_vector.put(this_byte);

            // Set flags
            switch ((this_byte >> 4) & 7) {
            case 0:
            case 1:
            case 2:
            case 3:
            case 6:
                sysex = false;
                dual = true;
                break;

            case 4:
            case 5:
                sysex = false;
                dual = false;
                break;

            case 7:
                switch (this_byte & 7) {
                case 0:
                    sysex = true;
                    dual = false;
                    break;

                case 1:
                case 3:
                    sysex = false;
                    dual = false;
                    break;

                case 2:
                    sysex = false;
                    dual = true;
                    break;

                default:
                    break;
                }
                break;

                default:
                    break;
            }
        }

        // Invalidate the output buffer by setting flags
        else {
            sysex = true;
            dual = true;
        }
    }
}

uint32_t MIDIparser::available() { return !output_queue.empty(); }

MIDIMessage MIDIparser::grab() {

    MIDIMessage output;
    output = output_queue.front();
    output_queue.pop();
    return output;
}
