/*
 * IRSensor.cpp
 * Copyright (c) 2016, ZHAW
 * All rights reserved.
 */

#include <cmath>
#include "IRSensor.h"

#define cycles 25

/**
 * Creates an IRSensor object.
 * @param distance an analog input object to read the voltage of the sensor.
 * @param bit0 a digital output to set the first bit of the multiplexer.
 * @param bit1 a digital output to set the second bit of the multiplexer.
 * @param bit2 a digital output to set the third bit of the multiplexer.
 * @param number the number of the sensor, either 0, 1, 2, 3, 4 or 5.
 */
IRSensor::IRSensor(AnalogIn* distance,AnalogIn* distance2, DigitalOut* bit0, DigitalOut* bit1, DigitalOut* bit2, int number)
{
    init(distance,distance2, bit0, bit1, bit2, number);
}


IRSensor::IRSensor()
{
}

void IRSensor::init(AnalogIn* distance,AnalogIn* distance2, DigitalOut* bit0, DigitalOut* bit1, DigitalOut* bit2, int number)
{

    this->distance = distance;  // set local references to objects
    this->distance2 = distance2;
    this->bit0 = bit0;
    this->bit1 = bit1;
    this->bit2 = bit2;

    this->number = number;
}


/**
 * Deletes the IRSensor object.
 */
IRSensor::~IRSensor() {}

/**
 * Gets the distance measured with the IR sensor in [m].
 * @return the distance, given in [m].
 */
float IRSensor::read()
{
    float avarage = 0;
    static float mean[6] = {0};
    if(number == 3) {
        *bit0 = 0;
        *bit1 = 0;
        *bit2 = 1;
    } else {
        *bit0 = (number >> 0) & 1;
        *bit1 = (number >> 1) & 1;
        *bit2 = (number >> 2) & 1;
    }

    float d = 0;
    float voltage = 0;
    for(int i = 0; i < cycles; i++) {
        if(number == 4) {
            voltage = distance2->read();
        } else {
            voltage = distance->read();
        }
        if(voltage < 0.1f) voltage = 0.1;   // set voltage if not big enough
        switch (number) {
            case 0:
                d  = 0.18f/pow((1.5f*voltage-0.1f),0.7f)-0.095f;
                break;
            case 1:
                d  = 0.18f/pow((1.5f*voltage-0.1f),0.7f)-0.095f;
                break;
            case 2:
                d  = 0.095f/pow((0.94f*voltage-0.05f),0.8f)-0.04f;
                break;
            case 3:
                d  = 0.192f/pow((1.8f*voltage-0.13f),0.7f)-0.095f;
                break;
            case 4:
                d  = 0.14f/pow((1.2f*voltage-0.09f),0.7f)-0.088f;
                break;
            case 5:
                d  = 0.18f/pow((1.55f*voltage-0.12f),0.7f)-0.095f;
                break;
            default:
                break;
        }
        avarage += d;
    }
    return (avarage / cycles);
}



/**
 * The empty operator is a shorthand notation of the <code>read()</code> method.
 */
IRSensor::operator float()
{
    return read();
}
