/* IMU service for Bluetooth Low Energy
 */

#ifndef __IMU_GENERIC_SERVICE_H__
#define __IMU_GENERIC_SERVICE_H__

#include "BLEDevice.h"
#include "GattCharacteristic.h"

/* Service and Characteristic UUID
*/
const char IMUServiceUUID[]                 = "EGW-IMU Service";
const char AccCharacteristicUUID[]          = "BMX055_ACC";
//const char AccInterruptCharacteristicUUID[] = "BMX055_ACC";
const char GyrCharacteristicUUID[]          = "BMX055_GYR";
//const char GyrInterruptCharacteristicUUID[] = "BMX055_GYR_INT";
const char MagCharacteristicUUID[]          = "BMX055_MAG";
//const char MagInterruptCharacteristicUUID[] = "BMX055_MAG_INT";

const UUID IMU_SERVICE_UUID                 = stringToUUID(IMUServiceUUID);
const UUID ACC_CHARACTERISTIC_UUID          = stringToUUID(AccCharacteristicUUID);
//const UUID ACC_INT_CHARACTERISTIC_UUID      = stringToUUID(AccInterruptCharacteristicUUID);
const UUID GYR_CHARACTERISTIC_UUID          = stringToUUID(GyrCharacteristicUUID);
//const UUID GYR_INT_CHARACTERISTIC_UUID      = stringToUUID(GyrInterruptCharacteristicUUID);
const UUID MAG_CHARACTERISTIC_UUID          = stringToUUID(MagCharacteristicUUID);
//const UUID MAG_INT_CHARACTERISTIC_UUID      = stringToUUID(MagInterruptCharacteristicUUID);

static const unsigned MAX_PAYLOAD_BYTES     = 30;
static const unsigned MAX_WITH_INT          = 30;//24;

/* IMU Service*/

class IMUService {
public:
    /**
     * Constructor
     */
    IMUService(BLEDevice &_ble) :
        ble(_ble),
        payloadACC(),
        payloadGYR(),
        payloadMAG(),
        ACC_CHARACTERISTIC(ACC_CHARACTERISTIC_UUID, payloadACC.getValuePointer(), (MAX_WITH_INT), (MAX_WITH_INT),
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ),
        GYR_CHARACTERISTIC(GYR_CHARACTERISTIC_UUID, payloadGYR.getValuePointer(), (MAX_WITH_INT), (MAX_WITH_INT),
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ),
        MAG_CHARACTERISTIC(MAG_CHARACTERISTIC_UUID, payloadMAG.getValuePointer(), (MAX_WITH_INT), (MAX_WITH_INT),
                (GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ)){ setupService(); }
                
    /* Sets Accelerometer and Interrupt data registers */

    /*    
    void setACC(uint8_t * accValue) {
        payloadACC.setACC(accValue);
        //ble.updateCharacteristicValue(ACC_CHARACTERISTIC.getValueAttribute().getHandle(), payloadACC.getValuePointer(), MAX_PAYLOAD_BYTES);
    }
    
    void setACCINT(uint8_t * intValue) {
        payloadACC.setACCINT(intValue);
    }
    */
    
    /* Sets Gyroscope and Interrupt data registers */
    /*
    void setGYR(uint8_t * gyrValue) {
        payloadGYR.setGYR(gyrValue);
    }

    void setGYRINT(uint8_t * intValue) {
        payloadGYR.setGYRINT(intValue);
    }
    */
    
    /* Sets Magnetometer and Interrupt data registers */
    /*
    void setMAG(uint8_t * magValue) {
        payloadMAG.setMAG(magValue);
    }
    void setMAGINT(uint8_t * intValue) {
        payloadMAG.setMAGINT(intValue);
    }
    */

    /* Update Accelerometer and Interrupt Characteristics */
    //void updateACC(void) {
    void updateACC(uint8_t * accValue) {
        payloadACC.updateACC(accValue);
        ble.updateCharacteristicValue(ACC_CHARACTERISTIC.getValueAttribute().getHandle(), payloadACC.getValuePointer(), MAX_PAYLOAD_BYTES);
    }

    /*
    void updateACCINT(void) {
        //payloadACC.updateACCINT(intValue);
        ble.updateCharacteristicValue(ACC_CHARACTERISTIC.getValueAttribute().getHandle(), payloadACC.getINTPointer(), MAX_PAYLOAD_BYTES);
    }
    */
    
    /* Update Gyroscope and Interrupt Characteristics */
    void updateGYR(uint8_t * gyrValue) {
        payloadGYR.updateGYR(gyrValue);
        ble.updateCharacteristicValue(GYR_CHARACTERISTIC.getValueAttribute().getHandle(), payloadGYR.getValuePointer(), MAX_PAYLOAD_BYTES);
    }

    /*
    void updateGYRINT(void) {
        //payloadGYR.updateGYRINT(intValue);
        ble.updateCharacteristicValue(GYR_CHARACTERISTIC.getValueAttribute().getHandle(), payloadGYR.getINTPointer(), MAX_PAYLOAD_BYTES);
    }
    */
    
    /* Update Magnetometer and Interrupt Characteristics */
    void updateMAG(uint8_t * magValue) {
        payloadMAG.updateMAG(magValue);
        ble.updateCharacteristicValue(MAG_CHARACTERISTIC.getValueAttribute().getHandle(), payloadMAG.getValuePointer(), MAX_PAYLOAD_BYTES);
    }

    /*
    void updateMAGINT(void) {
        //payloadMAG.updateMAGINT(magValue);
        ble.updateCharacteristicValue(MAG_CHARACTERISTIC.getValueAttribute().getHandle(), payloadMAG.getINTPointer(), MAX_PAYLOAD_BYTES);
    }
    */
    
    /**
     * This callback allows the GenericService to receive updates to the
     * controlPoint Characteristic.
     */
    virtual void onDataWritten(const GattCharacteristicWriteCBParams *params) {
        // (params->charHandle == controlPoint.getValueAttribute().getHandle()) {
            /* Do something here if the new value is 1; else you can override this method by
             * extending this class.
             * @NOTE: if you are extending this class, be sure to also call
             * ble.onDataWritten(this, &ExtendedHRService::onDataWritten); in
             * your constructor.
             */
        //}
    }

private:
    void setupService(void) {
        static bool serviceAdded = false; /* We should only ever need to add the heart rate service once. */
        if (serviceAdded) {
            return;
        }

        GattCharacteristic *charTable[] = {&ACC_CHARACTERISTIC, &GYR_CHARACTERISTIC, &MAG_CHARACTERISTIC};
        GattService         IMUService(IMU_SERVICE_UUID, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.addService(IMUService);
        serviceAdded = true;

        ble.onDataWritten(this, &IMUService::onDataWritten);
    }

private:
    /* Accelerometer Data Structure 
    *   Data is stored in payloadACC as a uint8_t array where the
    *   first half (payloadACC[0 ... MAX_PAYLOAD_BYTES - 1]), is the ACC data
    *   the second half (payloadACC[MAX_PAYLOAD_BYTES ... MAX_PAYLOAD_BYTESx2] contains interrupts available from the IMU
    */
    struct IMU_ACC {
        
        //Constructor
        IMU_ACC() : payloadACC(){}
        
        void updateACC(uint8_t * accValue) {
            int tmp = MAX_PAYLOAD_BYTES - 1;
            for (int i = 0; i <= tmp; i++) {
                payloadACC[i] = accValue[i]; 
                //payloadACC[tmp - i] = accValue[i]; 
            }//end If
        }
        
        /*  
        void setACCINT(uint8_t * intValue) {
            for (int i = MAX_PAYLOAD_BYTES; i < (MAX_WITH_INT); i++) {
                payloadACC[i] = intValue[i - MAX_PAYLOAD_BYTES]; 
            }//end If
        }
        */
        
        uint8_t * getValuePointer(void) {
            return payloadACC;
        }
        
        const uint8_t * getValuePointer(void) const {
            return payloadACC;
        }
        
        /*
        uint8_t * getINTPointer(void) {
            return (payloadACC);// + MAX_PAYLOAD_BYTES);
        }

        
        const uint8_t * getINTPointer(void) const {
            return (payloadACC);// + MAX_PAYLOAD_BYTES);
        }
        */
        
    private:
        uint8_t payloadACC[MAX_WITH_INT];
    };
    
    /* Gyroscope Data Structure 
    *   Data is stored in payloadGYR as a uint8_t array where the
    *   first half (payloadGYR[0 ... MAX_PAYLOAD_BYTES - 1]), is the GYR data
    *   the second half (payloadGYR[MAX_PAYLOAD_BYTES ... MAX_PAYLOAD_BYTESx2] contains interrupts available from the IMU
    */
    struct IMU_GYR {
        //Constructor
        IMU_GYR() : payloadGYR(){}
        
        void updateGYR(uint8_t * gyrValue) {\
            int tmp = MAX_PAYLOAD_BYTES - 1;
            for (int i = 0; i <= tmp; i++) {
                payloadGYR[i] = gyrValue[i]; 
                //payloadGYR[tmp - i] = gyrValue[i]; 
            }//end If
        }
        
        /*
        void setGYRINT(uint8_t * intValue) {
            for (int i = MAX_PAYLOAD_BYTES; i < (MAX_WITH_INT); i++) {
                payloadGYR[i] = intValue[i - MAX_PAYLOAD_BYTES]; 
            }//end If
        }
        */
        
        uint8_t * getValuePointer(void) {
            return payloadGYR;
        }
        
        const uint8_t * getValuePointer(void) const{
            return payloadGYR;
        }
        
        /*
        uint8_t *getINTPointer(void) {
            return (payloadGYR);// + MAX_PAYLOAD_BYTES);
        }
        
        const uint8_t * getINTPointer(void) const{
            return (payloadGYR);// + MAX_PAYLOAD_BYTES);
        }
        */
        
    private:
        uint8_t payloadGYR[MAX_WITH_INT];
    };
    
    /* Magnetometer Data Structure 
    *   Data is stored in payloadMAG as a uint8_t array where the
    *   first half (payloadMAG[0 ... MAX_PAYLOAD_BYTES - 1]), is the MAG data
    *   the second half (payloadMAG[MAX_PAYLOAD_BYTES ... MAX_PAYLOAD_BYTESx2] contains interrupts available from the IMU
    */
    struct IMU_MAG {
        //Constructor
        IMU_MAG() : payloadMAG(){}
        
        void updateMAG(uint8_t * magValue) {
            int tmp = MAX_PAYLOAD_BYTES - 1;
            for (int i = 0; i <= tmp; i++) {
                payloadMAG[i] = magValue[i]; 
                //payloadMAG[MAX_PAYLOAD_BYTES - i] = magValue[i]; 
            }//end If
        }
        
        /*
        void setMAGINT(uint8_t * intValue) {
            for (int i = MAX_PAYLOAD_BYTES; i < (MAX_WITH_INT); i++) {
                payloadMAG[i] = intValue[i - MAX_PAYLOAD_BYTES]; 
            }//end If
        }
        
        */

        uint8_t * getValuePointer(void) {
            return payloadMAG;
        }
        
        const uint8_t * getValuePointer(void) const {
            return payloadMAG;
        }
        
        /*
        uint8_t * getINTPointer(void) {
            return (payloadMAG + MAX_PAYLOAD_BYTES);
        }
        
        const uint8_t * getINTPointer(void) const {
            return (payloadMAG + MAX_PAYLOAD_BYTES);
        }
        */
        
    private:
        uint8_t payloadMAG[MAX_WITH_INT];
    };
    
private:
    BLEDevice               &ble;
    IMU_ACC                 payloadACC;
    IMU_GYR                 payloadGYR;
    IMU_MAG                 payloadMAG;
    GattCharacteristic      ACC_CHARACTERISTIC;
    //GattCharacteristic      ACC_INT_CHARACTERISTIC;
    GattCharacteristic      GYR_CHARACTERISTIC;
    //GattCharacteristic      GYR_INT_CHARACTERISTIC;
    GattCharacteristic      MAG_CHARACTERISTIC;
    //GattCharacteristic      MAG_INT_CHARACTERISTIC;
};                    
#endif /* #ifndef __BLE_IMU_SERVICE_H__*/
