/* I2C Service for EGW
 */

#ifndef __I2C_SERVICE_H__
#define __I2C_SERVICE_H__

#include "BLEDevice.h"
#include "GattCharacteristic.h"

/* Service and Characteristic UUID
*/
const char I2CServiceUUID[]                 ="EGW-I2C Service";
const char readCharacteristicUUID[]         ="EGW-I2C Read";
const char writeCharacteristicUUID[]        ="EGW-I2C Write";
const UUID I2C_SERVICE_UUID                 =stringToUUID(I2CServiceUUID);
const UUID I2C_READ_UUID                    =stringToUUID(readCharacteristicUUID);
const UUID I2C_WRITE_UUID                   =stringToUUID(writeCharacteristicUUID);
static const unsigned MAX_PAYLOAD_BYTES     = 20;
static const unsigned MAX_WITH_INT          = 20;

/* I2C Service*/

class I2CService {
public:
    /**
     * Constructor
     */
    I2CService(BLEDevice &_ble) :
        ble(_ble),
        payloadREAD(),
        payloadWRITE(),
        I2C_READ_CHARACTERISTIC(I2C_READ_UUID, payloadREAD.getValuePointer(), MAX_WITH_INT, MAX_WITH_INT, (GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE)),
        I2C_WRITE_CHARACTERISTIC(I2C_WRITE_UUID, payloadWRITE.getValuePointer(), MAX_WITH_INT, MAX_WITH_INT, (GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE))
        { setupService(); }
                

    /* Update I2C Read characteristic */
    void updateREAD(uint8_t* readValue) {
        payloadREAD.updateREAD(readValue);
        ble.updateCharacteristicValue(I2C_READ_CHARACTERISTIC.getValueAttribute().getHandle(), payloadREAD.getValuePointer(), MAX_PAYLOAD_BYTES);    
    }
    
    /* Update I2C Write characteristic */
    void updateWRITE(uint8_t* writeValue) {
        payloadWRITE.updateWRITE(writeValue);
        ble.updateCharacteristicValue(I2C_WRITE_CHARACTERISTIC.getValueAttribute().getHandle(), payloadWRITE.getValuePointer(), MAX_PAYLOAD_BYTES);    
    }

    virtual void onDataWritten(const GattCharacteristicWriteCBParams *params) {
        // (params->charHandle == controlPoint.getValueAttribute().getHandle()) {
            /* Do something here if the new value is 1; else you can override this method by
             * extending this class.
             * @NOTE: if you are extending this class, be sure to also call
             * ble.onDataWritten(this, &ExtendedHRService::onDataWritten); in
             * your constructor.
             */
        //}
    }

private:
    void setupService(void) {
        static bool serviceAdded = false; /* We should only ever need to add the heart rate service once. */
        if (serviceAdded) {
            return;
        }

        GattCharacteristic *charTable[] = {&I2C_READ_CHARACTERISTIC, &I2C_WRITE_CHARACTERISTIC};
        GattService         I2CService(I2C_SERVICE_UUID, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.addService(I2CService);
        serviceAdded = true;

        ble.onDataWritten(this, &I2CService::onDataWritten);
    }
   
   
     /* I2C Read Data Structure 
    *   This structure holds the data read via I2C to send out
    */
    struct I2C_READ {
        //Constructor
        I2C_READ() : payloadREAD(){}
        
        void updateREAD(uint8_t* readValue) {
            int tmp = MAX_PAYLOAD_BYTES - 1;
            for (int i = 0; i <= tmp; i++) {
                payloadREAD[i] = readValue[i];
            }//end If
        }
        
        uint8_t * getValuePointer(void) {
            return payloadREAD;
        }
        
        const uint8_t * getValuePointer(void) const{
            return payloadREAD;
        }
        
    private:
        uint8_t payloadREAD[MAX_WITH_INT];
    };
    
    /* I2C Write Data Structure 
    *   This structure holds the data to write via I2C
    */
    struct I2C_WRITE {
        //Constructor
        I2C_WRITE() : payloadWRITE(){}
        
        void updateWRITE(uint8_t* writeValue) {
            int tmp = MAX_PAYLOAD_BYTES - 1;
            for (int i = 0; i <= tmp; i++) {
                payloadWRITE[i] = writeValue[i];
            }//end If
        }
        
        uint8_t * getValuePointer(void) {
            return payloadWRITE;
        }
        
        const uint8_t * getValuePointer(void) const{
            return payloadWRITE;
        }
        
    private:
        uint8_t payloadWRITE[MAX_WITH_INT];
    };
    
private:
    BLEDevice               &ble;
    I2C_READ                payloadREAD;
    I2C_WRITE               payloadWRITE;
    GattCharacteristic      I2C_READ_CHARACTERISTIC;
    GattCharacteristic      I2C_WRITE_CHARACTERISTIC;
};                    
#endif /* #ifndef __I2C_SERVICE_H__*/
