/* Generic Bluetooth Low Energy Service with 
 */

#ifndef __BLE_GENERIC_SERVICE_H__
#define __BLE_GENERIC_SERVICE_H__

#include "BLEDevice.h"
#include "GattCharacteristic.h"

/* Generating UUID's for non-standard BLE Sig services
*  Note: The stringToUUID generates the 128bit UUID by doing ASCII conversions for each character
         The conversion is limited to the first 16 characters of the string (this includes spaces)
*/
const char GenericServiceUUID[] = "EGWT Generic";
const char GenericCharacteristicUUID[] = "EGWT GenChar";
const UUID GENERIC_SERVICE_UUID           = stringToUUID(GenericServiceUUID);//"EGWT Generic    ");
const UUID GENERIC_CHARACTERISTIC_UUID    = stringToUUID(GenericCharacteristicUUID);//"MMA* GenChar    ");

/* Generic Service*/

class GenericService {
public:
    /**
     * Constructor.
     */
    GenericService(BLEDevice &_ble, uint8_t GenericValue) :
        ble(_ble),
        valueBytes(GenericValue),
        GenericCharacteristic(GENERIC_CHARACTERISTIC_UUID, valueBytes.getPointer(), GenericValueBytes::MAX_VALUE_BYTES , GenericValueBytes::MAX_VALUE_BYTES,
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY){ setupService(); }

    /**
     * Same constructor as above, but with a 16-bit HRM Counter value.
     */
    GenericService(BLEDevice &_ble, uint16_t GenericValue) :
        ble(_ble),
        valueBytes(GenericValue),
        GenericCharacteristic(GENERIC_CHARACTERISTIC_UUID, valueBytes.getPointer(), GenericValueBytes::MAX_VALUE_BYTES, GenericValueBytes::MAX_VALUE_BYTES,
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY){ setupService(); }

    /**
     * Set a new 8-bit value for heart rate.
     */
    void updateGeneric(uint8_t GenericValue) {
        valueBytes.updateGeneric(GenericValue);
        ble.updateCharacteristicValue(GenericCharacteristic.getValueAttribute().getHandle(), valueBytes.getPointer(), GenericValueBytes::MAX_VALUE_BYTES);//valueBytes.getNumValueBytes());
    }

    /**
     * Set a new 16-bit value for heart rate.
     */
    void updateGeneric(uint16_t GenericValue) {
        valueBytes.updateGeneric(GenericValue);
        ble.updateCharacteristicValue(GenericCharacteristic.getValueAttribute().getHandle(), valueBytes.getPointer(), GenericValueBytes::MAX_VALUE_BYTES);//valueBytes.getNumValueBytes());
    }
    
    void updateGeneric(int16_t GenericValue) {
        valueBytes.updateGeneric(GenericValue);
        ble.updateCharacteristicValue(GenericCharacteristic.getValueAttribute().getHandle(), valueBytes.getPointer(), GenericValueBytes::MAX_VALUE_BYTES);//valueBytes.getNumValueBytes());
    }
    
    void updateGeneric(uint32_t GenericValue) {
        valueBytes.updateGeneric(GenericValue);
        ble.updateCharacteristicValue(GenericCharacteristic.getValueAttribute().getHandle(), valueBytes.getPointer(), GenericValueBytes::MAX_VALUE_BYTES);//valueBytes.getNumValueBytes());
    }
    
    void updateGeneric(uint8_t * GenericValue) {
        valueBytes.updateGeneric(GenericValue);
        ble.updateCharacteristicValue(GenericCharacteristic.getValueAttribute().getHandle(), valueBytes.getPointer(), GenericValueBytes::MAX_VALUE_BYTES);//valueBytes.getNumValueBytes());
    }
    
    /**
     * This callback allows the GenericService to receive updates to the
     * controlPoint Characteristic.
     */
    virtual void onDataWritten(const GattCharacteristicWriteCBParams *params) {
        // (params->charHandle == controlPoint.getValueAttribute().getHandle()) {
            /* Do something here if the new value is 1; else you can override this method by
             * extending this class.
             * @NOTE: if you are extending this class, be sure to also call
             * ble.onDataWritten(this, &ExtendedHRService::onDataWritten); in
             * your constructor.
             */
        //}
    }

private:
    void setupService(void) {
        static bool serviceAdded = false; /* We should only ever need to add the heart rate service once. */
        if (serviceAdded) {
            return;
        }

        GattCharacteristic *charTable[] = {&GenericCharacteristic};
        GattService         GenericService(GENERIC_SERVICE_UUID, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.addService(GenericService);
        serviceAdded = true;

        ble.onDataWritten(this, &GenericService::onDataWritten);
    }

private:
    /* Private internal representation for the bytes used to work with the vaulue of the heart-rate characteristic. */
    struct GenericValueBytes {
        static const unsigned MAX_VALUE_BYTES  = 20; /* FLAGS + up to two bytes for heart-rate */
        static const unsigned FLAGS_BYTE_INDEX = 0;

        static const unsigned VALUE_FORMAT_BITNUM = 0;
        static const uint8_t  VALUE_FORMAT_FLAG   = (1 << VALUE_FORMAT_BITNUM);

        GenericValueBytes(uint8_t GenericValue) : valueBytes() {
            updateGeneric(GenericValue);
        }

        GenericValueBytes(uint16_t GenericValue) : valueBytes() {
            updateGeneric(GenericValue);
        }
        
         GenericValueBytes(int16_t GenericValue) : valueBytes() {
            updateGeneric(GenericValue);
        }
        
        GenericValueBytes(uint32_t GenericValue) : valueBytes() {
            updateGeneric(GenericValue);
        }

        void updateGeneric(uint8_t GenericValue) {
            valueBytes[FLAGS_BYTE_INDEX] &= ~VALUE_FORMAT_FLAG;
            valueBytes[FLAGS_BYTE_INDEX + 1] = GenericValue;
        }

        void updateGeneric(uint16_t GenericValue) {
            /*
            valueBytes[FLAGS_BYTE_INDEX] |= VALUE_FORMAT_FLAG;
            valueBytes[FLAGS_BYTE_INDEX + 1] = (uint8_t)(GenericValue & 0xFF);
            valueBytes[FLAGS_BYTE_INDEX + 2] = (uint8_t)(GenericValue >> 8);
            */
            uint8_t low = (uint8_t)GenericValue;
            uint8_t high = uint8_t(GenericValue >> 8);
            valueBytes[0] = high; //1
            valueBytes[1] = low; //2
        }
        
        void updateGeneric(int16_t GenericValue) {
            /*
            valueBytes[FLAGS_BYTE_INDEX] |= VALUE_FORMAT_FLAG;
            valueBytes[FLAGS_BYTE_INDEX + 1] = (uint8_t)(GenericValue & 0xFF);
            valueBytes[FLAGS_BYTE_INDEX + 2] = (uint8_t)(GenericValue >> 8);
            */
            int8_t low = (int8_t)GenericValue;
            int8_t high = int8_t(GenericValue >> 8);
            valueBytes[0] = high; //1
            valueBytes[1] = low; //2
        }
        
        void updateGeneric(uint32_t GenericValue) {
            /*
            valueBytes[FLAGS_BYTE_INDEX] |= VALUE_FORMAT_FLAG;
            valueBytes[FLAGS_BYTE_INDEX + 1] = (uint8_t)(GenericValue & 0xFF);
            valueBytes[FLAGS_BYTE_INDEX + 2] = (uint8_t)(GenericValue >> 8);
            */
            
            for (int i = 0; i < GenericValueBytes::MAX_VALUE_BYTES ; i++) 
            {
                uint32_t buff = GenericValue;
                valueBytes[GenericValueBytes::MAX_VALUE_BYTES - i] = (uint8_t)(buff >> (8 * (i-1)));
            }
        }
        
        void updateGeneric(uint8_t * GenericValue) {
            for (int i = 0; i <= GenericValueBytes::MAX_VALUE_BYTES; i++) {
                valueBytes[i] = GenericValue[i];
                //valueBytes[GenericValueBytes::MAX_VALUE_BYTES - i] = GenericValue[i];    
            }//end If
        }

        uint8_t *getPointer(void) {
            return valueBytes;
        }

        const uint8_t *getPointer(void) const {
            return valueBytes;
        }

        unsigned getNumValueBytes(void) const {
            return 1 + ((valueBytes[FLAGS_BYTE_INDEX] & VALUE_FORMAT_FLAG) ? sizeof(uint16_t) : sizeof(uint8_t));
        }

    private:
        /* First byte = 8-bit values, no extra info, Second byte = uint8_t HRM value */
        /* See --> https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.heart_rate_measurement.xml */
        uint8_t valueBytes[MAX_VALUE_BYTES];
    };

private:
    BLEDevice           &ble;
    GenericValueBytes    valueBytes;
    GattCharacteristic   GenericCharacteristic;
};                    
#endif /* #ifndef __BLE_GENERIC_SERVICE_H__*/
