/* HTTPMap.h */
/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


#ifndef HTTPMAP_H_
#define HTTPMAP_H_

#include "../IHTTPData.h"

#define HTTPMAP_TABLE_SIZE 32

/** Map of key/value pairs
 * Used to transmit POST data using the application/x-www-form-urlencoded encoding
 */
class HTTPMap: public IHTTPDataIn, public IHTTPDataOut
{
public:
  /**
   Instantiates HTTPMap
   It supports at most 32 key/values pairs
   */
  HTTPMap();

  /** Create an HTTPMap instance for input
   * @param str Buffer to store the incoming string
   * @param size Size of the buffer
   */
  HTTPMap(char* buf, size_t size);

  /** Put Key/Value pair
   The references to the parameters must remain valid as long as the clear() function is not called
   @param key The key to use
   @param value The corresponding value
   */
  void put(const char* key, const char* value);

  /** Pop one Key/Value pair
   The returned refrences remain valid as long as the original buffer is not modified
   @param key The key
   @param value The corresponding value
   */
  bool pop(char*& key, char*& value);

  /** Get Value specified in Key
   The value is copied into the provided buffer
   @param key The key of the value to get
   @param val_buf The output buffer into which the value will be copied
   @param max The size of the buffer
   @param len The number of bytes copied into the output buffer
   */
  bool get(const char* key, char* val_buf, const size_t max);

  /** Clear table
   */
  void clear();

protected:
  //IHTTPDataOut
  virtual void readReset();

  virtual int read(char* buf, size_t len, size_t* pReadLen);

  virtual int getDataType(char* type, size_t maxTypeLen); //Internet media type for Content-Type header

  virtual bool getIsChunked(); //For Transfer-Encoding header

  virtual size_t getDataLen(); //For Content-Length header

  //IHTTPDataIn
  virtual void writeReset();
  
  virtual int write(const char* buf, size_t len);

  virtual void setDataType(const char* type); //Internet media type from Content-Type header

  virtual void setIsChunked(bool chunked); //From Transfer-Encoding header

  virtual void setDataLen(size_t len); //From Content-Length header, or if the transfer is chunked, next chunk length

private:
  const char* m_keys[HTTPMAP_TABLE_SIZE];
  const char* m_values[HTTPMAP_TABLE_SIZE];
  
  char* m_str;
  size_t m_size;

  size_t m_pos;
  size_t m_count;
  
  size_t m_read_pos;
};

#endif /* HTTPMAP_H_ */
