/* HTTPMap.cpp */
/* Copyright (C) 2012 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "HTTPMap.h"

#include <cstring>

#include <cctype>

#define OK 0

#define MIN(x,y) (((x)<(y))?(x):(y))
#define MAX(x,y) (((x)>(y))?(x):(y))

using std::strncpy;
using std::memcpy;

static char UrlDecode(char* str, size_t& offset);
static char HexDecode(char c);

HTTPMap::HTTPMap() : m_pos(0), m_count(0)
{

}

HTTPMap::HTTPMap(char* str, size_t size) : m_str(str), m_size(size), m_pos(0), m_read_pos(0)
{

}

void HTTPMap::put(const char* key, const char* value)
{
  if(m_count >= HTTPMAP_TABLE_SIZE)
  {
    return;
  }
  m_keys[m_count] = key;
  m_values[m_count] = value;
  m_count++;
}

void HTTPMap::clear()
{
  m_count = 0;
  m_pos = 0;
}

/*virtual*/ void HTTPMap::readReset()
{
  m_pos = 0;
}

/*virtual*/ int HTTPMap::read(char* buf, size_t len, size_t* pReadLen)
{
  if(m_pos >= m_count)
  {
    *pReadLen = 0;
    m_pos = 0;
    return OK;
  }

  //URL encode
  char* out = buf;
  const char* in = m_keys[m_pos];
  if( (m_pos != 0) && (out - buf < len - 1) )
  {
    *out='&';
    out++;
  }

  while( (*in != '\0') && (out - buf < len - 3) )
  {
    if (std::isalnum(*in) || *in == '-' || *in == '_' || *in == '.' || *in == '~')
    {
      *out = *in;
      out++;
    }
    else if( *in == ' ' )
    {
      *out='+';
      out++;
    }
    else
    {
      char hex[] = "0123456789abcdef";
      *out='%';
      out++;
      *out=hex[(*in>>4)&0xf];
      out++;
      *out=hex[(*in)&0xf];
      out++;
    }
    in++;
  }

  if( out - buf < len - 1 )
  {
    *out='=';
    out++;
  }

  in = m_values[m_pos];
  while( (*in != '\0') && (out - buf < len - 3) )
  {
    if (std::isalnum(*in) || *in == '-' || *in == '_' || *in == '.' || *in == '~')
    {
      *out = *in;
      out++;
    }
    else if( *in == ' ' )
    {
      *out='+';
      out++;
    }
    else
    {
      char hex[] = "0123456789abcdef";
      *out='%';
      out++;
      *out=hex[(*in>>4)&0xf];
      out++;
      *out=hex[(*in)&0xf];
      out++;
    }
    in++;
  }

  *pReadLen = out - buf;

  m_pos++;
  return OK;
}

/*virtual*/ int HTTPMap::getDataType(char* type, size_t maxTypeLen) //Internet media type for Content-Type header
{
  strncpy(type, "application/x-www-form-urlencoded", maxTypeLen-1);
  type[maxTypeLen-1] = '\0';
  return OK;
}

/*virtual*/ bool HTTPMap::getIsChunked() //For Transfer-Encoding header
{
  return false; ////Data is computed one key/value pair at a time
}

/*virtual*/ size_t HTTPMap::getDataLen() //For Content-Length header
{
  size_t count = 0;
  for(size_t i = 0; i< m_count; i++)
  {
    //URL encode
    const char* in = m_keys[i];
    if( i != 0 )
    {
      count++;
    }

    while( (*in != '\0') )
    {
      if (std::isalnum(*in) || *in == '-' || *in == '_' || *in == '.' || *in == '~')
      {
        count++;
      }
      else if( *in == ' ' )
      {
        count++;
      }
      else
      {
        count+=3;
      }
      in++;
    }

    count ++;

    in = m_values[i];
    while( (*in != '\0') )
    {
      if (std::isalnum(*in) || *in == '-' || *in == '_' || *in == '.' || *in == '~')
      {
        count++;
      }
      else if( *in == ' ' )
      {
        count++;
      }
      else
      {
        count+=3;
      }
      in++;
    }
  }
  return count;
}


/*virtual*/ void HTTPMap::writeReset()
{
  m_pos = 0;
}

/*virtual*/ int HTTPMap::write(const char* buf, size_t len)
{
  size_t writeLen = MIN(len, m_size - 1 - m_pos);
  memcpy(m_str + m_pos, buf, writeLen);
  m_pos += writeLen;
  m_str[m_pos] = '\0';
  return OK;
}

/*virtual*/ void HTTPMap::setDataType(const char* type) //Internet media type from Content-Type header
{

}

/*virtual*/ void HTTPMap::setIsChunked(bool chunked) //From Transfer-Encoding header
{

}

/*virtual*/ void HTTPMap::setDataLen(size_t len) //From Content-Length header, or if the transfer is chunked, next chunk length
{

}


bool HTTPMap::pop(char*& key, char*& value)
{
    size_t j = m_read_pos;
    
    key = &m_str[m_read_pos];
    while(m_read_pos < m_size && m_str[m_read_pos] != '=' && m_str[m_read_pos] != '\0')
    {
        m_str[j++] = UrlDecode(m_str, m_read_pos);
        m_read_pos++;
    }
    
    if(m_read_pos >= m_size || m_str[m_read_pos] == '\0')
    {
        key = NULL;
        value = NULL;
        return false;
    }
        
    m_str[j++] = '\0';
    j = ++m_read_pos;
    value = &m_str[m_read_pos];
    
    while(m_read_pos < m_size-1 && m_str[m_read_pos] != '&' && m_str[m_read_pos] != '\0')
    {
        m_str[j++] = UrlDecode(m_str, m_read_pos);
        m_read_pos++;
    }
    
    m_str[j++] = '\0';
    
    return true;
}


bool HTTPMap::get(const char* key, char* val_buf, const size_t max)
{
    size_t i, j = m_read_pos;
    
    while(j < m_size && m_str[j] != '\0')
    {
        i = 0;
        while(key[i] == m_str[j] && j < m_size && m_str[j] != '=' && m_str[j] != '\0')
        {
            i++;
            j++;
        }
        
        if(m_str[j] == '=')
        {
            j++;
            i = 0;
            while(i < max-1 && j < m_size && m_str[j] != '&' && m_str[j] != '\0')
            {
                val_buf[i++] = UrlDecode(m_str, j);
                j++;
            }
            
            val_buf[i] = '\0';
            
            return true;
        }
        
        while(j < m_size && m_str[j] != '&' && m_str[j] != '\0')
        {
            val_buf[i++] = UrlDecode(m_str, j);
            j++;
        }
        
        if(m_str[j] == '&')
            j++;
            
    }
    
    return false;
}

static char UrlDecode(char* str, size_t& offset)
{
    char c;
    
    if(str[offset] == '%')
    {
        c = HexDecode(str[offset+2]) | (HexDecode(str[offset+1]) << 4);
        offset+=2;
        
        if (c == 0)
        {
            return 0x1A; // SUB
        }
        
        return c;
    }
    else if(str[offset] == '+')
    {
        return ' ';
    }
    else
    {
        return str[offset];
    }
}
static char HexDecode(char c)
{
        if(c >= '0' && c <= '9')
        {
            return c - '0';
        }
        else if(c >= 'A' && c <= 'F')
        {
            return(c - ('A' - 0xA));
        }
        else if(c >= 'a' && c <= 'f')
        {
            return(c - ('a' - 0xA));
        }
        else
        {
            return 0x00;
        }
}
