#include "telemetry/Telemetry.hpp"
#include "FRDM-TFC/TFC.h"

/*
    Example of the 'Telemetry' library, a portable communication library for embedded devices.
    
    This code fetches all the data from the Freescale Cup RC-car.
    It relies on a more advanced shield library that is able to read current feedback from HBridges.
    
    The available data is the following :
    * Potentiometer 0
    * Potentiometer 1
    * 4-bit DIP switch
    * Battery level
    * current inside HBridge A
    * current inside HBridge B
    
    You can use the Pytelemetry Command Line Interface to open plots, visualize the received data,
    and communicate with the car.
    See https://github.com/Overdrivr/pytelemetrycli
*/

struct TM_state
{
    float throttle;
};

// Definition of the callback function
void process(TM_state * state, TM_msg * msg);

int main()
{
    // Init the shield
    TFC_Init();
    
    // Create a Telemetry instance, running on uart at 115200 bauds
    Telemetry tm(115200);
    
    // a data structure to hold writeable parameters
    // i.e. the car direction and throttle that will be both be controlled from the laptop
    TM_state state;
    state.throttle = 0.0;
    
    // Suscribe our custom processing function (= callback function), and pass the data structure
    // so that we can access it inside
    // This way, everytime a frame is received, Telemetry will call this function for us,
    // and pass the TM_state data structure to it.
    tm.sub(process, &state);
    
    Timer refresh_timer;
    refresh_timer.start();
    
    // Activate the engines !
    TFC_HBRIDGE_ENABLE;

    for( ; ; )
    {
        // 20 times per second, re-send data
        if(refresh_timer.read_ms() > 50)
        {
            tm.pub_f32("HBcurrentA",TFC_ReadMotorCurrent(0));
            tm.pub_f32("HBcurrentB",TFC_ReadMotorCurrent(1));
            tm.pub_f32("pot0",TFC_ReadPot(0));
            tm.pub_f32("pot1",TFC_ReadPot(1));
            tm.pub_f32("bat",TFC_ReadBatteryVoltage());
            tm.pub_u8("dip",TFC_GetDIP_Switch());
            tm.pub_f32("throttle",state.throttle);
            
            TFC_SetMotorPWM(state.throttle ,state.throttle);
            // Update in order to process received data
            tm.update();
        }    
    }   
}

// This is our processing function called every time a frame is received
// First parameter is a pointer to the data structure we defined in main
// Second parameter is a pointer to a data structure containing all info about received frame 

void process(TM_state * state, TM_msg * msg)
{
    // If data type matches 'float32'
    if(msg->type == TM_float32)
    {
        float th = 0;
        // Emplace the data into 'th'
        if(emplace_f32(msg,&th))
        {
            // If the emplace has worked, push the value to the actual value state->throttle
            // This way the processing is entirely safe
            state->throttle = th;   
        }
    }   
}