/* WiFi Example with ThingSpeak
 * Copyright (c) 2016 ARM Limited
 *                    Fuji Electronics Hiroaki Okoshi 2016.12.22 Modify.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "TCPSocket.h"

#if TARGET_UBLOX_EVK_ODIN_W2
#include "OdinWiFiInterface.h"
OdinWiFiInterface wifi;
#else
#if !TARGET_FF_ARDUINO
#error [NOT_SUPPORTED] Only Arduino form factor devices are supported at this time
#endif
#include "ESP8266Interface.h"
ESP8266Interface wifi(D1, D0);
#endif

#include "LM75B.h"              //  温度センサー
LM75B sensor(D14,D15);

DigitalOut led1(LED1);

const char *sec2str(nsapi_security_t sec)
{
    switch (sec) {
        case NSAPI_SECURITY_NONE:
            return "None";
        case NSAPI_SECURITY_WEP:
            return "WEP";
        case NSAPI_SECURITY_WPA:
            return "WPA";
        case NSAPI_SECURITY_WPA2:
            return "WPA2";
        case NSAPI_SECURITY_WPA_WPA2:
            return "WPA/WPA2";
        case NSAPI_SECURITY_UNKNOWN:
        default:
            return "Unknown";
    }
}

void scan_demo(WiFiInterface *wifi)
{
    WiFiAccessPoint *ap;

    printf("Scan:\r\n");

    int count = wifi->scan(NULL,0);

    /* Limit number of network arbitrary to 15 */
    count = count < 15 ? count : 15;

    ap = new WiFiAccessPoint[count];
    count = wifi->scan(ap, count);
    for (int i = 0; i < count; i++) {
        printf("Network: %s secured: %s BSSID: %hhX:%hhX:%hhX:%hhx:%hhx:%hhx RSSI: %hhd Ch: %hhd\r\n", ap[i].get_ssid(),
               sec2str(ap[i].get_security()), ap[i].get_bssid()[0], ap[i].get_bssid()[1], ap[i].get_bssid()[2],
               ap[i].get_bssid()[3], ap[i].get_bssid()[4], ap[i].get_bssid()[5], ap[i].get_rssi(), ap[i].get_channel());
    }
    printf("%d networks available.\r\n", count);

    delete[] ap;
}

/*
 * ThingSpeak Demo
 */

void ThingSpeak_demo(NetworkInterface *net)
{
    TCPSocket socket;
    socket.open(net);   // ソケットオープン

    socket.connect("api.thingspeak.com", 80);
    char urlBuffer[128];

    // 温度情報取得
    float temperature = sensor.read();

    // メッセージ用テキストを生成
    urlBuffer[0]=0;
    sprintf(urlBuffer, "GET /update?api_key=%s&field1=%f\r\n\r\n", MBED_CONF_APP_THINGSPEAK_KEY, temperature);

    // ThingSpeed にメッセージを送信
    int BuffreLen = strlen(urlBuffer);
    int scount = socket.send(urlBuffer, BuffreLen+1);
    printf ("strlen=%d,urlBuffer=%.*s",BuffreLen,BuffreLen-2,urlBuffer);

    // for debug    // デバグ時に受信内容を確認する。
    // char rbuffer[64];
    // int rcount = socket.recv(rbuffer, sizeof rbuffer);
    // printf("recv %d [%.*s]\r\n", rcount, strstr(rbuffer, "\r\n")-rbuffer, rbuffer);

    socket.close(); //ソケットクローズ
}

int main()
{
    printf("WiFi example\r\n\r\n");

    scan_demo(&wifi);

    printf("\r\nConnecting...\r\n");
    int ret = wifi.connect(MBED_CONF_APP_WIFI_SSID, MBED_CONF_APP_WIFI_PASSWORD, NSAPI_SECURITY_WPA_WPA2);
    if (ret != 0) {
        printf("\r\nConnection error\r\n");
        return -1;
    }

    printf("Success\r\n\r\n");
    printf("MAC: %s\r\n", wifi.get_mac_address());
    printf("IP: %s\r\n", wifi.get_ip_address());
    printf("Netmask: %s\r\n", wifi.get_netmask());
    printf("Gateway: %s\r\n", wifi.get_gateway());
    printf("RSSI: %d\r\n\r\n", wifi.get_rssi());

    printf("ThingSpeak Demo\r\n");

    // 60秒に１回メッセージを送信する。
    for (int i=0 ; i<1000 ; i++) {
        led1 = !led1;
        printf ("%d:",i);
        ThingSpeak_demo(&wifi); //ThingSpeed に
        Thread::wait(60000);    //60秒に一回送信
    }

    wifi.disconnect();

    printf ("Done\r\n");
}
