 /** Ethernet Interface for send/receive Datastructs over udp 
 *
 *
 * By Sebastian Donner
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include "Telemetry.h"

/**Debug Schnittstelle
*/
#ifdef DEBUG
Serial debug(USBTX, USBRX);
debug.baud(115200);
#endif

/**Konstruktoren
*/
EthernetInterface   eth;
TCPSocketConnection sock_tcp;
UDPSocket           sock_udp_client;  //send socket
UDPSocket           sock_udp_server;   //receive socket


    

/** Create a new Ethernet interface
  *
  */
Telemetry::Telemetry(PinName ora, PinName gre):orange(ora),green(gre)
{ 
    InitSucceed = false;
    green = 0;
    orange = 0;
}


/** Init a Serial Debug interface
  *
  */
#ifdef DEBUG
void Telemetry::InitUSBSerialConnection()
{
    serial.baud(115200);
}
#endif




/** Init Ethernet interface
  * mit DHCP max. 10 Sekunden 
*/
bool Telemetry::InitEthernetConnection()
{   
    bool ReturnValue = false;
    #ifdef DEBUG
    debug.printf("Initalisiere LAN Verbindung mit DHCP\r\n\r\n");
    #endif
    
    //Schnittstelle nur einmal initialisieren, sonst gibt es Fehler!
    if (!InitSucceed)
    {
        if (eth.init()==0)  //Init Interface
        {
            InitSucceed = true;
        }
    }
    
    //Nur wenn Initialisierung erfolgreich war!
    if (InitSucceed)
    {
        #ifdef DEBUG
        serial.printf("Verbinde\r\n\r\n");
        #endif
        ip_self = eth.getIPAddress();
        
        if (eth.connect(10000)==0) //CONNECT
        {
            #ifdef DEBUG
            printf("IP Adresse: %s\r\n\r\n", eth.getIPAddress());
            #endif
            ReturnValue = true;
            green = 1;
        }
        else
        {
            #ifdef DEBUG
            serial.printf("DHCP fail!\r\n\r\n");
            #endif
            ReturnValue = false;
        }
    }
    
    return ReturnValue;
}


/** Init Ethernet interface
  * ohne DHCP max. 2.5 Sekunden
  * @param IPAddress Interface IP 
*/
bool Telemetry::InitEthernetConnection(const char* IPAddress, const char* SubNetMask, const char* GateWay)
{   
    bool ReturnValue = false;
    #ifdef DEBUG
    printf("Initalisiere LAN Verbindung ohne DHCP\r\n\r\n");
    printf("IP: %s - GateWay: %s - SubNetMask: %s\r\n\r\n",IPAdress, GateWay, SubNetMask);
    #endif
    
    //Schnittstelle nur einmal initialisieren, sonst gibt es Fehler!
    if (!InitSucceed)
    {
        if (eth.init(IPAddress, SubNetMask, GateWay)==0)  //Init Interface
        {
            InitSucceed = true;
        }
    }
    
    //Nur wenn Initialisierung erfolgreich war!
    if (InitSucceed)
    {   
        if (eth.connect(2500)==0) //CONNECT
        {
            green = 1;
            #ifdef DEBUG
            serial.printf("Init success!");
            #endif
            
            ReturnValue = true;
        }
        else
        {
            #ifdef DEBUG
            serial.printf("Init fail!");
            #endif
            ReturnValue = false;
        }
    }    
    
    return ReturnValue;
}


//! Close Connection
void Telemetry::CloseEthernetConnection()
{
    eth.disconnect();
    InitSucceed = false;
    
    #ifdef DEBUG
    serial.printf("LAN Stack close\r\n\r\n");
    #endif
}


//! Connect Port TCP 
void Telemetry::ConnectSocket_TCP(Endpoint Host)
{
    sock_tcp.connect(Host.get_address(), Host.get_port());
    sock_tcp.set_blocking(false, 0);
    #ifdef DEBUG
    serial.printf("Open TCP Socket to IP: %s:%d.\r\n\r\n",Host.get_address(), Host.get_port());
    #endif
}


//! Connect Port UDP receive 
void Telemetry::ConnectSocket_UDP_Client()
{
    sock_udp_client.init();
    sock_udp_client.set_blocking(false, 0);  

    #ifdef DEBUG
    serial.printf("Open UDP_Client Socket\r\n\r\n");
    #endif
}


//! Connect Port UDP send
void Telemetry::ConnectSocket_UDP_Server(int Port)
{
    sock_udp_server.init();
    sock_udp_server.bind(Port);
    #ifdef DEBUG
    serial.printf("Open UDP_Server on Port: %d.\r\n\r\n",Port);
    #endif
}


//! Close Port TCP
void Telemetry::CloseSocket_TCP()
{
    sock_tcp.close();

    #ifdef DEBUG
    serial.printf("TCP Socket closed.\r\n\r\n");
    #endif
}


//! Close Port UDP receive
void Telemetry::CloseSocket_UDP_Client()
{
 sock_udp_client.close();   
  #ifdef DEBUG
    serial.printf("UDP Client Socket closed.\r\n\r\n");
  #endif
}


//! Close Port UDP send
void Telemetry::CloseSocket_UDP_Server()
{
    sock_udp_server.close();

    #ifdef DEBUG
    serial.printf("UDP Server Socket closed.\r\n\r\n");
    #endif
}


//! Struct Check Sum calc
uint8_t Telemetry::do_cs(uint8_t* buffer)
{
    uint8_t ck_a=0;
    uint8_t ck_b=0;

    for(int i=0; i < (buffer[0]-1); i++)
    {
        ck_a += buffer[i];
        ck_b += ck_a;
    }
    
    return ck_b;
}


//! Read UDP Packet on Client Port
int Telemetry::Rec_Data_UDP_Client(char *buffer, int size)
{
  return sock_udp_client.receiveFrom(input_Client, buffer, size);    
}

//! Read UDP Packet on Server Port
int Telemetry::Rec_Data_UDP_Server(char *buffer, int size)
{
  return sock_udp_server.receiveFrom(input_Server, buffer, size);    
}

    
//! Check UDP Packet of containing Struct
bool Telemetry::Rec_Struct(uint8_t *buffer)
{    
  orange = !orange;
  if (buffer[buffer[0]-1] == do_cs(buffer)) return true;
  else return false;
}


//! Read TCP Packet
int Telemetry::Rec_Data_TCP(char *buffer,int size)
{ 
        orange = !orange;
        return sock_tcp.receive(buffer, size);
}


//! Send UDP Packet as Client
void Telemetry::Send_Data_UDP_Client(Endpoint Server, char* Daten, int size )
{   
    sock_udp_client.sendTo(Server, Daten, size);   
    orange = !orange;
    #ifdef DEBUG
    serial.printf("UDP Paket gesendet.\r\n\r\n");
    #endif
}

//! Send UDP Packet as Server
void Telemetry::Send_Data_UDP_Server(Endpoint Client, char* Daten, int size )
{   
    sock_udp_server.sendTo(Client, Daten, size);   
    orange = !orange;
    #ifdef DEBUG
    serial.printf("UDP Paket gesendet.\r\n\r\n");
    #endif
}

//! Send Struct as UDP Client
void Telemetry::Send_Struct_UDP_Client(Endpoint Server, uint8_t* Daten)
{   
    Daten[(*Daten - 1)] = do_cs(Daten);
    sock_udp_client.sendTo(Server, (char*)Daten, *Daten);   
    orange = !orange;
}

//! Send Struct as UDP Server
void Telemetry::Send_Struct_UDP_Server(Endpoint Client, uint8_t* Daten)
{   
    Daten[(*Daten - 1)] = do_cs(Daten);
    sock_udp_server.sendTo(Client, (char*)Daten, *Daten);   
    orange = !orange;
}

//! Send TCP Packet
void Telemetry::Send_Data_TCP(char* Host, char* Buffer)
{
    
    /* Umwandeln in char*
    const char *DataBuf = datenpaket.c_str();
    char DataPaket[datenpaket.length()];
    strcpy(DataPaket,DataBuf);
    */
    #ifdef DEBUG
    serial.printf("----\r\n%s----\r\n\r\n",DataPaket);
    serial.printf("Sende Paket.\r\n\r\n");
    #endif
    
    sock_tcp.send_all(Buffer, sizeof(Buffer)-1);
    
    #ifdef DEBUG
    serial.printf("Paket gesendet.\r\n\r\n");
    #endif
}




    
