#include <mbed.h>
#include "rtos.h"
#include "Periphery/SupportSystem.h"
#include "Misc/SystemTimer.h"
#include "Controller/QuadratureController.h"
#include "Controller/MachineDirectionController.h"

#define PI 3.14159265

Serial serialXbee(p28,p27), serialMinnow(p13, p14);
PwmOut drivePWM(p22), steerPWM(p21);
I2C i2c(p9, p10);

DigitalOut serialLED(LED1), ledTwo(LED2), ledThree(LED3), heartbeatLED(LED4);

DigitalIn buttonOne(p25), buttonTwo(p26), buttonThree(p29), buttonFour(p30);

IMU_RegisterDataBuffer_t *IMU_registerDataBuffer;
RadioDecoder_RegisterDataBuffer_t *RadioDecoder_registerDataBuffer;

QuadratureController *quadratureController;
MachineDirectionController *machineDirectionController;

// Queues von der Bahnplanung
Queue<float, 2> quadrature_queue;
Queue<float, 2> machine_direction_queue;
float steering_angle_minnow_queue;
float velocity_minnow_queue;


// Queues von dem Maussensor
Queue<float, 2> imu_queue_velocity;
Queue<float, 2> imu_queue_steering_angle;
float steering_angle_imu_queue;
float velocity_imu_queue;

void hearbeat_thread(void const *args);

void serial_thread(void const *args) {
    size_t length = 20;
    char receive_buffer[length];
    char* ptr = (char*) receive_buffer;
    char* end = ptr + length;
    char c, x = 22;
    float f;
    while (true) {
        if (ptr != end) {
            c = serialMinnow.getc();
            *ptr++ = c;
        } else {
            ptr = (char*)receive_buffer;
        }
        serialLED = !serialLED;
        
        if (c == '\r') {
            sscanf(receive_buffer, "[%c]=%f\r", &x, &f);
            ptr = (char*)receive_buffer;
        }
        
        if (x == 'v') {
            machine_direction_queue.put(&f);
            
            ledTwo = true;
            // Buffer wieder auf Anfang setzen
            ptr = (char*)receive_buffer;
            x = 22;
        } else if (x == 'g') {
            f = -1*f;
            quadrature_queue.put(&f);
            ledThree = true;
            // Buffer wieder auf Anfang setzen
            ptr = (char*)receive_buffer;
            x = 22;
        }
        
        //Thread::wait(100);
    }
}

void redirect_machine_direction_controller(void const *args) {
    machineDirectionController->cylic_control();
}

void redirect_quadrature_controller(void const *args) {
    quadratureController->cylic_control();
}

int main() {
    serialMinnow.baud(115200);

    drivePWM.period_ms(20);
    steerPWM.period_ms(20);

    SystemTimer *millis = new SystemTimer();

    SupportSystem *supportSystem = new SupportSystem(0x80, &i2c);
    
    quadratureController = new QuadratureController(&steerPWM, &quadrature_queue, &imu_queue_steering_angle);
    machineDirectionController = new MachineDirectionController(&drivePWM, &machine_direction_queue, &imu_queue_velocity);

    Thread machineDirectionControl(serial_thread);
    Thread hearbeatThread(hearbeat_thread);
    
    RtosTimer machine_direction_control_timer(redirect_machine_direction_controller);
    RtosTimer quadrature_control_timer(redirect_quadrature_controller);

    // Konfiguration AMF-IMU
    // [0]: Conversation Factor
    // [1]: Sensor Position X
    // [2]: Sensor Position Y
    // [3]: Sensor Position Angle
    float configData[4] = {0.002751114f, 167.0f, 0.0f, 271.5f};
    
    supportSystem->writeImuConfig(configData);
    
    bool timer_started = false;
    
    steering_angle_minnow_queue = 0;
    quadrature_queue.put(&steering_angle_minnow_queue);
    
    velocity_minnow_queue = 0;
    machine_direction_queue.put(&velocity_minnow_queue);
    
    machine_direction_control_timer.start(10);
    quadrature_control_timer.start(10);

    while(true) {
        IMU_registerDataBuffer = supportSystem->getImuRegisterDataBuffer();
        RadioDecoder_registerDataBuffer = supportSystem->getRadioDecoderRegisterDataBuffer();

        uint16_t rc_percentage = RadioDecoder_registerDataBuffer->channelActiveTime[0];
        uint8_t rc_valid = RadioDecoder_registerDataBuffer->channelValid[0];
        
        uint16_t drive_percentage = RadioDecoder_registerDataBuffer->channelActiveTime[1];
        uint8_t drive_valid = RadioDecoder_registerDataBuffer->channelValid[1];

        uint16_t steer_percentage = RadioDecoder_registerDataBuffer->channelActiveTime[2];
        uint8_t steer_valid = RadioDecoder_registerDataBuffer->channelValid[2];


        if (rc_percentage > (uint16_t) 1800 && rc_valid != 0) {
            // oben => Wettbewerb
            supportSystem->setLightManagerRemoteLight(false, true);
            if (!timer_started) {
                timer_started = true;
                machine_direction_control_timer.start(10);
                quadrature_control_timer.start(10);
            }
        } else if (rc_percentage > (uint16_t) 1200 && rc_valid != 0) {
            // unten => RC-Wettbewerb
            supportSystem->setLightManagerRemoteLight(true, true);
            if (drive_valid) {
                drivePWM.pulsewidth_us(drive_percentage);
            }
            if (steer_valid) {
                steerPWM.pulsewidth_us(steer_percentage);
            }
            if (timer_started) {
                timer_started = false;
                machine_direction_control_timer.stop();
                quadrature_control_timer.stop();
            }
        } else if (rc_percentage > (uint16_t) 800 && rc_valid != 0) {
            // mitte => RC-Training
            supportSystem->setLightManagerRemoteLight(true, true);
            if (drive_valid) {
                drivePWM.pulsewidth_us(drive_percentage);
            }
            if (steer_valid) {
                steerPWM.pulsewidth_us(steer_percentage);
            }
            if (timer_started) {
                timer_started = false;
                machine_direction_control_timer.stop();
                quadrature_control_timer.stop();
            }
        }
        
        velocity_imu_queue = IMU_registerDataBuffer->velocityXFilteredRegister;
        imu_queue_velocity.put(&velocity_imu_queue);
        
        float radius = IMU_registerDataBuffer->velocityXFilteredRegister/IMU_registerDataBuffer->velocityAngularFilteredRegister;
        
        steering_angle_imu_queue = atan(0.205/radius)*180/PI;
        imu_queue_steering_angle.put(&steering_angle_imu_queue);
        

        //serialMinnow.printf("%ld, ", difference_millis);
        //serialMinnow.printf("%f, ", velocity_minnow_queue);
        //serialMinnow.printf("%f, ", steering_angle_minnow_queue);
        //serialMinnow.printf("%d, ", IMU_registerDataBuffer->curveRadiusRegister);
        //serialMinnow.printf("%f, ", IMU_registerDataBuffer->velocityAngularRegister);
        //serialMinnow.printf("%f, ", IMU_registerDataBuffer->velocityAngularFilteredRegister);
        //serialMinnow.printf("%f, ", IMU_registerDataBuffer->velocityXRegister);
        //serialMinnow.printf("%f\r\n", IMU_registerDataBuffer->velocityXFilteredRegister);
        Thread::wait(10);
    }
}

void hearbeat_thread(void const *args) {
    while(true) {
        heartbeatLED = !heartbeatLED;
        Thread::wait(100);
    }
}
