/**
@file TemperatureScreen.h

*/

#ifndef TEMPERATURESCREEN_H
#define TEMPERATURESCREEN_H

#include "ListController.h"
#include "Inputs.h"

/**
@brief Displays the Temperature options\n
@brief It is a list with two selectable options :\n
@brief view temp and temp plot\n
@brief view temp shows the current temperature along with the sentiment\n
@brief temp plot plots the temperature against time on the LCD screen at various intervals\n
@author Augustine Kizito K
@date April 2015
*/



class TemperatureScreen : public ListController // inherit the list Controller properties
{
private:
    bool changeScreen; // tracks if the user pressed a button that changes the screen
    int nextScreen; // tracks whether the user wants to go to
    //the previous screen or next screen

    void onScrollUp(); // user pressed the scroll up button
    void onScrollDown(); // user pressed the scroll down button
    void onEnter(); // user pressed the enter button
    void onBack(); // user pressed the back button

public:
    /**
    Creates an instance of the Temperature Screen
    
    */
    TemperatureScreen()
        :ListController( "Temperature", // Title
                         "View Temp", // Option 1
                         "Temp Plot", // Option 2
                         "", // Option 3
                         "", // Option 4
                         2) // Number of Options
    {};
    /**
    Manages the execution of the Temperature Screen
    @returns
    
    -1 - Navigate to previous screen\n
    0  - User selected option 1\n
    1  - User selected option 2
    */
    int start(); 


};












int TemperatureScreen::start()
{

    // Activate PullDown Resistors
    enterButton.mode(PullDown);
    upButton.mode(PullDown);
    downButton.mode(PullDown);
    backButton.mode(PullDown);

    upButton.rise(this,&TemperatureScreen::onScrollUp); // call onScrollUp when the up button is pressed
    downButton.rise(this,&TemperatureScreen::onScrollDown); // call onScrollDown when the down button is pressed
    enterButton.rise(this,&TemperatureScreen::onEnter); // call onEnter when the enter button is pressed
    backButton.rise(this,&TemperatureScreen::onBack); // call onBack when the back button is pressed

    // Launch the List Controller
    ListController::showInfo();
    changeScreen = false;

    debounce.start(); // start the debounce timer

    while(!changeScreen) {

        // mbed goes to sleep to save power
        Sleep();
    }

    // detach all interrupts
    upButton.rise(NULL);
    downButton.rise(NULL);
    enterButton.rise(NULL);
    backButton.rise(NULL);

    return nextScreen;
}

void TemperatureScreen::onScrollUp()
{
    if (debounceSuccess()) {  // check if debouncing succeeded

        ListController::scrollUp(); // scroll up
        debounce.reset(); // reset the debounce timer
    }
}

void TemperatureScreen::onScrollDown()
{
    if (debounceSuccess()) { // check if debouncing succeeded

        ListController::scrollDown(); // scroll down
        debounce.reset(); // reset the debounce timer
    }
}

void TemperatureScreen::onEnter()
{
    if (debounceSuccess()) { // check if debouncing succeeded
        playSound();
        nextScreen = ListController::getCurrentOption(); //  get the option that user selected
        changeScreen = true; // user wants to go to a different screen
        debounce.reset(); // reset the debounce timer
    }
}

void TemperatureScreen::onBack()
{
    if(debounceSuccess()) { // if debouncing succeeded
      playSound();
      nextScreen = -1; // go to previous screen
      changeScreen = true; // user wants to go to a different screen
      debounce.reset(); // reset the debounce timer
    }

}

#endif